"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackDeployment = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const url_1 = require("url");
const cxapi = require("../../../cx-api");
const asset_manifest_1 = require("../private/asset-manifest");
const cloud_assembly_internals_1 = require("../private/cloud-assembly-internals");
const asset_type_1 = require("./asset-type");
/**
 * Deployment of a single Stack
 *
 * You don't need to instantiate this class -- it will
 * be automatically instantiated as necessary when you
 * add a `Stage` to a pipeline.
 */
class StackDeployment {
    constructor(props) {
        var _b, _c;
        /**
         * Other stacks this stack depends on
         */
        this.stackDependencies = [];
        /**
         * Steps that take place before stack is prepared. If your pipeline engine disables 'prepareStep', then this will happen before stack deploys
         */
        this.pre = [];
        /**
         * Steps that take place after stack is prepared but before stack deploys. Your pipeline engine may not disable `prepareStep`.
         */
        this.changeSet = [];
        /**
         * Steps to execute after stack deploys
         */
        this.post = [];
        this.stackArtifactId = props.stackArtifactId;
        this.constructPath = props.constructPath;
        this.account = props.account;
        this.region = props.region;
        this.tags = (_b = props.tags) !== null && _b !== void 0 ? _b : {};
        this.assumeRoleArn = props.assumeRoleArn;
        this.executionRoleArn = props.executionRoleArn;
        this.stackName = props.stackName;
        this.absoluteTemplatePath = props.absoluteTemplatePath;
        this.templateUrl = props.templateS3Uri ? s3UrlFromUri(props.templateS3Uri, props.region) : undefined;
        this.assets = new Array();
        for (const asset of (_c = props.assets) !== null && _c !== void 0 ? _c : []) {
            if (asset.isTemplate) {
                this.templateAsset = asset;
            }
            else {
                this.assets.push(asset);
            }
        }
    }
    /**
     * Build a `StackDeployment` from a Stack Artifact in a Cloud Assembly.
     */
    static fromArtifact(stackArtifact) {
        jsiiDeprecationWarnings.monocdk_cx_api_CloudFormationStackArtifact(stackArtifact);
        const artRegion = stackArtifact.environment.region;
        const region = artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = stackArtifact.environment.account;
        const account = artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        return new StackDeployment({
            account,
            region,
            tags: stackArtifact.tags,
            stackArtifactId: stackArtifact.id,
            constructPath: stackArtifact.hierarchicalId,
            stackName: stackArtifact.stackName,
            absoluteTemplatePath: path.join(stackArtifact.assembly.directory, stackArtifact.templateFile),
            assumeRoleArn: stackArtifact.assumeRoleArn,
            executionRoleArn: stackArtifact.cloudFormationExecutionRoleArn,
            assets: extractStackAssets(stackArtifact),
            templateS3Uri: stackArtifact.stackTemplateAssetObjectUrl,
        });
    }
    /**
     * Add a dependency on another stack
     */
    addStackDependency(stackDeployment) {
        jsiiDeprecationWarnings.monocdk_pipelines_StackDeployment(stackDeployment);
        this.stackDependencies.push(stackDeployment);
    }
    /**
     * Adds steps to each phase of the stack
     * @param pre steps executed before stack.prepare
     * @param changeSet steps executed after stack.prepare and before stack.deploy
     * @param post steps executed after stack.deploy
     */
    addStackSteps(pre, changeSet, post) {
        this.pre.push(...pre);
        this.changeSet.push(...changeSet);
        this.post.push(...post);
    }
}
exports.StackDeployment = StackDeployment;
_a = JSII_RTTI_SYMBOL_1;
StackDeployment[_a] = { fqn: "monocdk.pipelines.StackDeployment", version: "1.149.0" };
function extractStackAssets(stackArtifact) {
    const ret = new Array();
    const assetManifests = stackArtifact.dependencies.filter(cloud_assembly_internals_1.isAssetManifest);
    for (const manifestArtifact of assetManifests) {
        const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
        for (const entry of manifest.entries) {
            let assetType;
            let isTemplate = false;
            if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                assetType = asset_type_1.AssetType.DOCKER_IMAGE;
            }
            else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                isTemplate = entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile;
                assetType = asset_type_1.AssetType.FILE;
            }
            else {
                throw new Error(`Unrecognized asset type: ${entry.type}`);
            }
            ret.push({
                assetManifestPath: manifestArtifact.file,
                assetId: entry.id.assetId,
                assetSelector: entry.id.toString(),
                assetType,
                assetPublishingRoleArn: entry.destination.assumeRoleArn,
                isTemplate,
            });
        }
    }
    return ret;
}
/**
 * Takes an s3://bucket/object-key uri and returns a region-aware https:// url for it
 *
 * @param uri The s3 URI
 * @param region The region (if undefined, we will return the global endpoint)
 * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/VirtualHosting.html#virtual-hosted-style-access
 */
function s3UrlFromUri(uri, region) {
    const url = url_1.parse(uri);
    return `https://${url.hostname}.s3.${region ? `${region}.` : ''}amazonaws.com${url.path}`;
}
//# sourceMappingURL=data:application/json;base64,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