"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file.
//
// Some of these flags only affect the behavior of the construct library --
// these will be removed in the next major release and the behavior they are
// gating will become the only behavior.
//
// Other flags also affect the generated CloudFormation templates, in a way
// that prevents seamless upgrading. In the next major version, their
// behavior will become the default, but the flag still exists so users can
// switch it *off* in order to revert to the old behavior. These flags
// are marked with with the [PERMANENT] tag below.
//
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.NEW_PROJECT_DEFAULT_CONTEXT = exports.FUTURE_FLAGS = exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = exports.TARGET_PARTITIONS = exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = exports.LAMBDA_RECOGNIZE_VERSION_PROPS = exports.EFS_DEFAULT_ENCRYPTION_AT_REST = exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 *
 * [PERMANENT]
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 *
 * [PERMANENT]
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 *
 * [PERMANENT]
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
 * for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
 * either the UsagePlan or ApiKey changing.
 *
 * The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
 * existing UsagePlanKey logical ids unchanged.
 * However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
 * the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
 * which again is disallowed.
 *
 * In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
 * This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.
 *
 * [PERMANENT]
 */
exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = '@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId';
/**
 * Enable this feature flag to have elastic file systems encrypted at rest by default.
 *
 * Encryption can also be configured explicitly using the `encrypted` property.
 */
exports.EFS_DEFAULT_ENCRYPTION_AT_REST = '@aws-cdk/aws-efs:defaultEncryptionAtRest';
/**
 * Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the
 * `fn.currentVersion`.
 *
 * The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
 * not constitute creating a new Version.
 *
 * See 'currentVersion' section in the aws-lambda module's README for more details.
 *
 * [PERMANENT]
 */
exports.LAMBDA_RECOGNIZE_VERSION_PROPS = '@aws-cdk/aws-lambda:recognizeVersionProps';
/**
 * Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.
 *
 * The security policy can also be configured explicitly using the `minimumProtocolVersion` property.
 *
 * [PERMANENT]
 */
exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = '@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021';
/**
 * What regions to include in lookup tables of environment agnostic stacks
 *
 * Has no effect on stacks that have a defined region, but will limit the amount
 * of unnecessary regions included in stacks without a known region.
 *
 * The type of this value should be a list of strings.
 *
 * [PERMANENT]
 */
exports.TARGET_PARTITIONS = '@aws-cdk/core:target-partitions';
/**
 * Enable this feature flag to configure default logging behavior for the ECS Service Extensions. This will enable the
 * `awslogs` log driver for the application container of the service to send the container logs to CloudWatch Logs.
 *
 * This is a feature flag as the new behavior provides a better default experience for the users.
 *
 * [PERMANENT]
 */
exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER = '@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver';
/**
 * Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names.
 *
 * Previously, the generated Launch Template names were only unique within a stack because they were based only on the
 * `Instance` construct ID. If another stack that has an `Instance` with the same construct ID is deployed in the same
 * account and region, the deployments would always fail as the generated Launch Template names were the same.
 *
 * The new implementation addresses this issue by generating the Launch Template name with the `Names.uniqueId` method.
 */
exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME = '@aws-cdk/aws-ec2:uniqueImdsv2TemplateName';
/**
 * Flag values that should apply for new projects
 *
 * Add a flag in here (typically with the value `true`), to enable
 * backwards-breaking behavior changes only for new projects.  New projects
 * generated through `cdk init` will include these flags in their generated
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: true,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: true,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: true,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: true,
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: true,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: true,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: true,
    [exports.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER]: true,
    [exports.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME]: true,
};
/**
 * Values that will be set by default in a new project, which are not necessarily booleans (and don't expire)
 */
exports.NEW_PROJECT_DEFAULT_CONTEXT = {
    [exports.TARGET_PARTITIONS]: ['aws', 'aws-cn'],
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The default values of each of these flags.
 *
 * This is the effective value of the flag, unless it's overriden via
 * context.
 *
 * Adding new flags here is only allowed during the pre-release period of a new
 * major version!
 */
const FUTURE_FLAGS_DEFAULTS = {};
function futureFlagDefault(flag) {
    var _a;
    return (_a = FUTURE_FLAGS_DEFAULTS[flag]) !== null && _a !== void 0 ? _a : false;
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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