"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const aws_iam_1 = require("../../../aws-iam");
const core_1 = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../../core");
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 */
class WaiterStateMachine extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.isCompleteHandler.functionArn],
        }));
        role.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [props.timeoutHandler.functionArn],
        }));
        const definition = core_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: props.interval.toSeconds(),
                            MaxAttempts: props.maxAttempts,
                            BackoffRate: props.backoffRate,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: props.isCompleteHandler.functionArn,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: props.timeoutHandler.functionArn,
                },
            },
        });
        const resource = new core_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
    }
    grantStartExecution(identity) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
//# sourceMappingURL=data:application/json;base64,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