"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfCloudFormationIntrinsic = exports.minimalCloudFormationJoin = exports.CLOUDFORMATION_TOKEN_RESOLVER = exports.CloudFormationLang = void 0;
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_utils_provider_1 = require("./cfn-utils-provider");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        return lazy_1.Lazy.uncachedString({
            // We used to do this by hooking into `JSON.stringify()` by adding in objects
            // with custom `toJSON()` functions, but it's ultimately simpler just to
            // reimplement the `stringify()` function from scratch.
            produce: (ctx) => tokenAwareStringify(obj, space !== null && space !== void 0 ? space : 0, ctx),
        });
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return fnJoinConcat(parts);
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Return a CFN intrinsic mass concatting any number of CloudFormation expressions
 */
function fnJoinConcat(parts) {
    return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
}
/**
 * Perform a JSON.stringify()-like operation, except aware of Tokens and CloudFormation intrincics
 *
 * Tokens will be resolved and if any resolve to CloudFormation intrinsics, the intrinsics
 * will be lifted to the top of a giant `{ Fn::Join }` expression.
 *
 * If Tokens resolve to primitive types (for example, by using Lazies), we'll
 * use the primitive type to determine how to encode the value into the JSON.
 *
 * If Tokens resolve to CloudFormation intrinsics, we'll use the type of the encoded
 * value as a type hint to determine how to encode the value into the JSON. The difference
 * is that we add quotes (") around strings, and don't add anything around non-strings.
 *
 * The following structure:
 *
 *    { SomeAttr: resource.someAttr }
 *
 * Will JSONify to either:
 *
 *    '{ "SomeAttr": "' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ '" }'
 * or '{ "SomeAttr": ' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ ' }'
 *
 * Depending on whether `someAttr` is type-hinted to be a string or not.
 *
 * (Where ++ is the CloudFormation string-concat operation (`{ Fn::Join }`).
 *
 * -----------------------
 *
 * This work requires 2 features from the `resolve()` function:
 *
 * - INTRINSICS TYPE HINTS: intrinsics are represented by values like
 *   `{ Ref: 'XYZ' }`. These values can reference either a string or a list/number at
 *   deploy time, and from the value alone there's no way to know which. We need
 *   to know the type to know whether to JSONify this reference to:
 *
 *      '{ "referencedValue": "' ++ { Ref: XYZ } ++ '"}'
 *   or '{ "referencedValue": ' ++ { Ref: XYZ } ++ '}'
 *
 *   I.e., whether or not we need to enclose the reference in quotes or not.
 *
 *   We COULD have done this by resolving one token at a time, and looking at the
 *   type of the encoded token we were resolving to obtain a type hint. However,
 *   the `resolve()` and Token system resist a level-at-a-time resolve
 *   operation: because of the existence of post-processors, we must have done a
 *   complete recursive resolution of a token before we can look at its result
 *   (after which any type information about the sources of nested resolved
 *   values is lost).
 *
 *   To fix this, "type hints" have been added to the `resolve()` function,
 *   giving an idea of the type of the source value for compplex result values.
 *   This only works for objects (not strings and numbers) but fortunately
 *   we only care about the types of intrinsics, which are always complex values.
 *
 *   Type hinting could have been added to the `IResolvable` protocol as well,
 *   but for now we just use the type of an encoded value as a type hint. That way
 *   we don't need to annotate anything more at the L1 level--we will use the type
 *   encodings added by construct authors at the L2 levels. L1 users can escape the
 *   default decision of "string" by using `Token.asList()`.
 *
 * - COMPLEX KEYS: since tokens can be string-encoded, we can use string-encoded tokens
 *   as the keys in JavaScript objects. However, after resolution, those string-encoded
 *   tokens could resolve to intrinsics (`{ Ref: ... }`), which CANNOT be stored in
 *   JavaScript objects anymore.
 *
 *   We therefore need a protocol to store the resolved values somewhere in the JavaScript
 *   type model,  which can be returned by `resolve()`, and interpreted by `tokenAwareStringify()`
 *   to produce the correct JSON.
 *
 *   And example will quickly show the point:
 *
 *    User writes:
 *       { [resource.resourceName]: 'SomeValue' }
 *    ------ string actually looks like ------>
 *       { '${Token[1234]}': 'SomeValue' }
 *    ------ resolve ------->
 *       { '$IntrinsicKey$0': [ {Ref: Resource}, 'SomeValue' ] }
 *    ------ tokenAwareStringify ------->
 *       '{ "' ++ { Ref: Resource } ++ '": "SomeValue" }'
 */
function tokenAwareStringify(root, space, ctx) {
    let indent = 0;
    const ret = new Array();
    // First completely resolve the tree, then encode to JSON while respecting the type
    // hints we got for the resolved intrinsics.
    recurse(ctx.resolve(root, { allowIntrinsicKeys: true }));
    switch (ret.length) {
        case 0: return undefined;
        case 1: return renderSegment(ret[0]);
        default:
            return fnJoinConcat(ret.map(renderSegment));
    }
    /**
     * Stringify a JSON element
     */
    function recurse(obj) {
        if (obj === undefined) {
            return;
        }
        if (token_1.Token.isUnresolved(obj)) {
            throw new Error('This shouldnt happen anymore');
        }
        if (Array.isArray(obj)) {
            return renderCollection('[', ']', obj, recurse);
        }
        if (typeof obj === 'object' && obj != null && !(obj instanceof Date)) {
            // Treat as an intrinsic if this LOOKS like a CFN intrinsic (`{ Ref: ... }`)
            // AND it's the result of a token resolution. Otherwise, we just treat this
            // value as a regular old JSON object (that happens to look a lot like an intrinsic).
            if (isIntrinsic(obj) && resolve_1.resolvedTypeHint(obj)) {
                renderIntrinsic(obj);
                return;
            }
            return renderCollection('{', '}', definedEntries(obj), ([key, value]) => {
                if (key.startsWith(resolve_1.INTRINSIC_KEY_PREFIX)) {
                    [key, value] = value;
                }
                recurse(key);
                pushLiteral(prettyPunctuation(':'));
                recurse(value);
            });
        }
        // Otherwise we have a scalar, defer to JSON.stringify()s serialization
        pushLiteral(JSON.stringify(obj));
    }
    /**
     * Render an object or list
     */
    function renderCollection(pre, post, xs, each) {
        pushLiteral(pre);
        indent += space;
        let atLeastOne = false;
        for (const [comma, item] of sepIter(xs)) {
            if (comma) {
                pushLiteral(',');
            }
            pushLineBreak();
            each(item);
            atLeastOne = true;
        }
        indent -= space;
        if (atLeastOne) {
            pushLineBreak();
        }
        pushLiteral(post);
    }
    function renderIntrinsic(intrinsic) {
        switch (resolve_1.resolvedTypeHint(intrinsic)) {
            case resolve_1.ResolutionTypeHint.STRING:
                pushLiteral('"');
                pushIntrinsic(deepQuoteStringLiterals(intrinsic));
                pushLiteral('"');
                return;
            case resolve_1.ResolutionTypeHint.LIST:
                // We need this to look like:
                //
                //    '{"listValue":' ++ STRINGIFY(CFN_EVAL({ Ref: MyList })) ++ '}'
                //
                // However, STRINGIFY would need to execute at CloudFormation deployment time, and that doesn't exist.
                //
                // We could *ALMOST* use:
                //
                //   '{"listValue":["' ++ JOIN('","', { Ref: MyList }) ++ '"]}'
                //
                // But that has the unfortunate side effect that if `CFN_EVAL({ Ref: MyList }) == []`, then it would
                // evaluate to `[""]`, which is a different value. Since CloudFormation does not have arbitrary
                // conditionals there's no way to deal with this case properly.
                //
                // Therefore, if we encounter lists we need to defer to a custom resource to handle
                // them properly at deploy time.
                const stack = stack_1.Stack.of(ctx.scope);
                // Because this will be called twice (once during `prepare`, once during `resolve`),
                // we need to make sure to be idempotent, so use a cache.
                const stringifyResponse = stringifyCache.obtain(stack, JSON.stringify(intrinsic), () => cfn_utils_provider_1.CfnUtils.stringify(stack, `CdkJsonStringify${stringifyCounter++}`, intrinsic));
                pushIntrinsic(stringifyResponse);
                return;
            case resolve_1.ResolutionTypeHint.NUMBER:
                pushIntrinsic(intrinsic);
                return;
        }
        throw new Error(`Unexpected type hint: ${resolve_1.resolvedTypeHint(intrinsic)}`);
    }
    /**
     * Push a literal onto the current segment if it's also a literal, otherwise open a new Segment
     */
    function pushLiteral(lit) {
        let last = ret[ret.length - 1];
        if ((last === null || last === void 0 ? void 0 : last.type) !== 'literal') {
            last = { type: 'literal', parts: [] };
            ret.push(last);
        }
        last.parts.push(lit);
    }
    /**
     * Add a new intrinsic segment
     */
    function pushIntrinsic(intrinsic) {
        ret.push({ type: 'intrinsic', intrinsic });
    }
    /**
     * Push a line break if we are pretty-printing, otherwise don't
     */
    function pushLineBreak() {
        if (space > 0) {
            pushLiteral(`\n${' '.repeat(indent)}`);
        }
    }
    /**
     * Add a space after the punctuation if we are pretty-printing, no space if not
     */
    function prettyPunctuation(punc) {
        return space > 0 ? `${punc} ` : punc;
    }
}
/**
 * Render a segment
 */
function renderSegment(s) {
    switch (s.type) {
        case 'literal': return s.parts.join('');
        case 'intrinsic': return s.intrinsic;
    }
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    },
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isPlainString(values[i - 1]) && isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === 'string' && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== 'Fn::Join') {
            return false;
        }
        const [delim, list] = obj['Fn::Join'];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
/**
 * Separated iterator
 */
function* sepIter(xs) {
    let comma = false;
    for (const item of xs) {
        yield [comma, item];
        comma = true;
    }
}
/**
 * Object.entries() but skipping undefined values
 */
function* definedEntries(xs) {
    for (const [key, value] of Object.entries(xs)) {
        if (value !== undefined) {
            yield [key, value];
        }
    }
}
/**
 * Quote string literals inside an intrinsic
 *
 * Formally, this should only match string literals that will be interpreted as
 * string literals. Fortunately, the strings that should NOT be quoted are
 * Logical IDs and attribute names, which cannot contain quotes anyway. Hence,
 * we can get away not caring about the distinction and just quoting everything.
 */
function deepQuoteStringLiterals(x) {
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringLiterals);
    }
    if (typeof x === 'object' && x != null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            ret[deepQuoteStringLiterals(key)] = deepQuoteStringLiterals(value);
        }
        return ret;
    }
    if (typeof x === 'string') {
        return quoteString(x);
    }
    return x;
}
/**
 * Quote the characters inside a string, for use inside toJSON
 */
function quoteString(s) {
    s = JSON.stringify(s);
    return s.substring(1, s.length - 1);
}
let stringifyCounter = 1;
/**
 * A cache scoped to object instances, that's maintained externally to the object instances
 */
class ScopedCache {
    constructor() {
        this.cache = new WeakMap();
    }
    obtain(object, key, init) {
        let kvMap = this.cache.get(object);
        if (!kvMap) {
            kvMap = new Map();
            this.cache.set(object, kvMap);
        }
        let ret = kvMap.get(key);
        if (ret === undefined) {
            ret = init();
            kvMap.set(key, ret);
        }
        return ret;
    }
}
const stringifyCache = new ScopedCache();
//# sourceMappingURL=data:application/json;base64,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