"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_CfnParameterProps(props);
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     */
    get type() {
        return this._type;
    }
    set type(type) {
        this._type = type;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified
     * when a stack is created. If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     */
    get default() {
        return this._default;
    }
    set default(value) {
        this._default = value;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     */
    get allowedValues() {
        return this._allowedValues;
    }
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     */
    get description() {
        return this._description;
    }
    set description(desc) {
        this._description = desc;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get maxLength() {
        return this._maxLength;
    }
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get minLength() {
        return this._minLength;
    }
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get maxValue() {
        return this._maxValue;
    }
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get minValue() {
        return this._minValue;
    }
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this._noEcho;
    }
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type) && !isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string or number type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    resolve(_context) {
        jsiiDeprecationWarnings.monocdk_IResolveContext(_context);
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
_a = JSII_RTTI_SYMBOL_1;
CfnParameter[_a] = { fqn: "monocdk.CfnParameter", version: "1.149.0" };
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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