"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecr_assets_1 = require("../../../aws-ecr-assets");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
_a = JSII_RTTI_SYMBOL_1;
S3Location[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.S3Location", version: "1.149.0" };
/**
 * Creates `IDockerImage` instances.
 *
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        jsiiDeprecationWarnings.monocdk_aws_ecr_assets_DockerImageAssetProps(props);
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
_b = JSII_RTTI_SYMBOL_1;
DockerImage[_b] = { fqn: "monocdk.aws_stepfunctions_tasks.DockerImage", version: "1.149.0" };
/**
 * S3 Data Type.
 *
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_ContainerDefinitionOptions(options);
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        var _f, _g, _h;
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_ISageMakerTask(task);
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: (_f = this.options.image) === null || _f === void 0 ? void 0 : _f.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: (_g = this.options.modelS3Location) === null || _g === void 0 ? void 0 : _g.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: (_h = this.options.environmentVariables) === null || _h === void 0 ? void 0 : _h.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_c = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_c] = { fqn: "monocdk.aws_stepfunctions_tasks.ContainerDefinition", version: "1.149.0" };
/**
 * Specifies how many models the container hosts
 *
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
_d = JSII_RTTI_SYMBOL_1;
AcceleratorClass[_d] = { fqn: "monocdk.aws_stepfunctions_tasks.AcceleratorClass", version: "1.149.0" };
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_AcceleratorClass(acceleratorClass);
        jsiiDeprecationWarnings.monocdk_aws_ec2_InstanceSize(instanceSize);
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
_e = JSII_RTTI_SYMBOL_1;
AcceleratorType[_e] = { fqn: "monocdk.aws_stepfunctions_tasks.AcceleratorType", version: "1.149.0" };
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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