"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualClusterInput = exports.ReleaseLabel = exports.Classification = exports.EmrContainersStartJobRun = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const logs = require("../../../aws-logs");
const s3 = require("../../../aws-s3");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const cr = require("../../../custom-resources");
const awscli = require("../../../lambda-layer-awscli");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts a job run.
 *
 * A job is a unit of work that you submit to Amazon EMR on EKS for execution.
 * The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
 * A job run is an execution of the job on the virtual cluster.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
 */
class EmrContainersStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.props = props;
        /**
         * Render the EMR Containers ConfigurationProperty as JSON
         */
        this.applicationConfigPropertyToJson = (property) => {
            return {
                Classification: cdk.stringToCloudFormation(property.classification.classificationStatement),
                Properties: property.properties ? cdk.objectToCloudFormation(property.properties) : undefined,
                Configurations: property.nestedConfig ? cdk.listMapper(this.applicationConfigPropertyToJson)(property.nestedConfig) : undefined,
            };
        };
        this.assignLogGroup = () => {
            var _e, _f, _g;
            if ((_e = this.props.monitoring) === null || _e === void 0 ? void 0 : _e.logGroup) {
                return ((_f = this.props.monitoring) === null || _f === void 0 ? void 0 : _f.logGroup);
            }
            else {
                return (((_g = this.props.monitoring) === null || _g === void 0 ? void 0 : _g.logging) ? new logs.LogGroup(this, 'Monitoring Log Group') : undefined);
            }
        };
        this.assignLogBucket = () => {
            var _e, _f, _g;
            if ((_e = this.props.monitoring) === null || _e === void 0 ? void 0 : _e.logBucket) {
                return ((_f = this.props.monitoring) === null || _f === void 0 ? void 0 : _f.logBucket);
            }
            else {
                return (((_g = this.props.monitoring) === null || _g === void 0 ? void 0 : _g.logging) ? new s3.Bucket(this, 'Monitoring Bucket') : undefined);
            }
        };
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_tasks_EmrContainersStartJobRunProps(props);
        this.integrationPattern = (_e = props.integrationPattern) !== null && _e !== void 0 ? _e : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.props.applicationConfig) {
            this.validateAppConfig(this.props.applicationConfig);
        }
        if (this.props.jobDriver.sparkSubmitJobDriver) {
            this.validateSparkSubmitJobDriver(props.jobDriver.sparkSubmitJobDriver);
        }
        if (this.props.executionRole === undefined
            && sfn.JsonPath.isEncodedJsonPath(props.virtualCluster.id)) {
            throw new Error('Execution role cannot be undefined when the virtual cluster ID is not a concrete value. Provide an execution role with the correct trust policy');
        }
        this.logGroup = this.assignLogGroup();
        this.logBucket = this.assignLogBucket();
        this.role = (_f = this.props.executionRole) !== null && _f !== void 0 ? _f : this.createJobExecutionRole();
        this.grantPrincipal = this.role;
        this.grantMonitoringPolicies();
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _e, _f, _g, _h, _j;
        return {
            Resource: task_utils_1.integrationResourceArn('emr-containers', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                VirtualClusterId: this.props.virtualCluster.id,
                Name: this.props.jobName,
                ExecutionRoleArn: this.role.roleArn,
                ReleaseLabel: this.props.releaseLabel.label,
                JobDriver: {
                    SparkSubmitJobDriver: {
                        EntryPoint: (_e = this.props.jobDriver.sparkSubmitJobDriver) === null || _e === void 0 ? void 0 : _e.entryPoint.value,
                        EntryPointArguments: (_g = (_f = this.props.jobDriver.sparkSubmitJobDriver) === null || _f === void 0 ? void 0 : _f.entryPointArguments) === null || _g === void 0 ? void 0 : _g.value,
                        SparkSubmitParameters: (_h = this.props.jobDriver.sparkSubmitJobDriver) === null || _h === void 0 ? void 0 : _h.sparkSubmitParameters,
                    },
                },
                ConfigurationOverrides: {
                    ApplicationConfiguration: cdk.listMapper(this.applicationConfigPropertyToJson)(this.props.applicationConfig),
                    MonitoringConfiguration: {
                        CloudWatchMonitoringConfiguration: this.logGroup ? {
                            LogGroupName: this.logGroup.logGroupName,
                            LogStreamNamePrefix: this.props.monitoring.logStreamNamePrefix,
                        } : undefined,
                        PersistentAppUI: (((_j = this.props.monitoring) === null || _j === void 0 ? void 0 : _j.persistentAppUI) === false)
                            ? 'DISABLED'
                            : 'ENABLED',
                        S3MonitoringConfiguration: this.logBucket ? {
                            LogUri: this.logBucket.s3UrlForObject(),
                        } : undefined,
                    },
                },
                Tags: this.props.tags,
            }),
        };
    }
    validateAppConfigPropertiesLength(appConfig) {
        if ((appConfig === null || appConfig === void 0 ? void 0 : appConfig.properties) === undefined) {
            return;
        }
        else if (Object.keys(appConfig.properties).length > 100) {
            throw new Error(`Application configuration properties must have 100 or fewer entries. Received ${Object.keys(appConfig.properties).length}`);
        }
    }
    validatePropertiesNestedAppConfigBothNotUndefined(appConfig) {
        if ((appConfig === null || appConfig === void 0 ? void 0 : appConfig.properties) === undefined && (appConfig === null || appConfig === void 0 ? void 0 : appConfig.nestedConfig) === undefined) {
            throw new Error('Application configuration must have either properties or nested app configurations defined.');
        }
    }
    validateAppConfig(config) {
        if (config === undefined) {
            return;
        }
        else if (config.length > 100) {
            throw new Error(`Application configuration array must have 100 or fewer entries. Received ${config.length}`);
        }
        else {
            config.forEach(element => this.validateAppConfig(element.nestedConfig));
            config.forEach(element => this.validateAppConfigPropertiesLength(element));
            config.forEach(element => this.validatePropertiesNestedAppConfigBothNotUndefined(element));
        }
    }
    isArrayOfStrings(value) {
        return Array.isArray(value) && value.every(item => typeof item === 'string');
    }
    validateEntryPointArguments(entryPointArguments) {
        if (typeof entryPointArguments.value === 'string' && !sfn.JsonPath.isEncodedJsonPath(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path, but received a non JSON path string');
      .`);
        }
        if (!this.isArrayOfStrings(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path but received ${typeof entryPointArguments.value}.`);
        }
    }
    validateEntryPointArgumentsLength(entryPointArguments) {
        if (this.isArrayOfStrings(entryPointArguments.value)
            && (entryPointArguments.value.length > 10280 || entryPointArguments.value.length < 1)) {
            throw new Error(`Entry point arguments must be a string array between 1 and 10280 in length. Received ${entryPointArguments.value.length}.`);
        }
    }
    validateSparkSubmitParametersLength(sparkSubmitParameters) {
        if (sparkSubmitParameters.length > 102400 || sparkSubmitParameters.length < 1) {
            throw new Error(`Spark submit parameters must be between 1 and 102400 characters in length. Received ${sparkSubmitParameters.length}.`);
        }
    }
    validateEntryPoint(entryPoint) {
        if (!sfn.JsonPath.isEncodedJsonPath(entryPoint.value) && (entryPoint.value.length > 256 || entryPoint.value.length < 1)) {
            throw new Error(`Entry point must be between 1 and 256 characters in length. Received ${entryPoint.value.length}.`);
        }
    }
    validateSparkSubmitJobDriver(driver) {
        this.validateEntryPoint(driver.entryPoint);
        if (driver.entryPointArguments) {
            this.validateEntryPointArguments(driver.entryPointArguments);
            this.validateEntryPointArgumentsLength(driver.entryPointArguments);
        }
        if (driver.sparkSubmitParameters) {
            this.validateSparkSubmitParametersLength(driver.sparkSubmitParameters);
        }
    }
    // https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/creating-job-execution-role.html
    createJobExecutionRole() {
        var _e, _f, _g;
        const jobExecutionRole = new iam.Role(this, 'Job-Execution-Role', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('emr-containers.amazonaws.com'), new iam.ServicePrincipal('states.amazonaws.com')),
        });
        (_e = this.logBucket) === null || _e === void 0 ? void 0 : _e.grantReadWrite(jobExecutionRole);
        (_f = this.logGroup) === null || _f === void 0 ? void 0 : _f.grantWrite(jobExecutionRole);
        (_g = this.logGroup) === null || _g === void 0 ? void 0 : _g.grant(jobExecutionRole, 'logs:DescribeLogStreams');
        jobExecutionRole.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
        this.updateRoleTrustPolicy(jobExecutionRole);
        return jobExecutionRole;
    }
    grantMonitoringPolicies() {
        var _e, _f, _g;
        (_e = this.logBucket) === null || _e === void 0 ? void 0 : _e.grantReadWrite(this.role);
        (_f = this.logGroup) === null || _f === void 0 ? void 0 : _f.grantWrite(this.role);
        (_g = this.logGroup) === null || _g === void 0 ? void 0 : _g.grant(this.role, 'logs:DescribeLogStreams');
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
    }
    /**
     * If an execution role is not provided by user, the automatically generated job execution role must create a trust relationship
     * between itself and the identity of the EMR managed service account in order to run jobs on the Kubernetes namespace.
     *
     * This cannot occur if the user provided virtualClusterId is within an encoded JSON path.
     *
     * The trust relationship can be created by updating the trust policy of the job execution role.
     *
     * @param role the automatically generated job execution role
     */
    updateRoleTrustPolicy(role) {
        const eksClusterInfo = new cr.AwsCustomResource(this, 'GetEksClusterInfo', {
            onCreate: {
                service: 'EMRcontainers',
                action: 'describeVirtualCluster',
                parameters: {
                    id: this.props.virtualCluster.id,
                },
                outputPaths: ['virtualCluster.containerProvider.info.eksInfo.namespace', 'virtualCluster.containerProvider.id'],
                physicalResourceId: cr.PhysicalResourceId.of('id'),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        /* We make use of custom resources to call update-roll-trust-policy as this command is only available through
         * aws cli because this is only used during the initial setup and is not available through the sdk.
         * https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/update-role-trust-policy.html
         * Commands available through SDK: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EMRcontainers.html
         * Commands available through CLI: https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/index.html
        */
        const cliLayer = new awscli.AwsCliLayer(this, 'awsclilayer');
        const shellCliLambda = new lambda.SingletonFunction(this, 'Call Update-Role-Trust-Policy', {
            uuid: '8693BB64-9689-44B6-9AAF-B0CC9EB8757C',
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, 'utils/role-policy')),
            timeout: cdk.Duration.seconds(30),
            memorySize: 256,
            layers: [cliLayer],
        });
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'eks',
                    resource: 'cluster',
                    resourceName: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                }),
            ],
            actions: [
                'eks:DescribeCluster',
            ],
        }));
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [role.roleArn],
            actions: [
                'iam:GetRole',
                'iam:UpdateAssumeRolePolicy',
            ],
        }));
        const provider = new cr.Provider(this, 'CustomResourceProvider', {
            onEventHandler: shellCliLambda,
        });
        new cdk.CustomResource(this, 'Custom Resource', {
            properties: {
                eksNamespace: eksClusterInfo.getResponseField('virtualCluster.containerProvider.info.eksInfo.namespace'),
                eksClusterId: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                roleName: role.roleName,
            },
            serviceToken: provider.serviceToken,
        });
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : this.props.virtualCluster.id,
                    }),
                ],
                actions: ['emr-containers:StartJobRun'],
                conditions: {
                    StringEquals: {
                        'emr-containers:ExecutionRoleArn': this.role.roleArn,
                    },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : `${this.props.virtualCluster.id}/jobruns/*`,
                    }),
                ],
                actions: [
                    'emr-containers:DescribeJobRun',
                    'emr-containers:CancelJobRun',
                ],
            }));
        }
        return policyStatements;
    }
}
exports.EmrContainersStartJobRun = EmrContainersStartJobRun;
_a = JSII_RTTI_SYMBOL_1;
EmrContainersStartJobRun[_a] = { fqn: "monocdk.aws_stepfunctions_tasks.EmrContainersStartJobRun", version: "1.149.0" };
EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * The classification within a EMR Containers application configuration.
 * Class can be extended to add other classifications.
 * For example, new Classification('xxx-yyy');
 */
class Classification {
    /**
     * Creates a new Classification
     *
     * @param classificationStatement A literal string in case a new EMR classification is released, if not already defined.
     */
    constructor(classificationStatement) {
        this.classificationStatement = classificationStatement;
    }
}
exports.Classification = Classification;
_b = JSII_RTTI_SYMBOL_1;
Classification[_b] = { fqn: "monocdk.aws_stepfunctions_tasks.Classification", version: "1.149.0" };
/**
 * Sets the maximizeResourceAllocation property to true or false.
 * When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.
 *
 * For more info:
 * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
 */
Classification.SPARK = new Classification('spark');
/**
 * Sets values in the spark-defaults.conf file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html
 */
Classification.SPARK_DEFAULTS = new Classification('spark-defaults');
/**
 * Sets values in the spark-env.sh file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html#environment-variables
 */
Classification.SPARK_ENV = new Classification('spark-env');
/**
 * Sets values in the hive-site.xml for Spark.
 */
Classification.SPARK_HIVE_SITE = new Classification('spark-hive-site');
/**
 * Sets values in the log4j.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/log4j.properties.template
 */
Classification.SPARK_LOG4J = new Classification('spark-log4j');
/**
 * Sets values in the metrics.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/metrics.properties.template
 */
Classification.SPARK_METRICS = new Classification('spark-metrics');
/**
 * The Amazon EMR release version to use for the job run.
 *
 * Can be extended to include new EMR releases
 *
 * For example, `new ReleaseLabel('emr-x.xx.x-latest');`
 */
class ReleaseLabel {
    /**
     * Initializes the label string.
     *
     * @param label A literal string that contains the release-version ex. 'emr-x.x.x-latest'
     */
    constructor(label) {
        this.label = label;
    }
}
exports.ReleaseLabel = ReleaseLabel;
_c = JSII_RTTI_SYMBOL_1;
ReleaseLabel[_c] = { fqn: "monocdk.aws_stepfunctions_tasks.ReleaseLabel", version: "1.149.0" };
/**
 * EMR Release version 5.32.0
 */
ReleaseLabel.EMR_5_32_0 = new ReleaseLabel('emr-5.32.0-latest');
/**
 * EMR Release version 5.33.0
 */
ReleaseLabel.EMR_5_33_0 = new ReleaseLabel('emr-5.33.0-latest');
/**
 * EMR Release version 6.2.0
 */
ReleaseLabel.EMR_6_2_0 = new ReleaseLabel('emr-6.2.0-latest');
/**
 * EMR Release version 6.3.0
 */
ReleaseLabel.EMR_6_3_0 = new ReleaseLabel('emr-6.3.0-latest');
/**
 * Class that returns a virtual cluster's id depending on input type
 */
class VirtualClusterInput {
    /**
     * Initializes the virtual cluster ID.
     *
     * @param id The VirtualCluster Id
     */
    constructor(id) {
        this.id = id;
    }
    /**
     * Input for a virtualClusterId from a Task Input
     */
    static fromTaskInput(taskInput) {
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_TaskInput(taskInput);
        return new VirtualClusterInput(taskInput.value);
    }
    /**
     * Input for virtualClusterId from a literal string
     */
    static fromVirtualClusterId(virtualClusterId) {
        return new VirtualClusterInput(virtualClusterId);
    }
}
exports.VirtualClusterInput = VirtualClusterInput;
_d = JSII_RTTI_SYMBOL_1;
VirtualClusterInput[_d] = { fqn: "monocdk.aws_stepfunctions_tasks.VirtualClusterInput", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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