"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationProduct = exports.Product = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const association_manager_1 = require("./private/association-manager");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class ProductBase extends core_1.Resource {
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.productId, tagOptions);
    }
}
/**
 * Abstract class for Service Catalog Product.
 */
class Product extends ProductBase {
    /**
     * Creates a Product construct that represents an external product.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param productArn Product Arn
     */
    static fromProductArn(scope, id, productArn) {
        const arn = core_1.Stack.of(scope).splitArn(productArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const productId = arn.resourceName;
        if (!productId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + productArn);
        }
        return new class extends ProductBase {
            constructor() {
                super(...arguments);
                this.productId = productId;
                this.productArn = productArn;
            }
        }(scope, id);
    }
}
exports.Product = Product;
_a = JSII_RTTI_SYMBOL_1;
Product[_a] = { fqn: "monocdk.aws_servicecatalog.Product", version: "1.149.0" };
/**
 * A Service Catalog Cloudformation Product.
 */
class CloudFormationProduct extends Product {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_servicecatalog_CloudFormationProductProps(props);
        this.validateProductProps(props);
        const product = new servicecatalog_generated_1.CfnCloudFormationProduct(this, 'Resource', {
            acceptLanguage: props.messageLanguage,
            description: props.description,
            distributor: props.distributor,
            name: props.productName,
            owner: props.owner,
            provisioningArtifactParameters: this.renderProvisioningArtifacts(props),
            replaceProvisioningArtifacts: props.replaceProductVersionIds,
            supportDescription: props.supportDescription,
            supportEmail: props.supportEmail,
            supportUrl: props.supportUrl,
        });
        this.productId = product.ref;
        this.productArn = core_1.Stack.of(this).formatArn({
            service: 'catalog',
            resource: 'product',
            resourceName: product.ref,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    renderProvisioningArtifacts(props) {
        return props.productVersions.map(productVersion => {
            const template = productVersion.cloudFormationTemplate.bind(this);
            validation_1.InputValidator.validateUrl(this.node.path, 'provisioning template url', template.httpUrl);
            return {
                name: productVersion.productVersionName,
                description: productVersion.description,
                disableTemplateValidation: productVersion.validateTemplate === false ? true : false,
                info: {
                    LoadTemplateFromURL: template.httpUrl,
                },
            };
        });
    }
    ;
    validateProductProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'product product name', 1, 100, props.productName);
        validation_1.InputValidator.validateLength(this.node.path, 'product owner', 1, 8191, props.owner);
        validation_1.InputValidator.validateLength(this.node.path, 'product description', 0, 8191, props.description);
        validation_1.InputValidator.validateLength(this.node.path, 'product distributor', 0, 8191, props.distributor);
        validation_1.InputValidator.validateEmail(this.node.path, 'support email', props.supportEmail);
        validation_1.InputValidator.validateUrl(this.node.path, 'support url', props.supportUrl);
        validation_1.InputValidator.validateLength(this.node.path, 'support description', 0, 8191, props.supportDescription);
        if (props.productVersions.length == 0) {
            throw new Error(`Invalid product versions for resource ${this.node.path}, must contain at least 1 product version`);
        }
        props.productVersions.forEach(productVersion => {
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact name', 0, 100, productVersion.productVersionName);
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact description', 0, 8191, productVersion.description);
        });
    }
}
exports.CloudFormationProduct = CloudFormationProduct;
_b = JSII_RTTI_SYMBOL_1;
CloudFormationProduct[_b] = { fqn: "monocdk.aws_servicecatalog.CloudFormationProduct", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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