"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const cdk = require("../../core");
const product_stack_synthesizer_1 = require("./private/product-stack-synthesizer");
/**
 * A Service Catalog product stack, which is similar in form to a Cloudformation nested stack.
 * You can add the resources to this stack that you want to define for your service catalog product.
 *
 * This stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 */
class ProductStack extends cdk.Stack {
    constructor(scope, id) {
        super(scope, id, {
            synthesizer: new product_stack_synthesizer_1.ProductStackSynthesizer(),
        });
        this._parentStack = findParentStack(scope);
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${cdk.Names.uniqueId(this)}.product.template.json`;
    }
    /**
     * Fetch the template URL.
     *
     * @internal
     */
    _getTemplateUrl() {
        return cdk.Lazy.uncachedString({ produce: () => this._templateUrl });
    }
    /**
     * Synthesize the product stack template, overrides the `super` class method.
     *
     * Defines an asset at the parent stack which represents the template of this
     * product stack.
     *
     * @internal
     */
    _synthesizeTemplate(session) {
        const cfn = JSON.stringify(this._toCloudFormation(), undefined, 2);
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        this._templateUrl = this._parentStack.synthesizer.addFileAsset({
            packaging: cdk.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        }).httpUrl;
        fs.writeFileSync(path.join(session.assembly.outdir, this.templateFile), cfn);
    }
}
exports.ProductStack = ProductStack;
_a = JSII_RTTI_SYMBOL_1;
ProductStack[_a] = { fqn: "monocdk.aws_servicecatalog.ProductStack", version: "1.149.0" };
/**
 * Validates the scope for a product stack, which must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    try {
        const parentStack = cdk.Stack.of(scope);
        return parentStack;
    }
    catch (e) {
        throw new Error('Product stacks must be defined within scope of another non-product stack');
    }
}
//# sourceMappingURL=data:application/json;base64,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