"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = exports.SecretStringValueBeta1 = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * An experimental class used to specify an initial secret value for a Secret.
 * The class wraps a simple string (or JSON representation) in order to provide some safety checks and warnings
 * about the dangers of using plaintext strings as initial secret seed values via CDK/CloudFormation.
 */
class SecretStringValueBeta1 {
    constructor(_secretValue) {
        this._secretValue = _secretValue;
    }
    /**
     * Creates a `SecretStringValueBeta1` from a plaintext value.
     * This approach is inherently unsafe, as the secret value may be visible in your source control repository
     * and will also appear in plaintext in the resulting CloudFormation template, including in the AWS Console or APIs.
     * Usage of this method is discouraged, especially for production workloads.
     */
    static fromUnsafePlaintext(secretValue) { return new SecretStringValueBeta1(secretValue); }
    /**
     * Creates a `SecretValueValueBeta1` from a string value coming from a Token.
     * The intent is to enable creating secrets from references (e.g., `Ref`, `Fn::GetAtt`) from other resources.
     * This might be the direct output of another Construct, or the output of a Custom Resource.
     * This method throws if it determines the input is an unsafe plaintext string.
     *
     * For example:
     * ```ts
     *     // Creates a new IAM user, access and secret keys, and stores the secret access key in a Secret.
     *     const user = new iam.User(this, 'User');
     *     const accessKey = new iam.AccessKey(this, 'AccessKey', { user });
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(accessKey.secretAccessKey.toString());
     *     new secretsmanager.Secret(this, 'Secret', {
     *       secretStringBeta1: secretValue,
     *     });
     * ```
     *
     * The secret may also be embedded in a string representation of a JSON structure:
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(JSON.stringify({
     *       username: user.userName,
     *       database: 'foo',
     *       password: accessKey.secretAccessKey.toString(),
     *     }));
     *
     * Note that the value being a Token does *not* guarantee safety. For example, a Lazy-evaluated string
     * (e.g., `Lazy.string({ produce: () => 'myInsecurePassword' }))`) is a Token, but as the output is
     * ultimately a plaintext string, and so insecure.
     *
     * @param secretValueFromToken a secret value coming from a Construct attribute or Custom Resource output
     */
    static fromToken(secretValueFromToken) {
        if (!core_1.Token.isUnresolved(secretValueFromToken)) {
            throw new Error('SecretStringValueBeta1 appears to be plaintext (unsafe) string (or resolved Token); use fromUnsafePlaintext if this is intentional');
        }
        return new SecretStringValueBeta1(secretValueFromToken);
    }
    /** Returns the secret value */
    secretValue() { return this._secretValue; }
}
exports.SecretStringValueBeta1 = SecretStringValueBeta1;
_a = JSII_RTTI_SYMBOL_1;
SecretStringValueBeta1[_a] = { fqn: "monocdk.aws_secretsmanager.SecretStringValueBeta1", version: "1.149.0" };
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    get secretFullArn() { return this.secretArn; }
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        const statement = result.principalStatement || result.resourceStatement;
        if (versionStages != null && statement) {
            statement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        const crossAccount = core_1.Token.compareStrings(core_1.Stack.of(this).account, grantee.grantPrincipal.principalAccount || '');
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && (!this.encryptionKey && crossAccount === core_1.TokenComparison.DIFFERENT)) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && !this.encryptionKey) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policy) === null || _d === void 0 ? void 0 : _d.document.validateForResourcePolicy()) || []);
        return errors;
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     */
    get arnForPolicies() {
        return this.secretFullArn ? this.secretFullArn : `${this.secretArn}-??????`;
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.replicaRegions = [];
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_SecretProps(props);
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        if (props.generateSecretString && props.secretStringBeta1) {
            throw new Error('Cannot specify both `generateSecretString` and `secretStringBeta1`.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: (_d = props.generateSecretString) !== null && _d !== void 0 ? _d : (props.secretStringBeta1 ? undefined : {}),
            secretString: (_e = props.secretStringBeta1) === null || _e === void 0 ? void 0 : _e.secretValue(),
            name: this.physicalName,
            replicaRegions: core_1.Lazy.any({ produce: () => this.replicaRegions }, { omitEmptyArray: true }),
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core_1.RemovalPolicy.DESTROY,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.encryptionKey = props.encryptionKey;
        const parseOwnedSecretName = core_1.FeatureFlags.of(this).isEnabled(cxapi.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME);
        this.secretName = parseOwnedSecretName
            ? parseSecretNameForOwnedSecret(this, this.secretArn, props.secretName)
            : parseSecretName(this, this.secretArn);
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_f = this.encryptionKey) === null || _f === void 0 ? void 0 : _f.grantEncryptDecrypt(principal);
        (_g = this.encryptionKey) === null || _g === void 0 ? void 0 : _g.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
        for (const replica of (_h = props.replicaRegions) !== null && _h !== void 0 ? _h : []) {
            this.addReplicaRegion(replica.region, replica.encryptionKey);
        }
    }
    /** @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn` */
    static fromSecretArn(scope, id, secretArn) {
        jsiiDeprecationWarnings.print("monocdk.aws_secretsmanager.Secret#fromSecretArn", "use `fromSecretCompleteArn` or `fromSecretPartialArn`");
        const attrs = arnIsComplete(secretArn) ? { secretCompleteArn: secretArn } : { secretPartialArn: secretArn };
        return Secret.fromSecretAttributes(scope, id, attrs);
    }
    /** Imports a secret by complete ARN. The complete ARN is the ARN with the Secrets Manager-supplied suffix. */
    static fromSecretCompleteArn(scope, id, secretCompleteArn) {
        return Secret.fromSecretAttributes(scope, id, { secretCompleteArn });
    }
    /** Imports a secret by partial ARN. The partial ARN is the ARN without the Secrets Manager-supplied suffix. */
    static fromSecretPartialArn(scope, id, secretPartialArn) {
        return Secret.fromSecretAttributes(scope, id, { secretPartialArn });
    }
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope, id, secretName) {
        jsiiDeprecationWarnings.print("monocdk.aws_secretsmanager.Secret#fromSecretName", "use `fromSecretNameV2`");
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Imports a secret by secret name.
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     */
    static fromSecretNameV2(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretName = secretName;
                this.secretArn = this.partialArn;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Creates a "partial" ARN from the secret name. The "full" ARN would include the SecretsManager-provided suffix.
            get partialArn() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: secretName,
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_SecretAttributes(attrs);
        let secretArn;
        let secretArnIsPartial;
        if (attrs.secretArn) {
            if (attrs.secretCompleteArn || attrs.secretPartialArn) {
                throw new Error('cannot use `secretArn` with `secretCompleteArn` or `secretPartialArn`');
            }
            secretArn = attrs.secretArn;
            secretArnIsPartial = false;
        }
        else {
            if ((attrs.secretCompleteArn && attrs.secretPartialArn) ||
                (!attrs.secretCompleteArn && !attrs.secretPartialArn)) {
                throw new Error('must use only one of `secretCompleteArn` or `secretPartialArn`');
            }
            if (attrs.secretCompleteArn && !arnIsComplete(attrs.secretCompleteArn)) {
                throw new Error('`secretCompleteArn` does not appear to be complete; missing 6-character suffix');
            }
            [secretArn, secretArnIsPartial] = attrs.secretCompleteArn ? [attrs.secretCompleteArn, false] : [attrs.secretPartialArn, true];
        }
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = secretArn;
                this.secretName = parseSecretName(scope, secretArn);
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return secretArnIsPartial ? undefined : secretArn; }
        }(scope, id, { environmentFromArn: secretArn });
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        jsiiDeprecationWarnings.print("monocdk.aws_secretsmanager.Secret#addTargetAttachment", "use `attach()` instead");
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_AttachedSecretOptions(options);
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a replica region for the secret
     *
     * @param region The name of the region
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     */
    addReplicaRegion(region, encryptionKey) {
        jsiiDeprecationWarnings.monocdk_aws_kms_IKey(encryptionKey);
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && !core_1.Token.isUnresolved(region) && region === stack.region) {
            throw new Error('Cannot add the region where this stack is deployed as a replica region.');
        }
        this.replicaRegions.push({
            region,
            kmsKeyId: encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.keyArn,
        });
    }
}
exports.Secret = Secret;
_b = JSII_RTTI_SYMBOL_1;
Secret[_b] = { fqn: "monocdk.aws_secretsmanager.Secret", version: "1.149.0" };
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_SecretTargetAttachmentProps(props);
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
_c = JSII_RTTI_SYMBOL_1;
SecretTargetAttachment[_c] = { fqn: "monocdk.aws_secretsmanager.SecretTargetAttachment", version: "1.149.0" };
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${6-character SecretsManager suffix}`
        // If there is no hyphen (or 6-character suffix) assume no suffix was provided, and return the whole name.
        const lastHyphenIndex = resourceName.lastIndexOf('-');
        const hasSecretsSuffix = lastHyphenIndex !== -1 && resourceName.substr(lastHyphenIndex + 1).length === 6;
        return hasSecretsSuffix ? resourceName.substr(0, lastHyphenIndex) : resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
/**
 * Parses the secret name from the ARN of an owned secret. With owned secrets we know a few things we don't with imported secrets:
 * - The ARN is guaranteed to be a full ARN, with suffix.
 * - The name -- if provided -- will tell us how many hyphens to expect in the final secret name.
 * - If the name is not provided, we know the format used by CloudFormation for auto-generated names.
 *
 * Note: This is done rather than just returning the secret name passed in by the user to keep the relationship
 * explicit between the Secret and wherever the secretName might be used (i.e., using Tokens).
 */
function parseSecretNameForOwnedSecret(construct, secretArn, secretName) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (!resourceName) {
        throw new Error('invalid ARN format; no secret name provided');
    }
    // Secret name was explicitly provided, but is unresolved; best option is to use it directly.
    // If it came from another Secret, it should (hopefully) already be properly formatted.
    if (secretName && core_1.Token.isUnresolved(secretName)) {
        return secretName;
    }
    // If no secretName was provided, the name will be automatically generated by CloudFormation.
    // The autogenerated names have the form of `${logicalID}-${random}`.
    // Otherwise, we can use the existing secretName to determine how to parse the resulting resourceName.
    const secretNameHyphenatedSegments = secretName ? secretName.split('-').length : 2;
    // 2 => [0, 1]
    const segmentIndexes = [...new Array(secretNameHyphenatedSegments)].map((_, i) => i);
    // Create the secret name from the resource name by joining all the known segments together.
    // This should have the effect of stripping the final hyphen and SecretManager suffix.
    return core_1.Fn.join('-', segmentIndexes.map(i => core_1.Fn.select(i, core_1.Fn.split('-', resourceName))));
}
/** Performs a best guess if an ARN is complete, based on if it ends with a 6-character suffix. */
function arnIsComplete(secretArn) {
    return core_1.Token.isUnresolved(secretArn) || /-[a-z0-9]{6}$/i.test(secretArn);
}
//# sourceMappingURL=data:application/json;base64,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