import * as iam from '../../aws-iam';
import * as kms from '../../aws-kms';
import { IResource, RemovalPolicy, Resource, SecretValue } from '../../core';
import { Construct } from 'constructs';
import { RotationSchedule, RotationScheduleOptions } from './rotation-schedule';
/**
 * A secret in AWS Secrets Manager.
 */
export interface ISecret extends IResource {
    /**
     * The customer-managed encryption key that is used to encrypt this secret, if any. When not specified, the default
     * KMS key for the account and region is being used.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The ARN of the secret in AWS Secrets Manager. Will return the full ARN if available, otherwise a partial arn.
     * For secrets imported by the deprecated `fromSecretName`, it will return the `secretName`.
     * @attribute
     */
    readonly secretArn: string;
    /**
     * The full ARN of the secret in AWS Secrets Manager, which is the ARN including the Secrets Manager-supplied 6-character suffix.
     * This is equal to `secretArn` in most cases, but is undefined when a full ARN is not available (e.g., secrets imported by name).
     */
    readonly secretFullArn?: string;
    /**
     * The name of the secret.
     *
     * For "owned" secrets, this will be the full resource name (secret name + suffix), unless the
     * '@aws-cdk/aws-secretsmanager:parseOwnedSecretName' feature flag is set.
     */
    readonly secretName: string;
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     * @attribute
     */
    readonly secretValue: SecretValue;
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     */
    secretValueFromJson(key: string): SecretValue;
    /**
     * Grants reading the secret value to some role.
     *
     * @param grantee       the principal being granted permission.
     * @param versionStages the version stages the grant is limited to. If not specified, no restriction on the version
     *                      stages is applied.
     */
    grantRead(grantee: iam.IGrantable, versionStages?: string[]): iam.Grant;
    /**
     * Grants writing and updating the secret value to some role.
     *
     * @param grantee       the principal being granted permission.
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Adds a rotation schedule to the secret.
     */
    addRotationSchedule(id: string, options: RotationScheduleOptions): RotationSchedule;
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Denies the `DeleteSecret` action to all principals within the current
     * account.
     */
    denyAccountRootDelete(): void;
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     */
    attach(target: ISecretAttachmentTarget): ISecret;
}
/**
 * The properties required to create a new secret in AWS Secrets Manager.
 */
export interface SecretProps {
    /**
     * An optional, human-friendly description of the secret.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * The customer-managed encryption key to use for encrypting the secret value.
     *
     * @default - A default KMS key for the account and region is used.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Configuration for how to generate a secret value.
     *
     * Only one of `secretString` and `generateSecretString` can be provided.
     *
     * @default - 32 characters with upper-case letters, lower-case letters, punctuation and numbers (at least one from each
     * category), per the default values of ``SecretStringGenerator``.
     */
    readonly generateSecretString?: SecretStringGenerator;
    /**
     * A name for the secret. Note that deleting secrets from SecretsManager does not happen immediately, but after a 7 to
     * 30 days blackout period. During that period, it is not possible to create another secret that shares the same name.
     *
     * @default - A name is generated by CloudFormation.
     */
    readonly secretName?: string;
    /**
     * Initial value for the secret
     *
     * **NOTE:** *It is **highly** encouraged to leave this field undefined and allow SecretsManager to create the secret value.
     * The secret string -- if provided -- will be included in the output of the cdk as part of synthesis,
     * and will appear in the CloudFormation template in the console. This can be secure(-ish) if that value is merely reference to
     * another resource (or one of its attributes), but if the value is a plaintext string, it will be visible to anyone with access
     * to the CloudFormation template (via the AWS Console, SDKs, or CLI).
     *
     * Specifies text data that you want to encrypt and store in this new version of the secret.
     * May be a simple string value, or a string representation of a JSON structure.
     *
     * Only one of `secretString` and `generateSecretString` can be provided.
     *
     * @default - SecretsManager generates a new secret value.
     */
    readonly secretStringBeta1?: SecretStringValueBeta1;
    /**
     * Policy to apply when the secret is removed from this stack.
     *
     * @default - Not set.
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * A list of regions where to replicate this secret.
     *
     * @default - Secret is not replicated
     */
    readonly replicaRegions?: ReplicaRegion[];
}
/**
 * Secret replica region
 */
export interface ReplicaRegion {
    /**
     * The name of the region
     */
    readonly region: string;
    /**
     * The customer-managed encryption key to use for encrypting the secret value.
     *
     * @default - A default KMS key for the account and region is used.
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * An experimental class used to specify an initial secret value for a Secret.
 * The class wraps a simple string (or JSON representation) in order to provide some safety checks and warnings
 * about the dangers of using plaintext strings as initial secret seed values via CDK/CloudFormation.
 */
export declare class SecretStringValueBeta1 {
    private readonly _secretValue;
    /**
     * Creates a `SecretStringValueBeta1` from a plaintext value.
     * This approach is inherently unsafe, as the secret value may be visible in your source control repository
     * and will also appear in plaintext in the resulting CloudFormation template, including in the AWS Console or APIs.
     * Usage of this method is discouraged, especially for production workloads.
     */
    static fromUnsafePlaintext(secretValue: string): SecretStringValueBeta1;
    /**
     * Creates a `SecretValueValueBeta1` from a string value coming from a Token.
     * The intent is to enable creating secrets from references (e.g., `Ref`, `Fn::GetAtt`) from other resources.
     * This might be the direct output of another Construct, or the output of a Custom Resource.
     * This method throws if it determines the input is an unsafe plaintext string.
     *
     * For example:
     * ```ts
     *     // Creates a new IAM user, access and secret keys, and stores the secret access key in a Secret.
     *     const user = new iam.User(this, 'User');
     *     const accessKey = new iam.AccessKey(this, 'AccessKey', { user });
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(accessKey.secretAccessKey.toString());
     *     new secretsmanager.Secret(this, 'Secret', {
     *       secretStringBeta1: secretValue,
     *     });
     * ```
     *
     * The secret may also be embedded in a string representation of a JSON structure:
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(JSON.stringify({
     *       username: user.userName,
     *       database: 'foo',
     *       password: accessKey.secretAccessKey.toString(),
     *     }));
     *
     * Note that the value being a Token does *not* guarantee safety. For example, a Lazy-evaluated string
     * (e.g., `Lazy.string({ produce: () => 'myInsecurePassword' }))`) is a Token, but as the output is
     * ultimately a plaintext string, and so insecure.
     *
     * @param secretValueFromToken a secret value coming from a Construct attribute or Custom Resource output
     */
    static fromToken(secretValueFromToken: string): SecretStringValueBeta1;
    private constructor();
    /** Returns the secret value */
    secretValue(): string;
}
/**
 * Attributes required to import an existing secret into the Stack.
 * One ARN format (`secretArn`, `secretCompleteArn`, `secretPartialArn`) must be provided.
 */
export interface SecretAttributes {
    /**
     * The encryption key that is used to encrypt the secret, unless the default SecretsManager key is used.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The ARN of the secret in SecretsManager.
     * Cannot be used with `secretCompleteArn` or `secretPartialArn`.
     * @deprecated use `secretCompleteArn` or `secretPartialArn` instead.
     */
    readonly secretArn?: string;
    /**
     * The complete ARN of the secret in SecretsManager. This is the ARN including the Secrets Manager 6-character suffix.
     * Cannot be used with `secretArn` or `secretPartialArn`.
     */
    readonly secretCompleteArn?: string;
    /**
     * The partial ARN of the secret in SecretsManager. This is the ARN without the Secrets Manager 6-character suffix.
     * Cannot be used with `secretArn` or `secretCompleteArn`.
     */
    readonly secretPartialArn?: string;
}
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
declare abstract class SecretBase extends Resource implements ISecret {
    abstract readonly encryptionKey?: kms.IKey;
    abstract readonly secretArn: string;
    abstract readonly secretName: string;
    protected abstract readonly autoCreatePolicy: boolean;
    private policy?;
    get secretFullArn(): string | undefined;
    grantRead(grantee: iam.IGrantable, versionStages?: string[]): iam.Grant;
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    get secretValue(): SecretValue;
    secretValueFromJson(jsonField: string): SecretValue;
    addRotationSchedule(id: string, options: RotationScheduleOptions): RotationSchedule;
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    protected validate(): string[];
    denyAccountRootDelete(): void;
    /**
     * Provides an identifier for this secret for use in IAM policies.
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     */
    protected get arnForPolicies(): string;
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target: ISecretAttachmentTarget): ISecret;
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
export declare class Secret extends SecretBase {
    /** @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn` */
    static fromSecretArn(scope: Construct, id: string, secretArn: string): ISecret;
    /** Imports a secret by complete ARN. The complete ARN is the ARN with the Secrets Manager-supplied suffix. */
    static fromSecretCompleteArn(scope: Construct, id: string, secretCompleteArn: string): ISecret;
    /** Imports a secret by partial ARN. The partial ARN is the ARN without the Secrets Manager-supplied suffix. */
    static fromSecretPartialArn(scope: Construct, id: string, secretPartialArn: string): ISecret;
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope: Construct, id: string, secretName: string): ISecret;
    /**
     * Imports a secret by secret name.
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     */
    static fromSecretNameV2(scope: Construct, id: string, secretName: string): ISecret;
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope: Construct, id: string, attrs: SecretAttributes): ISecret;
    readonly encryptionKey?: kms.IKey;
    readonly secretArn: string;
    readonly secretName: string;
    private replicaRegions;
    protected readonly autoCreatePolicy = true;
    constructor(scope: Construct, id: string, props?: SecretProps);
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id: string, options: AttachedSecretOptions): SecretTargetAttachment;
    /**
     * Adds a replica region for the secret
     *
     * @param region The name of the region
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     */
    addReplicaRegion(region: string, encryptionKey?: kms.IKey): void;
}
/**
 * A secret attachment target.
 */
export interface ISecretAttachmentTarget {
    /**
     * Renders the target specifications.
     */
    asSecretAttachmentTarget(): SecretAttachmentTargetProps;
}
/**
 * The type of service or database that's being associated with the secret.
 */
export declare enum AttachmentTargetType {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    INSTANCE = "AWS::RDS::DBInstance",
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    CLUSTER = "AWS::RDS::DBCluster",
    /**
     * AWS::RDS::DBInstance
     */
    RDS_DB_INSTANCE = "AWS::RDS::DBInstance",
    /**
     * AWS::RDS::DBCluster
     */
    RDS_DB_CLUSTER = "AWS::RDS::DBCluster",
    /**
     * AWS::RDS::DBProxy
     */
    RDS_DB_PROXY = "AWS::RDS::DBProxy",
    /**
     * AWS::Redshift::Cluster
     */
    REDSHIFT_CLUSTER = "AWS::Redshift::Cluster",
    /**
     * AWS::DocDB::DBInstance
     */
    DOCDB_DB_INSTANCE = "AWS::DocDB::DBInstance",
    /**
     * AWS::DocDB::DBCluster
     */
    DOCDB_DB_CLUSTER = "AWS::DocDB::DBCluster"
}
/**
 * Attachment target specifications.
 */
export interface SecretAttachmentTargetProps {
    /**
     * The id of the target to attach the secret to.
     */
    readonly targetId: string;
    /**
     * The type of the target to attach the secret to.
     */
    readonly targetType: AttachmentTargetType;
}
/**
 * Options to add a secret attachment to a secret.
 */
export interface AttachedSecretOptions {
    /**
     * The target to attach the secret to.
     */
    readonly target: ISecretAttachmentTarget;
}
/**
 * Construction properties for an AttachedSecret.
 */
export interface SecretTargetAttachmentProps extends AttachedSecretOptions {
    /**
     * The secret to attach to the target.
     */
    readonly secret: ISecret;
}
export interface ISecretTargetAttachment extends ISecret {
    /**
     * Same as `secretArn`
     *
     * @attribute
     */
    readonly secretTargetAttachmentSecretArn: string;
}
/**
 * An attached secret.
 */
export declare class SecretTargetAttachment extends SecretBase implements ISecretTargetAttachment {
    static fromSecretTargetAttachmentSecretArn(scope: Construct, id: string, secretTargetAttachmentSecretArn: string): ISecretTargetAttachment;
    readonly encryptionKey?: kms.IKey;
    readonly secretArn: string;
    readonly secretName: string;
    /**
     * @attribute
     */
    readonly secretTargetAttachmentSecretArn: string;
    protected readonly autoCreatePolicy = true;
    constructor(scope: Construct, id: string, props: SecretTargetAttachmentProps);
}
/**
 * Configuration to generate secrets such as passwords automatically.
 */
export interface SecretStringGenerator {
    /**
     * Specifies that the generated password shouldn't include uppercase letters.
     *
     * @default false
     */
    readonly excludeUppercase?: boolean;
    /**
     * Specifies whether the generated password must include at least one of every allowed character type.
     *
     * @default true
     */
    readonly requireEachIncludedType?: boolean;
    /**
     * Specifies that the generated password can include the space character.
     *
     * @default false
     */
    readonly includeSpace?: boolean;
    /**
     * A string that includes characters that shouldn't be included in the generated password. The string can be a minimum
     * of ``0`` and a maximum of ``4096`` characters long.
     *
     * @default no exclusions
     */
    readonly excludeCharacters?: string;
    /**
     * The desired length of the generated password.
     *
     * @default 32
     */
    readonly passwordLength?: number;
    /**
     * Specifies that the generated password shouldn't include punctuation characters.
     *
     * @default false
     */
    readonly excludePunctuation?: boolean;
    /**
     * Specifies that the generated password shouldn't include lowercase letters.
     *
     * @default false
     */
    readonly excludeLowercase?: boolean;
    /**
     * Specifies that the generated password shouldn't include digits.
     *
     * @default false
     */
    readonly excludeNumbers?: boolean;
    /**
     * A properly structured JSON string that the generated password can be added to. The ``generateStringKey`` is
     * combined with the generated random string and inserted into the JSON structure that's specified by this parameter.
     * The merged JSON string is returned as the completed SecretString of the secret. If you specify ``secretStringTemplate``
     * then ``generateStringKey`` must be also be specified.
     */
    readonly secretStringTemplate?: string;
    /**
     * The JSON key name that's used to add the generated password to the JSON structure specified by the
     * ``secretStringTemplate`` parameter. If you specify ``generateStringKey`` then ``secretStringTemplate``
     * must be also be specified.
     */
    readonly generateStringKey?: string;
}
export {};
