"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const s3 = require("../../aws-s3");
const core = require("../../core");
const s3objectlambda_generated_1 = require("./s3objectlambda.generated");
class AccessPointBase extends core.Resource {
    /** Implement the {@link IAccessPoint.domainName} field. */
    get domainName() {
        const urlSuffix = this.stack.urlSuffix;
        return `${this.accessPointName}-${this.stack.account}.s3-object-lambda.${urlSuffix}`;
    }
    /** Implement the {@link IAccessPoint.regionalDomainName} field. */
    get regionalDomainName() {
        const urlSuffix = this.stack.urlSuffix;
        const region = this.stack.region;
        return `${this.accessPointName}-${this.stack.account}.s3-object-lambda.${region}.${urlSuffix}`;
    }
    /** Implement the {@link IAccessPoint.virtualHostedUrlForObject} method. */
    virtualHostedUrlForObject(key, options) {
        var _b;
        const domainName = ((_b = options === null || options === void 0 ? void 0 : options.regional) !== null && _b !== void 0 ? _b : true) ? this.regionalDomainName : this.domainName;
        const prefix = `https://${domainName}`;
        if (!key) {
            return prefix;
        }
        if (key.startsWith('/')) {
            key = key.slice(1);
        }
        if (key.endsWith('/')) {
            key = key.slice(0, -1);
        }
        return `${prefix}/${key}`;
    }
}
/**
 * Checks the access point name against the rules in https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-names
 * @param name The name of the access point
 */
function validateAccessPointName(name) {
    if (name.length < 3 || name.length > 50) {
        throw new Error('Access point name must be between 3 and 50 characters long');
    }
    if (name.endsWith('-s3alias')) {
        throw new Error('Access point name cannot end with the suffix -s3alias');
    }
    if (name[0] === '-' || name[name.length - 1] === '-') {
        throw new Error('Access point name cannot begin or end with a dash');
    }
    if (!/^[0-9a-z](.(?![\.A-Z_]))+[0-9a-z]$/.test(name)) {
        throw new Error('Access point name must begin with a number or lowercase letter and not contain underscores, uppercase letters, or periods');
    }
}
/**
  * An S3 object lambda access point for intercepting and
  * transforming `GetObject` requests.
  */
class AccessPoint extends AccessPointBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.accessPointName,
        });
        jsiiDeprecationWarnings.monocdk_aws_s3objectlambda_AccessPointProps(props);
        if (props.accessPointName) {
            validateAccessPointName(props.accessPointName);
        }
        const supporting = new s3.CfnAccessPoint(this, 'SupportingAccessPoint', {
            bucket: props.bucket.bucketName,
        });
        const allowedFeatures = [];
        if (props.supportsGetObjectPartNumber) {
            allowedFeatures.push('GetObject-PartNumber');
        }
        if (props.supportsGetObjectRange) {
            allowedFeatures.push('GetObject-Range');
        }
        const accessPoint = new s3objectlambda_generated_1.CfnAccessPoint(this, id, {
            name: this.physicalName,
            objectLambdaConfiguration: {
                allowedFeatures,
                cloudWatchMetricsEnabled: props.cloudWatchMetricsEnabled,
                supportingAccessPoint: supporting.attrArn,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: {
                            AwsLambda: {
                                FunctionArn: props.handler.functionArn,
                                FunctionPayload: props.payload ? JSON.stringify(props.payload) : undefined,
                            },
                        },
                    },
                ],
            },
        });
        this.accessPointName = accessPoint.ref;
        this.accessPointArn = accessPoint.attrArn;
        this.accessPointCreationDate = accessPoint.attrCreationDate;
        props.handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3-object-lambda:WriteGetObjectResponse'],
            resources: ['*'],
        }));
    }
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_s3objectlambda_AccessPointAttributes(attrs);
        const arn = core.Arn.split(attrs.accessPointArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arn.resourceName) {
            throw new Error('Unable to parse acess point name');
        }
        const name = arn.resourceName;
        class Import extends AccessPointBase {
            constructor() {
                super(...arguments);
                this.accessPointArn = attrs.accessPointArn;
                this.accessPointCreationDate = attrs.accessPointCreationDate;
                this.accessPointName = name;
            }
        }
        return new Import(scope, id);
    }
}
exports.AccessPoint = AccessPoint;
_a = JSII_RTTI_SYMBOL_1;
AccessPoint[_a] = { fqn: "monocdk.aws_s3objectlambda.AccessPoint", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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