"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationsResourceHandler = exports.NotificationsResourceHandlerProps = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam");
const cdk = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../core");
class NotificationsResourceHandlerProps {
}
exports.NotificationsResourceHandlerProps = NotificationsResourceHandlerProps;
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.role = (_a = props.role) !== null && _a !== void 0 ? _a : new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*'],
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const handlerSource = fs.readFileSync(path.join(__dirname, 'lambda/index.py'), 'utf8');
        if (handlerSource.length > 4096) {
            throw new Error(`Source of Notifications Resource Handler is too large (${handlerSource.length} > 4096)`);
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: handlerSource },
                Handler: 'index.handler',
                Role: this.role.roleArn,
                Runtime: 'python3.7',
                Timeout: 300,
            },
        });
        resource.node.addDependency(this.role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context, props = {}) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId, props);
        }
        return lambda;
    }
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
//# sourceMappingURL=data:application/json;base64,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