"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("../../aws-efs");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const lambda_layer_awscli_1 = require("../../lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../core");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends core_1.Construct {
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        this.requestDestinationArn = false;
        jsiiDeprecationWarnings.monocdk_aws_s3_deployment_BucketDeploymentProps(props);
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // to avoid redundant stack updates, only include "SourceMarkers" if one of
        // the sources actually has markers.
        const hasMarkers = sources.some(source => source.markers);
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                SourceMarkers: hasMarkers ? sources.map(source => { var _d; return (_d = source.markers) !== null && _d !== void 0 ? _d : {}; }) : undefined,
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependencees on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? props.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.substr(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(props.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Doing this replaces calling `otherResource.node.addDependency(deployment)`.
     */
    get deployedBucket() {
        var _d;
        this.requestDestinationArn = true;
        this._deployedBucket = (_d = this._deployedBucket) !== null && _d !== void 0 ? _d : s3.Bucket.fromBucketArn(this, 'DestinationBucket', cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')));
        return this._deployedBucket;
    }
    renderUniqueId(memoryLimit, vpc) {
        let uuid = '';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if user specify to use VPC, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        var _d;
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return (_d = stack.node.tryFindChild(uuid)) !== null && _d !== void 0 ? _d : new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "monocdk.aws_s3_deployment.BucketDeployment", version: "1.149.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { jsiiDeprecationWarnings.monocdk_Duration(t); return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { jsiiDeprecationWarnings.monocdk_Duration(t); return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "monocdk.aws_s3_deployment.CacheControl", version: "1.149.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { jsiiDeprecationWarnings.print("monocdk.aws_s3_deployment.Expires#atDate", "use core.Expiration"); return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { jsiiDeprecationWarnings.print("monocdk.aws_s3_deployment.Expires#atTimestamp", "use core.Expiration"); return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { jsiiDeprecationWarnings.print("monocdk.aws_s3_deployment.Expires#after", "use core.Expiration"); jsiiDeprecationWarnings.monocdk_Duration(t); return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { jsiiDeprecationWarnings.print("monocdk.aws_s3_deployment.Expires#fromString", "use core.Expiration"); return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "monocdk.aws_s3_deployment.Expires", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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