import * as kms from '../../aws-kms';
import * as cdk from '../../core';
import { Construct } from 'constructs';
import { ICluster } from './cluster';
import { DatabaseOptions } from './database-options';
import { ITable, TableAction } from './table';
import { Construct as CoreConstruct } from '../../core';
/**
 * Properties for configuring a Redshift user.
 */
export interface UserProps extends DatabaseOptions {
    /**
     * The name of the user.
     *
     * For valid values, see: https://docs.aws.amazon.com/redshift/latest/dg/r_names.html
     *
     * @default - a name is generated
     */
    readonly username?: string;
    /**
     * KMS key to encrypt the generated secret.
     *
     * @default - the default AWS managed key is used
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The policy to apply when this resource is removed from the application.
     *
     * @default cdk.RemovalPolicy.Destroy
     */
    readonly removalPolicy?: cdk.RemovalPolicy;
}
/**
 * Represents a user in a Redshift database.
 */
export interface IUser extends cdk.IConstruct {
    /**
     * The name of the user.
     */
    readonly username: string;
    /**
     * The password of the user.
     */
    readonly password: cdk.SecretValue;
    /**
     * The cluster where the table is located.
     */
    readonly cluster: ICluster;
    /**
     * The name of the database where the table is located.
     */
    readonly databaseName: string;
    /**
     * Grant this user privilege to access a table.
     */
    addTablePrivileges(table: ITable, ...actions: TableAction[]): void;
}
/**
 * A full specification of a Redshift user that can be used to import it fluently into the CDK application.
 */
export interface UserAttributes extends DatabaseOptions {
    /**
     * The name of the user.
     */
    readonly username: string;
    /**
     * The password of the user.
     *
     * Do not put passwords in CDK code directly.
     */
    readonly password: cdk.SecretValue;
}
declare abstract class UserBase extends CoreConstruct implements IUser {
    abstract readonly username: string;
    abstract readonly password: cdk.SecretValue;
    abstract readonly cluster: ICluster;
    abstract readonly databaseName: string;
    /**
     * The tables that user will have access to
     */
    private privileges?;
    protected abstract readonly databaseProps: DatabaseOptions;
    addTablePrivileges(table: ITable, ...actions: TableAction[]): void;
}
/**
 * A user in a Redshift cluster.
 */
export declare class User extends UserBase {
    /**
     * Specify a Redshift user using credentials that already exist.
     */
    static fromUserAttributes(scope: Construct, id: string, attrs: UserAttributes): IUser;
    readonly username: string;
    readonly password: cdk.SecretValue;
    readonly cluster: ICluster;
    readonly databaseName: string;
    protected databaseProps: DatabaseOptions;
    private resource;
    constructor(scope: Construct, id: string, props: UserProps);
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is destroyed by default.
     */
    applyRemovalPolicy(policy: cdk.RemovalPolicy): void;
}
export {};
