import * as cdk from '../../core';
import { Construct } from 'constructs';
import { ICluster } from './cluster';
import { DatabaseOptions } from './database-options';
import { IUser } from './user';
import { Construct as CoreConstruct } from '../../core';
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
export declare enum TableAction {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    SELECT = 0,
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    INSERT = 1,
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    UPDATE = 2,
    /**
     * Grants privilege to delete a data row from a table.
     */
    DELETE = 3,
    /**
     * Grants privilege to drop a table.
     */
    DROP = 4,
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    REFERENCES = 5,
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    ALL = 6
}
/**
 * A column in a Redshift table.
 */
export interface Column {
    /**
     * The name of the column.
     */
    readonly name: string;
    /**
     * The data type of the column.
     */
    readonly dataType: string;
    /**
     * Boolean value that indicates whether the column is to be configured as DISTKEY.
     *
     * @default - column is not DISTKEY
     */
    readonly distKey?: boolean;
    /**
     * Boolean value that indicates whether the column is to be configured as SORTKEY.
     *
     * @default - column is not a SORTKEY
     */
    readonly sortKey?: boolean;
}
/**
 * Properties for configuring a Redshift table.
 */
export interface TableProps extends DatabaseOptions {
    /**
     * The name of the table.
     *
     * @default - a name is generated
     */
    readonly tableName?: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The distribution style of the table.
     *
     * @default TableDistStyle.AUTO
     */
    readonly distStyle?: TableDistStyle;
    /**
     * The sort style of the table.
     *
     * @default TableSortStyle.AUTO if no sort key is specified, TableSortStyle.COMPOUND if a sort key is specified
     */
    readonly sortStyle?: TableSortStyle;
    /**
     * The policy to apply when this resource is removed from the application.
     *
     * @default cdk.RemovalPolicy.Retain
     */
    readonly removalPolicy?: cdk.RemovalPolicy;
}
/**
 * Represents a table in a Redshift database.
 */
export interface ITable extends cdk.IConstruct {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The cluster where the table is located.
     */
    readonly cluster: ICluster;
    /**
     * The name of the database where the table is located.
     */
    readonly databaseName: string;
    /**
     * Grant a user privilege to access this table.
     */
    grant(user: IUser, ...actions: TableAction[]): void;
}
/**
 * A full specification of a Redshift table that can be used to import it fluently into the CDK application.
 */
export interface TableAttributes {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * The columns of the table.
     */
    readonly tableColumns: Column[];
    /**
     * The cluster where the table is located.
     */
    readonly cluster: ICluster;
    /**
     * The name of the database where the table is located.
     */
    readonly databaseName: string;
}
declare abstract class TableBase extends CoreConstruct implements ITable {
    abstract readonly tableName: string;
    abstract readonly tableColumns: Column[];
    abstract readonly cluster: ICluster;
    abstract readonly databaseName: string;
    grant(user: IUser, ...actions: TableAction[]): void;
}
/**
 * A table in a Redshift cluster.
 */
export declare class Table extends TableBase {
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope: Construct, id: string, attrs: TableAttributes): ITable;
    readonly tableName: string;
    readonly tableColumns: Column[];
    readonly cluster: ICluster;
    readonly databaseName: string;
    private resource;
    constructor(scope: Construct, id: string, props: TableProps);
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy: cdk.RemovalPolicy): void;
    private validateDistKeyColumns;
    private validateDistStyle;
    private validateSortStyle;
    private getDefaultSortStyle;
}
/**
 * The data distribution style of a table.
 */
export declare enum TableDistStyle {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    AUTO = "AUTO",
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    EVEN = "EVEN",
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    KEY = "KEY",
    /**
     * A copy of the entire table is distributed to every node.
     */
    ALL = "ALL"
}
/**
 * The sort style of a table.
 */
export declare enum TableSortStyle {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    AUTO = "AUTO",
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    COMPOUND = "COMPOUND",
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    INTERLEAVED = "INTERLEAVED"
}
export {};
