"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTablePrivileges = void 0;
const cdk = require("../../../core");
const table_1 = require("../table");
const database_query_1 = require("./database-query");
const handler_name_1 = require("./database-query-provider/handler-name");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../core");
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
class UserTablePrivileges extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.privileges = (_a = props.privileges) !== null && _a !== void 0 ? _a : [];
        new database_query_1.DatabaseQuery(this, 'Resource', {
            ...props,
            handler: handler_name_1.HandlerName.UserTablePrivileges,
            properties: {
                username: props.user.username,
                tablePrivileges: cdk.Lazy.any({
                    produce: () => {
                        const reducedPrivileges = this.privileges.reduce((privileges, { table, actions }) => {
                            const tableName = table.tableName;
                            if (!(tableName in privileges)) {
                                privileges[tableName] = [];
                            }
                            actions = actions.concat(privileges[tableName]);
                            if (actions.includes(table_1.TableAction.ALL)) {
                                actions = [table_1.TableAction.ALL];
                            }
                            if (actions.includes(table_1.TableAction.UPDATE) || actions.includes(table_1.TableAction.DELETE)) {
                                actions.push(table_1.TableAction.SELECT);
                            }
                            privileges[tableName] = Array.from(new Set(actions));
                            return privileges;
                        }, {});
                        const serializedPrivileges = Object.entries(reducedPrivileges).map(([tableName, actions]) => ({
                            tableName: tableName,
                            actions: actions.map(action => table_1.TableAction[action]),
                        }));
                        return serializedPrivileges;
                    },
                }),
            },
        });
    }
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table, ...actions) {
        this.privileges.push({ table, actions });
    }
}
exports.UserTablePrivileges = UserTablePrivileges;
//# sourceMappingURL=data:application/json;base64,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