"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClusterFromSnapshot = exports.ServerlessCluster = exports.AuroraCapacityUnit = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const secretsmanager = require("../../aws-secretsmanager");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const perms_1 = require("./perms");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * Aurora capacity units (ACUs).
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.setting-capacity.html
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 *
 */
var AuroraCapacityUnit;
(function (AuroraCapacityUnit) {
    /** 1 Aurora Capacity Unit */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_1"] = 1] = "ACU_1";
    /** 2 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_2"] = 2] = "ACU_2";
    /** 4 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_4"] = 4] = "ACU_4";
    /** 8 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_8"] = 8] = "ACU_8";
    /** 16 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_16"] = 16] = "ACU_16";
    /** 32 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_32"] = 32] = "ACU_32";
    /** 64 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_64"] = 64] = "ACU_64";
    /** 128 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_128"] = 128] = "ACU_128";
    /** 192 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_192"] = 192] = "ACU_192";
    /** 256 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_256"] = 256] = "ACU_256";
    /** 384 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_384"] = 384] = "ACU_384";
})(AuroraCapacityUnit = exports.AuroraCapacityUnit || (exports.AuroraCapacityUnit = {}));
/**
 * New or imported Serverless Cluster
 */
class ServerlessClusterBase extends core_1.Resource {
    /**
     * The ARN of the cluster
     */
    get clusterArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: this.clusterIdentifier,
        });
    }
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present
     *
     * @param grantee The principal to grant access to
     */
    grantDataApiAccess(grantee) {
        var _c;
        if (this.enableDataApi === false) {
            throw new Error('Cannot grant Data API access when the Data API is disabled');
        }
        this.enableDataApi = true;
        const ret = iam.Grant.addToPrincipal({
            grantee,
            actions: perms_1.DATA_API_ACTIONS,
            resourceArns: ['*'],
            scope: this,
        });
        (_c = this.secret) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        return ret;
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create an Aurora Serverless Cluster
 *
 * @resource AWS::RDS::DBCluster
 */
class ServerlessClusterNew extends ServerlessClusterBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        if (props.vpc === undefined) {
            if (props.vpcSubnets !== undefined) {
                throw new Error('A VPC is required to use vpcSubnets in ServerlessCluster. Please add a VPC or remove vpcSubnets');
            }
            if (props.subnetGroup !== undefined) {
                throw new Error('A VPC is required to use subnetGroup in ServerlessCluster. Please add a VPC or remove subnetGroup');
            }
            if (props.securityGroups !== undefined) {
                throw new Error('A VPC is required to use securityGroups in ServerlessCluster. Please add a VPC or remove securityGroups');
            }
        }
        let subnetGroup = props.subnetGroup;
        this.securityGroups = (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [];
        if (props.vpc !== undefined) {
            const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
            // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
            if (subnetIds.length < 2) {
                core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
            }
            subnetGroup = (_d = props.subnetGroup) !== null && _d !== void 0 ? _d : new subnet_group_1.SubnetGroup(this, 'Subnets', {
                description: `Subnets for ${id} database`,
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets,
                removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
            });
            this.securityGroups = (_e = props.securityGroups) !== null && _e !== void 0 ? _e : [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'RDS security group',
                    vpc: props.vpc,
                }),
            ];
        }
        if (props.backupRetention) {
            const backupRetentionDays = props.backupRetention.toDays();
            if (backupRetentionDays < 1 || backupRetentionDays > 35) {
                throw new Error(`backup retention period must be between 1 and 35 days. received: ${backupRetentionDays}`);
            }
        }
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_f = props.parameterGroup) !== null && _f !== void 0 ? _f : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_g = props.clusterIdentifier) === null || _g === void 0 ? void 0 : _g.toLowerCase() : props.clusterIdentifier;
        this.newCfnProps = {
            backupRetentionPeriod: (_h = props.backupRetention) === null || _h === void 0 ? void 0 : _h.toDays(),
            databaseName: props.defaultDatabaseName,
            dbClusterIdentifier: clusterIdentifier,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            dbSubnetGroupName: subnetGroup === null || subnetGroup === void 0 ? void 0 : subnetGroup.subnetGroupName,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            engine: props.engine.engineType,
            engineVersion: (_j = props.engine.engineVersion) === null || _j === void 0 ? void 0 : _j.fullVersion,
            engineMode: 'serverless',
            enableHttpEndpoint: core_1.Lazy.any({ produce: () => this.enableDataApi }),
            scalingConfiguration: props.scaling ? this.renderScalingConfiguration(props.scaling) : undefined,
            storageEncrypted: true,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
        };
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.number({ produce: () => this.clusterEndpoint.port })),
        });
    }
    renderScalingConfiguration(options) {
        var _c;
        const minCapacity = options.minCapacity;
        const maxCapacity = options.maxCapacity;
        if (minCapacity && maxCapacity && minCapacity > maxCapacity) {
            throw new Error('maximum capacity must be greater than or equal to minimum capacity.');
        }
        const secondsToAutoPause = (_c = options.autoPause) === null || _c === void 0 ? void 0 : _c.toSeconds();
        if (secondsToAutoPause && (secondsToAutoPause < 300 || secondsToAutoPause > 86400)) {
            throw new Error('auto pause time must be between 5 minutes and 1 day.');
        }
        return {
            autoPause: (secondsToAutoPause === 0) ? false : true,
            minCapacity: options.minCapacity,
            maxCapacity: options.maxCapacity,
            secondsUntilAutoPause: (secondsToAutoPause === 0) ? undefined : secondsToAutoPause,
        };
    }
}
/**
 * Create an Aurora Serverless Cluster
 *
 * @resource AWS::RDS::DBCluster
 *
 */
class ServerlessCluster extends ServerlessClusterNew {
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id, props);
        jsiiDeprecationWarnings.monocdk_aws_rds_ServerlessClusterProps(props);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.enableDataApi = props.enableDataApi;
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            masterUsername: credentials.username,
            masterUserPassword: (_c = credentials.password) === null || _c === void 0 ? void 0 : _c.toString(),
            kmsKeyId: (_d = props.storageEncryptionKey) === null || _d === void 0 ? void 0 : _d.keyArn,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        cluster.applyRemovalPolicy((_e = props.removalPolicy) !== null && _e !== void 0 ? _e : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromServerlessClusterAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_rds_ServerlessClusterAttributes(attrs);
        return new ImportedServerlessCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_rds_RotationSingleUserOptions(options);
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        if (this.vpc === undefined) {
            throw new Error('Cannot add single user rotation for a cluster without VPC.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcSubnets),
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        jsiiDeprecationWarnings.monocdk_aws_rds_RotationMultiUserOptions(options);
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        if (this.vpc === undefined) {
            throw new Error('Cannot add multi user rotation for a cluster without VPC.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcSubnets),
            secret: options.secret,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
}
exports.ServerlessCluster = ServerlessCluster;
_a = JSII_RTTI_SYMBOL_1;
ServerlessCluster[_a] = { fqn: "monocdk.aws_rds.ServerlessCluster", version: "1.149.0" };
/**
 * Represents an imported database cluster.
 */
class ImportedServerlessCluster extends ServerlessClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.enableDataApi = true;
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.secret = attrs.secret;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
}
/**
 * A Aurora Serverless Cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBCluster
 */
class ServerlessClusterFromSnapshot extends ServerlessClusterNew {
    constructor(scope, id, props) {
        var _c, _d, _e, _f;
        super(scope, id, props);
        jsiiDeprecationWarnings.monocdk_aws_rds_ServerlessClusterFromSnapshotProps(props);
        this.enableDataApi = props.enableDataApi;
        let credentials = props.credentials;
        let secret = credentials === null || credentials === void 0 ? void 0 : credentials.secret;
        if (!secret && (credentials === null || credentials === void 0 ? void 0 : credentials.generatePassword)) {
            if (!credentials.username) {
                throw new Error('`credentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
                replicaRegions: credentials.replicaRegions,
            });
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: (_d = (_c = secret === null || secret === void 0 ? void 0 : secret.secretValueFromJson('password')) === null || _c === void 0 ? void 0 : _c.toString()) !== null && _d !== void 0 ? _d : (_e = credentials === null || credentials === void 0 ? void 0 : credentials.password) === null || _e === void 0 ? void 0 : _e.toString(),
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        cluster.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
}
exports.ServerlessClusterFromSnapshot = ServerlessClusterFromSnapshot;
_b = JSII_RTTI_SYMBOL_1;
ServerlessClusterFromSnapshot[_b] = { fqn: "monocdk.aws_rds.ServerlessClusterFromSnapshot", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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