"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const secretsmanager = require("../../aws-secretsmanager");
const cdk = require("../../core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
_a = JSII_RTTI_SYMBOL_1;
SessionPinningFilter[_a] = { fqn: "monocdk.aws_rds.SessionPinningFilter", version: "1.149.0" };
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        jsiiDeprecationWarnings.monocdk_aws_rds_IDatabaseInstance(instance);
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        jsiiDeprecationWarnings.monocdk_aws_rds_IDatabaseCluster(cluster);
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(proxy) {
        var _d, _e, _f, _g, _h, _j;
        jsiiDeprecationWarnings.monocdk_aws_rds_DatabaseProxy(proxy);
        const engine = (_e = (_d = this.dbInstance) === null || _d === void 0 ? void 0 : _d.engine) !== null && _e !== void 0 ? _e : (_f = this.dbCluster) === null || _f === void 0 ? void 0 : _f.engine;
        if (!engine) {
            const errorResource = (_g = this.dbCluster) !== null && _g !== void 0 ? _g : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        // allow connecting to the Cluster/Instance from the Proxy
        (_h = this.dbCluster) === null || _h === void 0 ? void 0 : _h.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Cluster from the Proxy');
        (_j = this.dbInstance) === null || _j === void 0 ? void 0 : _j.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Instance from the Proxy');
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
_b = JSII_RTTI_SYMBOL_1;
ProxyTarget[_b] = { fqn: "monocdk.aws_rds.ProxyTarget", version: "1.149.0" };
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            throw new Error('For imported Database Proxies, the dbUser is required in grantConnect()');
        }
        const scopeStack = cdk.Stack.of(this);
        const proxyGeneratedId = scopeStack.splitArn(this.dbProxyArn, cdk.ArnFormat.COLON_RESOURCE_NAME).resourceName;
        const userArn = scopeStack.formatArn({
            service: 'rds-db',
            resource: 'dbuser',
            resourceName: `${proxyGeneratedId}/${dbUser}`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [userArn],
        });
    }
}
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g;
        super(scope, id, { physicalName: props.dbProxyName || id });
        jsiiDeprecationWarnings.monocdk_aws_rds_DatabaseProxyProps(props);
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new ec2.SecurityGroup(this, 'ProxySecurityGroup', {
                description: 'SecurityGroup for Database Proxy',
                vpc: props.vpc,
            }),
        ];
        this.connections = new ec2.Connections({ securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.secrets = props.secrets;
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_e = props.idleClientTimeout) === null || _e === void 0 ? void 0 : _e.toSeconds(),
            requireTls: (_f = props.requireTLS) !== null && _f !== void 0 ? _f : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(_ => _.securityGroupId) }),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        (_g = bindResult.dbClusters) === null || _g === void 0 ? void 0 : _g.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_rds_DatabaseProxyAttributes(attrs);
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
    grantConnect(grantee, dbUser) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        if (!dbUser) {
            if (this.secrets.length > 1) {
                throw new Error('When the Proxy contains multiple Secrets, you must pass a dbUser explicitly to grantConnect()');
            }
            // 'username' is the field RDS uses here,
            // see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy-secrets-arns
            dbUser = this.secrets[0].secretValueFromJson('username').toString();
        }
        return super.grantConnect(grantee, dbUser);
    }
}
exports.DatabaseProxy = DatabaseProxy;
_c = JSII_RTTI_SYMBOL_1;
DatabaseProxy[_c] = { fqn: "monocdk.aws_rds.DatabaseProxy", version: "1.149.0" };
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _d, _e;
    return {
        connectionBorrowTimeout: (_d = props.borrowTimeout) === null || _d === void 0 ? void 0 : _d.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_e = props.sessionPinningFilters) === null || _e === void 0 ? void 0 : _e.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJveHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwcm94eS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFDckMscUNBQXFDO0FBQ3JDLDJEQUEyRDtBQUMzRCxrQ0FBa0M7QUFLbEMseUNBQW1EO0FBQ25ELG1EQUFvRTtBQUVwRTs7OztHQUlHO0FBQ0gsTUFBYSxvQkFBb0I7SUFlL0I7SUFDRTs7T0FFRztJQUNhLFVBQWtCO1FBQWxCLGVBQVUsR0FBVixVQUFVLENBQVE7S0FDaEM7SUFaSjs7T0FFRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsVUFBa0I7UUFDakMsT0FBTyxJQUFJLG9CQUFvQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0tBQzdDOztBQWJILG9EQXFCQzs7O0FBcEJDOzs7O0dBSUc7QUFDb0IsMENBQXFCLEdBQUcsSUFBSSxvQkFBb0IsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0FBaUJuRzs7Ozs7R0FLRztBQUNILE1BQWEsV0FBVztJQW1CdEIsWUFDbUIsVUFBeUMsRUFDekMsU0FBdUM7UUFEdkMsZUFBVSxHQUFWLFVBQVUsQ0FBK0I7UUFDekMsY0FBUyxHQUFULFNBQVMsQ0FBOEI7S0FDekQ7SUFyQkQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBMkI7O1FBQ3BELE9BQU8sSUFBSSxXQUFXLENBQUMsUUFBUSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0tBQzdDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsT0FBeUI7O1FBQ2pELE9BQU8sSUFBSSxXQUFXLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQzVDO0lBT0Q7O09BRUc7SUFDSSxJQUFJLENBQUMsS0FBb0I7OztRQUM5QixNQUFNLE1BQU0sZUFBd0IsSUFBSSxDQUFDLFVBQVUsMENBQUUsTUFBTSx5Q0FBSSxJQUFJLENBQUMsU0FBUywwQ0FBRSxNQUFNLENBQUM7UUFFdEYsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLE1BQU0sYUFBYSxTQUFHLElBQUksQ0FBQyxTQUFTLG1DQUFJLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsYUFBYSxhQUFiLGFBQWEsdUJBQWIsYUFBYSxDQUFFLElBQUksQ0FBQyxJQUFJLEtBQUs7Z0JBQzNGLDBEQUEwRCxDQUFDLENBQUM7U0FDL0Q7UUFFRCxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDO1FBQ3pDLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDakIsTUFBTSxJQUFJLEtBQUssQ0FBQyxXQUFXLHdCQUFpQixDQUFDLE1BQU0sQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1NBQ25GO1FBRUQsMERBQTBEO1FBQzFELE1BQUEsSUFBSSxDQUFDLFNBQVMsMENBQUUsV0FBVyxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRSwwREFBMEQsRUFBRTtRQUNwSCxNQUFBLElBQUksQ0FBQyxVQUFVLDBDQUFFLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsMkRBQTJELEVBQUU7UUFFdEgsT0FBTztZQUNMLFlBQVk7WUFDWixVQUFVLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDekQsV0FBVyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQzdELENBQUM7S0FDSDs7QUFsREgsa0NBbURDOzs7QUE4T0Q7OztHQUdHO0FBQ0gsTUFBZSxpQkFBa0IsU0FBUSxHQUFHLENBQUMsUUFBUTtJQUs1QyxZQUFZLENBQUMsT0FBdUIsRUFBRSxNQUFlO1FBQzFELElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLHlFQUF5RSxDQUFDLENBQUM7U0FDNUY7UUFDRCxNQUFNLFVBQVUsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN0QyxNQUFNLGdCQUFnQixHQUFHLFVBQVUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQixDQUFDLENBQUMsWUFBWSxDQUFDO1FBQzlHLE1BQU0sT0FBTyxHQUFHLFVBQVUsQ0FBQyxTQUFTLENBQUM7WUFDbkMsT0FBTyxFQUFFLFFBQVE7WUFDakIsUUFBUSxFQUFFLFFBQVE7WUFDbEIsWUFBWSxFQUFFLEdBQUcsZ0JBQWdCLElBQUksTUFBTSxFQUFFO1lBQzdDLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7UUFDSCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQztZQUMzQixZQUFZLEVBQUUsQ0FBQyxPQUFPLENBQUM7U0FDeEIsQ0FBQyxDQUFDO0tBQ0o7Q0FDRjtBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLGFBQWMsU0FBUSxpQkFBaUI7SUErQ2xELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7O1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsWUFBWSxFQUFFLEtBQUssQ0FBQyxXQUFXLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQzs7UUFFNUQsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUN2RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBRUgsS0FBSyxNQUFNLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ2xDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDeEI7UUFFRCxNQUFNLGNBQWMsU0FBRyxLQUFLLENBQUMsY0FBYyxtQ0FBSTtZQUM3QyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO2dCQUNoRCxXQUFXLEVBQUUsa0NBQWtDO2dCQUMvQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7YUFDZixDQUFDO1NBQ0gsQ0FBQztRQUNGLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUUzRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVoRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7U0FDdEQ7UUFDRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFFN0IsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLDBCQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMvQyxJQUFJLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQzFCLE9BQU87b0JBQ0wsVUFBVSxFQUFFLFNBQVM7b0JBQ3JCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVU7b0JBQ2hELFNBQVMsRUFBRSxDQUFDLENBQUMsU0FBUztpQkFDdkIsQ0FBQztZQUNKLENBQUMsQ0FBQztZQUNGLFdBQVcsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM5QixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsWUFBWSxFQUFFLFVBQVUsQ0FBQyxZQUFZO1lBQ3JDLGlCQUFpQixRQUFFLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsU0FBUyxFQUFFO1lBQ3ZELFVBQVUsUUFBRSxLQUFLLENBQUMsVUFBVSxtQ0FBSSxJQUFJO1lBQ3BDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixtQkFBbUIsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLEVBQUUsQ0FBQztZQUNsSCxZQUFZLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLFNBQVM7U0FDbEUsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUNyQyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDO1FBQy9DLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7UUFFM0MsSUFBSSxxQkFBMkMsQ0FBQztRQUNoRCxJQUFJLFVBQVUsQ0FBQyxXQUFXLEVBQUU7WUFDMUIsbUNBQW1DO1lBQ25DLHFCQUFxQixHQUFHLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3hFO1FBRUQsSUFBSSxvQkFBMEMsQ0FBQztRQUMvQyxJQUFJLFVBQVUsQ0FBQyxVQUFVLEVBQUU7WUFDekIsb0JBQW9CLEdBQUcsVUFBVSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsSUFBSSxDQUFDLENBQUMscUJBQXFCLElBQUksQ0FBQyxDQUFDLG9CQUFvQixFQUFFO1lBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztTQUN2RjtRQUVELE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxxQ0FBcUIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDM0UsZUFBZSxFQUFFLFNBQVM7WUFDMUIsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzdCLHFCQUFxQjtZQUNyQixvQkFBb0I7WUFDcEIsK0JBQStCLEVBQUUsaUNBQWlDLENBQUMsS0FBSyxDQUFDO1NBQzFFLENBQUMsQ0FBQztRQUVILE1BQUEsVUFBVSxDQUFDLFVBQVUsMENBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxFQUFFO0tBQy9FO0lBckhEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLDJCQUEyQixDQUN2QyxLQUFnQixFQUNoQixFQUFVLEVBQ1YsS0FBOEI7O1FBRTlCLE1BQU0sTUFBTyxTQUFRLGlCQUFpQjtZQUF0Qzs7Z0JBQ2tCLGdCQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztnQkFDaEMsZUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7Z0JBQzlCLGFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQzVDLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBeUdEOztPQUVHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU87WUFDTCxRQUFRLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDMUIsVUFBVSxFQUFFLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZO1NBQzdELENBQUM7S0FDSDtJQUVNLFlBQVksQ0FBQyxPQUF1QixFQUFFLE1BQWU7O1FBQzFELElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDWCxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQywrRkFBK0YsQ0FBQyxDQUFDO2FBQ2xIO1lBQ0QseUNBQXlDO1lBQ3pDLHlHQUF5RztZQUN6RyxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNyRTtRQUNELE9BQU8sS0FBSyxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLENBQUM7S0FDNUM7O0FBN0lILHNDQThJQzs7O0FBRUQ7O0dBRUc7QUFDSCxTQUFTLGlDQUFpQyxDQUN4QyxLQUF5Qjs7SUFFekIsT0FBTztRQUNMLHVCQUF1QixRQUFFLEtBQUssQ0FBQyxhQUFhLDBDQUFFLFNBQVMsRUFBRTtRQUN6RCxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7UUFDMUIscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtRQUNsRCx5QkFBeUIsRUFBRSxLQUFLLENBQUMseUJBQXlCO1FBQzFELHFCQUFxQixRQUFFLEtBQUssQ0FBQyxxQkFBcUIsMENBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztLQUMzRSxDQUFDO0FBQ0osQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICcuLi8uLi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICcuLi8uLi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNlY3JldHNtYW5hZ2VyIGZyb20gJy4uLy4uL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IElEYXRhYmFzZUNsdXN0ZXIgfSBmcm9tICcuL2NsdXN0ZXItcmVmJztcbmltcG9ydCB7IElFbmdpbmUgfSBmcm9tICcuL2VuZ2luZSc7XG5pbXBvcnQgeyBJRGF0YWJhc2VJbnN0YW5jZSB9IGZyb20gJy4vaW5zdGFuY2UnO1xuaW1wb3J0IHsgZW5naW5lRGVzY3JpcHRpb24gfSBmcm9tICcuL3ByaXZhdGUvdXRpbCc7XG5pbXBvcnQgeyBDZm5EQlByb3h5LCBDZm5EQlByb3h5VGFyZ2V0R3JvdXAgfSBmcm9tICcuL3Jkcy5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIFNlc3Npb25QaW5uaW5nRmlsdGVyXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvcmRzLXByb3h5Lmh0bWwjcmRzLXByb3h5LXBpbm5pbmdcbiAqL1xuZXhwb3J0IGNsYXNzIFNlc3Npb25QaW5uaW5nRmlsdGVyIHtcbiAgLyoqXG4gICAqIFlvdSBjYW4gb3B0IG91dCBvZiBzZXNzaW9uIHBpbm5pbmcgZm9yIHRoZSBmb2xsb3dpbmcga2luZHMgb2YgYXBwbGljYXRpb24gc3RhdGVtZW50czpcbiAgICpcbiAgICogLSBTZXR0aW5nIHNlc3Npb24gdmFyaWFibGVzIGFuZCBjb25maWd1cmF0aW9uIHNldHRpbmdzLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFWENMVURFX1ZBUklBQkxFX1NFVFMgPSBuZXcgU2Vzc2lvblBpbm5pbmdGaWx0ZXIoJ0VYQ0xVREVfVkFSSUFCTEVfU0VUUycpO1xuXG4gIC8qKlxuICAgKiBjdXN0b20gZmlsdGVyXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG9mKGZpbHRlck5hbWU6IHN0cmluZyk6IFNlc3Npb25QaW5uaW5nRmlsdGVyIHtcbiAgICByZXR1cm4gbmV3IFNlc3Npb25QaW5uaW5nRmlsdGVyKGZpbHRlck5hbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihcbiAgICAvKipcbiAgICAgKiBGaWx0ZXIgbmFtZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBmaWx0ZXJOYW1lOiBzdHJpbmcsXG4gICkge31cbn1cblxuLyoqXG4gKiBQcm94eSB0YXJnZXQ6IEluc3RhbmNlIG9yIENsdXN0ZXJcbiAqXG4gKiBBIHRhcmdldCBncm91cCBpcyBhIGNvbGxlY3Rpb24gb2YgZGF0YWJhc2VzIHRoYXQgdGhlIHByb3h5IGNhbiBjb25uZWN0IHRvLlxuICogQ3VycmVudGx5LCB5b3UgY2FuIHNwZWNpZnkgb25seSBvbmUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyLlxuICovXG5leHBvcnQgY2xhc3MgUHJveHlUYXJnZXQge1xuICAvKipcbiAgICogRnJvbSBpbnN0YW5jZVxuICAgKlxuICAgKiBAcGFyYW0gaW5zdGFuY2UgUkRTIGRhdGFiYXNlIGluc3RhbmNlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21JbnN0YW5jZShpbnN0YW5jZTogSURhdGFiYXNlSW5zdGFuY2UpOiBQcm94eVRhcmdldCB7XG4gICAgcmV0dXJuIG5ldyBQcm94eVRhcmdldChpbnN0YW5jZSwgdW5kZWZpbmVkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGcm9tIGNsdXN0ZXJcbiAgICpcbiAgICogQHBhcmFtIGNsdXN0ZXIgUkRTIGRhdGFiYXNlIGNsdXN0ZXJcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNsdXN0ZXIoY2x1c3RlcjogSURhdGFiYXNlQ2x1c3Rlcik6IFByb3h5VGFyZ2V0IHtcbiAgICByZXR1cm4gbmV3IFByb3h5VGFyZ2V0KHVuZGVmaW5lZCwgY2x1c3Rlcik7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKFxuICAgIHByaXZhdGUgcmVhZG9ubHkgZGJJbnN0YW5jZTogSURhdGFiYXNlSW5zdGFuY2UgfCB1bmRlZmluZWQsXG4gICAgcHJpdmF0ZSByZWFkb25seSBkYkNsdXN0ZXI6IElEYXRhYmFzZUNsdXN0ZXIgfCB1bmRlZmluZWQpIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBCaW5kIHRoaXMgdGFyZ2V0IHRvIHRoZSBzcGVjaWZpZWQgZGF0YWJhc2UgcHJveHkuXG4gICAqL1xuICBwdWJsaWMgYmluZChwcm94eTogRGF0YWJhc2VQcm94eSk6IFByb3h5VGFyZ2V0Q29uZmlnIHtcbiAgICBjb25zdCBlbmdpbmU6IElFbmdpbmUgfCB1bmRlZmluZWQgPSB0aGlzLmRiSW5zdGFuY2U/LmVuZ2luZSA/PyB0aGlzLmRiQ2x1c3Rlcj8uZW5naW5lO1xuXG4gICAgaWYgKCFlbmdpbmUpIHtcbiAgICAgIGNvbnN0IGVycm9yUmVzb3VyY2UgPSB0aGlzLmRiQ2x1c3RlciA/PyB0aGlzLmRiSW5zdGFuY2U7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENvdWxkIG5vdCBkZXRlcm1pbmUgZW5naW5lIGZvciBwcm94eSB0YXJnZXQgJyR7ZXJyb3JSZXNvdXJjZT8ubm9kZS5wYXRofScuIGAgK1xuICAgICAgICAnUGxlYXNlIHByb3ZpZGUgaXQgZXhwbGljaXRseSB3aGVuIGltcG9ydGluZyB0aGUgcmVzb3VyY2UnKTtcbiAgICB9XG5cbiAgICBjb25zdCBlbmdpbmVGYW1pbHkgPSBlbmdpbmUuZW5naW5lRmFtaWx5O1xuICAgIGlmICghZW5naW5lRmFtaWx5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEVuZ2luZSAnJHtlbmdpbmVEZXNjcmlwdGlvbihlbmdpbmUpfScgZG9lcyBub3Qgc3VwcG9ydCBwcm94aWVzYCk7XG4gICAgfVxuXG4gICAgLy8gYWxsb3cgY29ubmVjdGluZyB0byB0aGUgQ2x1c3Rlci9JbnN0YW5jZSBmcm9tIHRoZSBQcm94eVxuICAgIHRoaXMuZGJDbHVzdGVyPy5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbShwcm94eSwgJ0FsbG93IGNvbm5lY3Rpb25zIHRvIHRoZSBkYXRhYmFzZSBDbHVzdGVyIGZyb20gdGhlIFByb3h5Jyk7XG4gICAgdGhpcy5kYkluc3RhbmNlPy5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbShwcm94eSwgJ0FsbG93IGNvbm5lY3Rpb25zIHRvIHRoZSBkYXRhYmFzZSBJbnN0YW5jZSBmcm9tIHRoZSBQcm94eScpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGVuZ2luZUZhbWlseSxcbiAgICAgIGRiQ2x1c3RlcnM6IHRoaXMuZGJDbHVzdGVyID8gW3RoaXMuZGJDbHVzdGVyXSA6IHVuZGVmaW5lZCxcbiAgICAgIGRiSW5zdGFuY2VzOiB0aGlzLmRiSW5zdGFuY2UgPyBbdGhpcy5kYkluc3RhbmNlXSA6IHVuZGVmaW5lZCxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogVGhlIHJlc3VsdCBvZiBiaW5kaW5nIGEgYFByb3h5VGFyZ2V0YCB0byBhIGBEYXRhYmFzZVByb3h5YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcm94eVRhcmdldENvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgZW5naW5lIGZhbWlseSBvZiB0aGUgZGF0YWJhc2UgaW5zdGFuY2Ugb3IgY2x1c3RlciB0aGlzIHByb3h5IGNvbm5lY3RzIHdpdGguXG4gICAqL1xuICByZWFkb25seSBlbmdpbmVGYW1pbHk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIGluc3RhbmNlcyB0byB3aGljaCB0aGlzIHByb3h5IGNvbm5lY3RzLlxuICAgKiBFaXRoZXIgdGhpcyBvciBgZGJDbHVzdGVyc2Agd2lsbCBiZSBzZXQgYW5kIHRoZSBvdGhlciBgdW5kZWZpbmVkYC5cbiAgICogQGRlZmF1bHQgLSBgdW5kZWZpbmVkYCBpZiBgZGJDbHVzdGVyc2AgaXMgc2V0LlxuICAgKi9cbiAgcmVhZG9ubHkgZGJJbnN0YW5jZXM/OiBJRGF0YWJhc2VJbnN0YW5jZVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgZGF0YWJhc2UgY2x1c3RlcnMgdG8gd2hpY2ggdGhpcyBwcm94eSBjb25uZWN0cy5cbiAgICogRWl0aGVyIHRoaXMgb3IgYGRiSW5zdGFuY2VzYCB3aWxsIGJlIHNldCBhbmQgdGhlIG90aGVyIGB1bmRlZmluZWRgLlxuICAgKiBAZGVmYXVsdCAtIGB1bmRlZmluZWRgIGlmIGBkYkluc3RhbmNlc2AgaXMgc2V0LlxuICAgKi9cbiAgcmVhZG9ubHkgZGJDbHVzdGVycz86IElEYXRhYmFzZUNsdXN0ZXJbXTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBhIG5ldyBEYXRhYmFzZVByb3h5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VQcm94eU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGlkZW50aWZpZXIgZm9yIHRoZSBwcm94eS5cbiAgICogVGhpcyBuYW1lIG11c3QgYmUgdW5pcXVlIGZvciBhbGwgcHJveGllcyBvd25lZCBieSB5b3VyIEFXUyBhY2NvdW50IGluIHRoZSBzcGVjaWZpZWQgQVdTIFJlZ2lvbi5cbiAgICogQW4gaWRlbnRpZmllciBtdXN0IGJlZ2luIHdpdGggYSBsZXR0ZXIgYW5kIG11c3QgY29udGFpbiBvbmx5IEFTQ0lJIGxldHRlcnMsIGRpZ2l0cywgYW5kIGh5cGhlbnM7XG4gICAqIGl0IGNhbid0IGVuZCB3aXRoIGEgaHlwaGVuIG9yIGNvbnRhaW4gdHdvIGNvbnNlY3V0aXZlIGh5cGhlbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gR2VuZXJhdGVkIGJ5IENsb3VkRm9ybWF0aW9uIChyZWNvbW1lbmRlZClcbiAgICovXG4gIHJlYWRvbmx5IGRiUHJveHlOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZHVyYXRpb24gZm9yIGEgcHJveHkgdG8gd2FpdCBmb3IgYSBjb25uZWN0aW9uIHRvIGJlY29tZSBhdmFpbGFibGUgaW4gdGhlIGNvbm5lY3Rpb24gcG9vbC5cbiAgICogT25seSBhcHBsaWVzIHdoZW4gdGhlIHByb3h5IGhhcyBvcGVuZWQgaXRzIG1heGltdW0gbnVtYmVyIG9mIGNvbm5lY3Rpb25zIGFuZCBhbGwgY29ubmVjdGlvbnMgYXJlIGJ1c3kgd2l0aCBjbGllbnRcbiAgICogc2Vzc2lvbnMuXG4gICAqXG4gICAqIFZhbHVlIG11c3QgYmUgYmV0d2VlbiAxIHNlY29uZCBhbmQgMSBob3VyLCBvciBgRHVyYXRpb24uc2Vjb25kcygwKWAgdG8gcmVwcmVzZW50IHVubGltaXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgY2RrLkR1cmF0aW9uLnNlY29uZHMoMTIwKVxuICAgKi9cbiAgcmVhZG9ubHkgYm9ycm93VGltZW91dD86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogT25lIG9yIG1vcmUgU1FMIHN0YXRlbWVudHMgZm9yIHRoZSBwcm94eSB0byBydW4gd2hlbiBvcGVuaW5nIGVhY2ggbmV3IGRhdGFiYXNlIGNvbm5lY3Rpb24uXG4gICAqIFR5cGljYWxseSB1c2VkIHdpdGggU0VUIHN0YXRlbWVudHMgdG8gbWFrZSBzdXJlIHRoYXQgZWFjaCBjb25uZWN0aW9uIGhhcyBpZGVudGljYWwgc2V0dGluZ3Mgc3VjaCBhcyB0aW1lIHpvbmVcbiAgICogYW5kIGNoYXJhY3RlciBzZXQuXG4gICAqIEZvciBtdWx0aXBsZSBzdGF0ZW1lbnRzLCB1c2Ugc2VtaWNvbG9ucyBhcyB0aGUgc2VwYXJhdG9yLlxuICAgKiBZb3UgY2FuIGFsc28gaW5jbHVkZSBtdWx0aXBsZSB2YXJpYWJsZXMgaW4gYSBzaW5nbGUgU0VUIHN0YXRlbWVudCwgc3VjaCBhcyBTRVQgeD0xLCB5PTIuXG4gICAqXG4gICAqIG5vdCBjdXJyZW50bHkgc3VwcG9ydGVkIGZvciBQb3N0Z3JlU1FMLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGluaXRpYWxpemF0aW9uIHF1ZXJ5XG4gICAqL1xuICByZWFkb25seSBpbml0UXVlcnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIHNpemUgb2YgdGhlIGNvbm5lY3Rpb24gcG9vbCBmb3IgZWFjaCB0YXJnZXQgaW4gYSB0YXJnZXQgZ3JvdXAuXG4gICAqIEZvciBBdXJvcmEgTXlTUUwsIGl0IGlzIGV4cHJlc3NlZCBhcyBhIHBlcmNlbnRhZ2Ugb2YgdGhlIG1heF9jb25uZWN0aW9ucyBzZXR0aW5nIGZvciB0aGUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQlxuICAgKiBjbHVzdGVyIHVzZWQgYnkgdGhlIHRhcmdldCBncm91cC5cbiAgICpcbiAgICogMS0xMDBcbiAgICpcbiAgICogQGRlZmF1bHQgMTAwXG4gICAqL1xuICByZWFkb25seSBtYXhDb25uZWN0aW9uc1BlcmNlbnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIENvbnRyb2xzIGhvdyBhY3RpdmVseSB0aGUgcHJveHkgY2xvc2VzIGlkbGUgZGF0YWJhc2UgY29ubmVjdGlvbnMgaW4gdGhlIGNvbm5lY3Rpb24gcG9vbC5cbiAgICogQSBoaWdoIHZhbHVlIGVuYWJsZXMgdGhlIHByb3h5IHRvIGxlYXZlIGEgaGlnaCBwZXJjZW50YWdlIG9mIGlkbGUgY29ubmVjdGlvbnMgb3Blbi5cbiAgICogQSBsb3cgdmFsdWUgY2F1c2VzIHRoZSBwcm94eSB0byBjbG9zZSBpZGxlIGNsaWVudCBjb25uZWN0aW9ucyBhbmQgcmV0dXJuIHRoZSB1bmRlcmx5aW5nIGRhdGFiYXNlIGNvbm5lY3Rpb25zXG4gICAqIHRvIHRoZSBjb25uZWN0aW9uIHBvb2wuXG4gICAqIEZvciBBdXJvcmEgTXlTUUwsIGl0IGlzIGV4cHJlc3NlZCBhcyBhIHBlcmNlbnRhZ2Ugb2YgdGhlIG1heF9jb25uZWN0aW9ucyBzZXR0aW5nIGZvciB0aGUgUkRTIERCIGluc3RhbmNlXG4gICAqIG9yIEF1cm9yYSBEQiBjbHVzdGVyIHVzZWQgYnkgdGhlIHRhcmdldCBncm91cC5cbiAgICpcbiAgICogYmV0d2VlbiAwIGFuZCBNYXhDb25uZWN0aW9uc1BlcmNlbnRcbiAgICpcbiAgICogQGRlZmF1bHQgNTBcbiAgICovXG4gIHJlYWRvbmx5IG1heElkbGVDb25uZWN0aW9uc1BlcmNlbnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEVhY2ggaXRlbSBpbiB0aGUgbGlzdCByZXByZXNlbnRzIGEgY2xhc3Mgb2YgU1FMIG9wZXJhdGlvbnMgdGhhdCBub3JtYWxseSBjYXVzZSBhbGwgbGF0ZXIgc3RhdGVtZW50cyBpbiBhIHNlc3Npb25cbiAgICogdXNpbmcgYSBwcm94eSB0byBiZSBwaW5uZWQgdG8gdGhlIHNhbWUgdW5kZXJseWluZyBkYXRhYmFzZSBjb25uZWN0aW9uLlxuICAgKiBJbmNsdWRpbmcgYW4gaXRlbSBpbiB0aGUgbGlzdCBleGVtcHRzIHRoYXQgY2xhc3Mgb2YgU1FMIG9wZXJhdGlvbnMgZnJvbSB0aGUgcGlubmluZyBiZWhhdmlvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBzZXNzaW9uIHBpbm5pbmcgZmlsdGVyc1xuICAgKi9cbiAgcmVhZG9ubHkgc2Vzc2lvblBpbm5pbmdGaWx0ZXJzPzogU2Vzc2lvblBpbm5pbmdGaWx0ZXJbXTtcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgcHJveHkgaW5jbHVkZXMgZGV0YWlsZWQgaW5mb3JtYXRpb24gYWJvdXQgU1FMIHN0YXRlbWVudHMgaW4gaXRzIGxvZ3MuXG4gICAqIFRoaXMgaW5mb3JtYXRpb24gaGVscHMgeW91IHRvIGRlYnVnIGlzc3VlcyBpbnZvbHZpbmcgU1FMIGJlaGF2aW9yIG9yIHRoZSBwZXJmb3JtYW5jZSBhbmQgc2NhbGFiaWxpdHkgb2YgdGhlIHByb3h5IGNvbm5lY3Rpb25zLlxuICAgKiBUaGUgZGVidWcgaW5mb3JtYXRpb24gaW5jbHVkZXMgdGhlIHRleHQgb2YgU1FMIHN0YXRlbWVudHMgdGhhdCB5b3Ugc3VibWl0IHRocm91Z2ggdGhlIHByb3h5LlxuICAgKiBUaHVzLCBvbmx5IGVuYWJsZSB0aGlzIHNldHRpbmcgd2hlbiBuZWVkZWQgZm9yIGRlYnVnZ2luZywgYW5kIG9ubHkgd2hlbiB5b3UgaGF2ZSBzZWN1cml0eSBtZWFzdXJlcyBpbiBwbGFjZSB0byBzYWZlZ3VhcmQgYW55IHNlbnNpdGl2ZVxuICAgKiBpbmZvcm1hdGlvbiB0aGF0IGFwcGVhcnMgaW4gdGhlIGxvZ3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBkZWJ1Z0xvZ2dpbmc/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIHJlcXVpcmUgb3IgZGlzYWxsb3cgQVdTIElkZW50aXR5IGFuZCBBY2Nlc3MgTWFuYWdlbWVudCAoSUFNKSBhdXRoZW50aWNhdGlvbiBmb3IgY29ubmVjdGlvbnMgdG8gdGhlIHByb3h5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaWFtQXV0aD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygc2Vjb25kcyB0aGF0IGEgY29ubmVjdGlvbiB0byB0aGUgcHJveHkgY2FuIGJlIGluYWN0aXZlIGJlZm9yZSB0aGUgcHJveHkgZGlzY29ubmVjdHMgaXQuXG4gICAqIFlvdSBjYW4gc2V0IHRoaXMgdmFsdWUgaGlnaGVyIG9yIGxvd2VyIHRoYW4gdGhlIGNvbm5lY3Rpb24gdGltZW91dCBsaW1pdCBmb3IgdGhlIGFzc29jaWF0ZWQgZGF0YWJhc2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IGNkay5EdXJhdGlvbi5taW51dGVzKDMwKVxuICAgKi9cbiAgcmVhZG9ubHkgaWRsZUNsaWVudFRpbWVvdXQ/OiBjZGsuRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIEEgQm9vbGVhbiBwYXJhbWV0ZXIgdGhhdCBzcGVjaWZpZXMgd2hldGhlciBUcmFuc3BvcnQgTGF5ZXIgU2VjdXJpdHkgKFRMUykgZW5jcnlwdGlvbiBpcyByZXF1aXJlZCBmb3IgY29ubmVjdGlvbnMgdG8gdGhlIHByb3h5LlxuICAgKiBCeSBlbmFibGluZyB0aGlzIHNldHRpbmcsIHlvdSBjYW4gZW5mb3JjZSBlbmNyeXB0ZWQgVExTIGNvbm5lY3Rpb25zIHRvIHRoZSBwcm94eS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVxdWlyZVRMUz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElBTSByb2xlIHRoYXQgdGhlIHByb3h5IHVzZXMgdG8gYWNjZXNzIHNlY3JldHMgaW4gQVdTIFNlY3JldHMgTWFuYWdlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHJvbGUgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNyZXQgdGhhdCB0aGUgcHJveHkgdXNlcyB0byBhdXRoZW50aWNhdGUgdG8gdGhlIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3Rlci5cbiAgICogVGhlc2Ugc2VjcmV0cyBhcmUgc3RvcmVkIHdpdGhpbiBBbWF6b24gU2VjcmV0cyBNYW5hZ2VyLlxuICAgKiBPbmUgb3IgbW9yZSBzZWNyZXRzIGFyZSByZXF1aXJlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldHM6IHNlY3JldHNtYW5hZ2VyLklTZWNyZXRbXTtcblxuICAvKipcbiAgICogT25lIG9yIG1vcmUgVlBDIHNlY3VyaXR5IGdyb3VwcyB0byBhc3NvY2lhdGUgd2l0aCB0aGUgbmV3IHByb3h5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNlY3VyaXR5IGdyb3Vwc1xuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldHMgdXNlZCBieSB0aGUgcHJveHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIFZQQyBkZWZhdWx0IHN0cmF0ZWd5IGlmIG5vdCBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIFZQQyB0byBhc3NvY2lhdGUgd2l0aCB0aGUgbmV3IHByb3h5LlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBEYXRhYmFzZVByb3h5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VQcm94eVByb3BzIGV4dGVuZHMgRGF0YWJhc2VQcm94eU9wdGlvbnMge1xuICAvKipcbiAgICogREIgcHJveHkgdGFyZ2V0OiBJbnN0YW5jZSBvciBDbHVzdGVyXG4gICAqL1xuICByZWFkb25seSBwcm94eVRhcmdldDogUHJveHlUYXJnZXRcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRoYXQgZGVzY3JpYmUgYW4gZXhpc3RpbmcgREIgUHJveHlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZVByb3h5QXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBEQiBQcm94eSBOYW1lXG4gICAqL1xuICByZWFkb25seSBkYlByb3h5TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEQiBQcm94eSBBUk5cbiAgICovXG4gIHJlYWRvbmx5IGRiUHJveHlBcm46IHN0cmluZztcblxuICAvKipcbiAgICogRW5kcG9pbnRcbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWN1cml0eSBncm91cHMgb2YgdGhlIGluc3RhbmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM6IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xufVxuXG4vKipcbiAqIERCIFByb3h5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSURhdGFiYXNlUHJveHkgZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIERCIFByb3h5IE5hbWVcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGJQcm94eU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogREIgUHJveHkgQVJOXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRiUHJveHlBcm46IHN0cmluZztcblxuICAvKipcbiAgICogRW5kcG9pbnRcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogR3JhbnQgdGhlIGdpdmVuIGlkZW50aXR5IGNvbm5lY3Rpb24gYWNjZXNzIHRvIHRoZSBwcm94eS5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgdGhlIFByaW5jaXBhbCB0byBncmFudCB0aGUgcGVybWlzc2lvbnMgdG9cbiAgICogQHBhcmFtIGRiVXNlciB0aGUgbmFtZSBvZiB0aGUgZGF0YWJhc2UgdXNlciB0byBhbGxvdyBjb25uZWN0aW5nIGFzIHRvIHRoZSBwcm94eVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGlmIHRoZSBQcm94eSBoYWQgYmVlbiBwcm92aWRlZCBhIHNpbmdsZSBTZWNyZXQgdmFsdWUsXG4gICAqICAgdGhlIHVzZXIgd2lsbCBiZSB0YWtlbiBmcm9tIHRoYXQgU2VjcmV0XG4gICAqL1xuICBncmFudENvbm5lY3QoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIGRiVXNlcj86IHN0cmluZyk6IGlhbS5HcmFudDtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIFJEUyBEYXRhYmFzZSBQcm94eS5cbiAqXG4gKi9cbmFic3RyYWN0IGNsYXNzIERhdGFiYXNlUHJveHlCYXNlIGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSURhdGFiYXNlUHJveHkge1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZGJQcm94eU5hbWU6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRiUHJveHlBcm46IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGVuZHBvaW50OiBzdHJpbmc7XG5cbiAgcHVibGljIGdyYW50Q29ubmVjdChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgZGJVc2VyPzogc3RyaW5nKTogaWFtLkdyYW50IHtcbiAgICBpZiAoIWRiVXNlcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGb3IgaW1wb3J0ZWQgRGF0YWJhc2UgUHJveGllcywgdGhlIGRiVXNlciBpcyByZXF1aXJlZCBpbiBncmFudENvbm5lY3QoKScpO1xuICAgIH1cbiAgICBjb25zdCBzY29wZVN0YWNrID0gY2RrLlN0YWNrLm9mKHRoaXMpO1xuICAgIGNvbnN0IHByb3h5R2VuZXJhdGVkSWQgPSBzY29wZVN0YWNrLnNwbGl0QXJuKHRoaXMuZGJQcm94eUFybiwgY2RrLkFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FKS5yZXNvdXJjZU5hbWU7XG4gICAgY29uc3QgdXNlckFybiA9IHNjb3BlU3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdyZHMtZGInLFxuICAgICAgcmVzb3VyY2U6ICdkYnVzZXInLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHtwcm94eUdlbmVyYXRlZElkfS8ke2RiVXNlcn1gLFxuICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUsXG4gICAgfSk7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogWydyZHMtZGI6Y29ubmVjdCddLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbdXNlckFybl0sXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBSRFMgRGF0YWJhc2UgUHJveHlcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpSRFM6OkRCUHJveHlcbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFiYXNlUHJveHkgZXh0ZW5kcyBEYXRhYmFzZVByb3h5QmFzZVxuICBpbXBsZW1lbnRzIGVjMi5JQ29ubmVjdGFibGUsIHNlY3JldHNtYW5hZ2VyLklTZWNyZXRBdHRhY2htZW50VGFyZ2V0IHtcbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBkYXRhYmFzZSBwcm94eS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURhdGFiYXNlUHJveHlBdHRyaWJ1dGVzKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBhdHRyczogRGF0YWJhc2VQcm94eUF0dHJpYnV0ZXMsXG4gICk6IElEYXRhYmFzZVByb3h5IHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBEYXRhYmFzZVByb3h5QmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZGJQcm94eU5hbWUgPSBhdHRycy5kYlByb3h5TmFtZTtcbiAgICAgIHB1YmxpYyByZWFkb25seSBkYlByb3h5QXJuID0gYXR0cnMuZGJQcm94eUFybjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBlbmRwb2ludCA9IGF0dHJzLmVuZHBvaW50O1xuICAgIH1cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIERCIFByb3h5IE5hbWVcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRiUHJveHlOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERCIFByb3h5IEFSTlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGJQcm94eUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBFbmRwb2ludFxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogQWNjZXNzIHRvIG5ldHdvcmsgY29ubmVjdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcblxuICBwcml2YXRlIHJlYWRvbmx5IHNlY3JldHM6IHNlY3JldHNtYW5hZ2VyLklTZWNyZXRbXTtcbiAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogQ2ZuREJQcm94eTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VQcm94eVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7IHBoeXNpY2FsTmFtZTogcHJvcHMuZGJQcm94eU5hbWUgfHwgaWQgfSk7XG5cbiAgICBjb25zdCByb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ0lBTVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgncmRzLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIGZvciAoY29uc3Qgc2VjcmV0IG9mIHByb3BzLnNlY3JldHMpIHtcbiAgICAgIHNlY3JldC5ncmFudFJlYWQocm9sZSk7XG4gICAgfVxuXG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cHMgPSBwcm9wcy5zZWN1cml0eUdyb3VwcyA/PyBbXG4gICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1Byb3h5U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgZGVzY3JpcHRpb246ICdTZWN1cml0eUdyb3VwIGZvciBEYXRhYmFzZSBQcm94eScsXG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgfSksXG4gICAgXTtcbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzIH0pO1xuXG4gICAgY29uc3QgYmluZFJlc3VsdCA9IHByb3BzLnByb3h5VGFyZ2V0LmJpbmQodGhpcyk7XG5cbiAgICBpZiAocHJvcHMuc2VjcmV0cy5sZW5ndGggPCAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09uZSBvciBtb3JlIHNlY3JldHMgYXJlIHJlcXVpcmVkLicpO1xuICAgIH1cbiAgICB0aGlzLnNlY3JldHMgPSBwcm9wcy5zZWNyZXRzO1xuXG4gICAgdGhpcy5yZXNvdXJjZSA9IG5ldyBDZm5EQlByb3h5KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGF1dGg6IHByb3BzLnNlY3JldHMubWFwKF8gPT4ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGF1dGhTY2hlbWU6ICdTRUNSRVRTJyxcbiAgICAgICAgICBpYW1BdXRoOiBwcm9wcy5pYW1BdXRoID8gJ1JFUVVJUkVEJyA6ICdESVNBQkxFRCcsXG4gICAgICAgICAgc2VjcmV0QXJuOiBfLnNlY3JldEFybixcbiAgICAgICAgfTtcbiAgICAgIH0pLFxuICAgICAgZGJQcm94eU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgZGVidWdMb2dnaW5nOiBwcm9wcy5kZWJ1Z0xvZ2dpbmcsXG4gICAgICBlbmdpbmVGYW1pbHk6IGJpbmRSZXN1bHQuZW5naW5lRmFtaWx5LFxuICAgICAgaWRsZUNsaWVudFRpbWVvdXQ6IHByb3BzLmlkbGVDbGllbnRUaW1lb3V0Py50b1NlY29uZHMoKSxcbiAgICAgIHJlcXVpcmVUbHM6IHByb3BzLnJlcXVpcmVUTFMgPz8gdHJ1ZSxcbiAgICAgIHJvbGVBcm46IHJvbGUucm9sZUFybixcbiAgICAgIHZwY1NlY3VyaXR5R3JvdXBJZHM6IGNkay5MYXp5Lmxpc3QoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzLm1hcChfID0+IF8uc2VjdXJpdHlHcm91cElkKSB9KSxcbiAgICAgIHZwY1N1Ym5ldElkczogcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMudnBjU3VibmV0cykuc3VibmV0SWRzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kYlByb3h5TmFtZSA9IHRoaXMucmVzb3VyY2UucmVmO1xuICAgIHRoaXMuZGJQcm94eUFybiA9IHRoaXMucmVzb3VyY2UuYXR0ckRiUHJveHlBcm47XG4gICAgdGhpcy5lbmRwb2ludCA9IHRoaXMucmVzb3VyY2UuYXR0ckVuZHBvaW50O1xuXG4gICAgbGV0IGRiSW5zdGFuY2VJZGVudGlmaWVyczogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgaWYgKGJpbmRSZXN1bHQuZGJJbnN0YW5jZXMpIHtcbiAgICAgIC8vIHN1cHBvcnQgZm9yIG9ubHkgc2luZ2xlIGluc3RhbmNlXG4gICAgICBkYkluc3RhbmNlSWRlbnRpZmllcnMgPSBbYmluZFJlc3VsdC5kYkluc3RhbmNlc1swXS5pbnN0YW5jZUlkZW50aWZpZXJdO1xuICAgIH1cblxuICAgIGxldCBkYkNsdXN0ZXJJZGVudGlmaWVyczogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgaWYgKGJpbmRSZXN1bHQuZGJDbHVzdGVycykge1xuICAgICAgZGJDbHVzdGVySWRlbnRpZmllcnMgPSBiaW5kUmVzdWx0LmRiQ2x1c3RlcnMubWFwKChjKSA9PiBjLmNsdXN0ZXJJZGVudGlmaWVyKTtcbiAgICB9XG5cbiAgICBpZiAoISFkYkluc3RhbmNlSWRlbnRpZmllcnMgJiYgISFkYkNsdXN0ZXJJZGVudGlmaWVycykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc3BlY2lmeSBib3RoIGRiSW5zdGFuY2VJZGVudGlmaWVycyBhbmQgZGJDbHVzdGVySWRlbnRpZmllcnMnKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm94eVRhcmdldEdyb3VwID0gbmV3IENmbkRCUHJveHlUYXJnZXRHcm91cCh0aGlzLCAnUHJveHlUYXJnZXRHcm91cCcsIHtcbiAgICAgIHRhcmdldEdyb3VwTmFtZTogJ2RlZmF1bHQnLFxuICAgICAgZGJQcm94eU5hbWU6IHRoaXMuZGJQcm94eU5hbWUsXG4gICAgICBkYkluc3RhbmNlSWRlbnRpZmllcnMsXG4gICAgICBkYkNsdXN0ZXJJZGVudGlmaWVycyxcbiAgICAgIGNvbm5lY3Rpb25Qb29sQ29uZmlndXJhdGlvbkluZm86IHRvQ29ubmVjdGlvblBvb2xDb25maWd1cmF0aW9uSW5mbyhwcm9wcyksXG4gICAgfSk7XG5cbiAgICBiaW5kUmVzdWx0LmRiQ2x1c3RlcnM/LmZvckVhY2goKGMpID0+IHByb3h5VGFyZ2V0R3JvdXAubm9kZS5hZGREZXBlbmRlbmN5KGMpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXJzIHRoZSBzZWNyZXQgYXR0YWNobWVudCB0YXJnZXQgc3BlY2lmaWNhdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgYXNTZWNyZXRBdHRhY2htZW50VGFyZ2V0KCk6IHNlY3JldHNtYW5hZ2VyLlNlY3JldEF0dGFjaG1lbnRUYXJnZXRQcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHRhcmdldElkOiB0aGlzLmRiUHJveHlOYW1lLFxuICAgICAgdGFyZ2V0VHlwZTogc2VjcmV0c21hbmFnZXIuQXR0YWNobWVudFRhcmdldFR5cGUuUkRTX0RCX1BST1hZLFxuICAgIH07XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRDb25uZWN0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlLCBkYlVzZXI/OiBzdHJpbmcpOiBpYW0uR3JhbnQge1xuICAgIGlmICghZGJVc2VyKSB7XG4gICAgICBpZiAodGhpcy5zZWNyZXRzLmxlbmd0aCA+IDEpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdXaGVuIHRoZSBQcm94eSBjb250YWlucyBtdWx0aXBsZSBTZWNyZXRzLCB5b3UgbXVzdCBwYXNzIGEgZGJVc2VyIGV4cGxpY2l0bHkgdG8gZ3JhbnRDb25uZWN0KCknKTtcbiAgICAgIH1cbiAgICAgIC8vICd1c2VybmFtZScgaXMgdGhlIGZpZWxkIFJEUyB1c2VzIGhlcmUsXG4gICAgICAvLyBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQXVyb3JhVXNlckd1aWRlL3Jkcy1wcm94eS5odG1sI3Jkcy1wcm94eS1zZWNyZXRzLWFybnNcbiAgICAgIGRiVXNlciA9IHRoaXMuc2VjcmV0c1swXS5zZWNyZXRWYWx1ZUZyb21Kc29uKCd1c2VybmFtZScpLnRvU3RyaW5nKCk7XG4gICAgfVxuICAgIHJldHVybiBzdXBlci5ncmFudENvbm5lY3QoZ3JhbnRlZSwgZGJVc2VyKTtcbiAgfVxufVxuXG4vKipcbiAqIENvbm5lY3Rpb25Qb29sQ29uZmlndXJhdGlvbiAoTDIgPT4gTDEpXG4gKi9cbmZ1bmN0aW9uIHRvQ29ubmVjdGlvblBvb2xDb25maWd1cmF0aW9uSW5mbyhcbiAgcHJvcHM6IERhdGFiYXNlUHJveHlQcm9wcyxcbik6IENmbkRCUHJveHlUYXJnZXRHcm91cC5Db25uZWN0aW9uUG9vbENvbmZpZ3VyYXRpb25JbmZvRm9ybWF0UHJvcGVydHkge1xuICByZXR1cm4ge1xuICAgIGNvbm5lY3Rpb25Cb3Jyb3dUaW1lb3V0OiBwcm9wcy5ib3Jyb3dUaW1lb3V0Py50b1NlY29uZHMoKSxcbiAgICBpbml0UXVlcnk6IHByb3BzLmluaXRRdWVyeSxcbiAgICBtYXhDb25uZWN0aW9uc1BlcmNlbnQ6IHByb3BzLm1heENvbm5lY3Rpb25zUGVyY2VudCxcbiAgICBtYXhJZGxlQ29ubmVjdGlvbnNQZXJjZW50OiBwcm9wcy5tYXhJZGxlQ29ubmVjdGlvbnNQZXJjZW50LFxuICAgIHNlc3Npb25QaW5uaW5nRmlsdGVyczogcHJvcHMuc2Vzc2lvblBpbm5pbmdGaWx0ZXJzPy5tYXAoXyA9PiBfLmZpbHRlck5hbWUpLFxuICB9O1xufVxuIl19