"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const secretsmanager = require("../../aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _e;
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _e, _f;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            // major version 8.0 uses a different name for the S3 import parameter
            const s3ImportParam = ((_f = this.engineVersion) === null || _f === void 0 ? void 0 : _f.majorVersion) === '8.0'
                ? 'aws_default_s3_role'
                : 'aurora_load_from_s3_role';
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter(s3ImportParam, options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
_a = JSII_RTTI_SYMBOL_1;
AuroraEngineVersion[_a] = { fqn: "monocdk.aws_rds.AuroraEngineVersion", version: "1.149.0" };
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
    static builtIn_8_0(minorVersion) {
        return new AuroraMysqlEngineVersion(`8.0.mysql_aurora.${minorVersion}`, '8.0');
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
_b = JSII_RTTI_SYMBOL_1;
AuroraMysqlEngineVersion[_b] = { fqn: "monocdk.aws_rds.AuroraMysqlEngineVersion", version: "1.149.0" };
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/** Version "5.7.mysql_aurora.2.08.2". */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/** Version "5.7.mysql_aurora.2.09.0". */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
/** Version "5.7.mysql_aurora.2.09.1". */
AuroraMysqlEngineVersion.VER_2_09_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.1');
/** Version "5.7.mysql_aurora.2.09.2". */
AuroraMysqlEngineVersion.VER_2_09_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.2');
/** Version "5.7.mysql_aurora.2.09.3". */
AuroraMysqlEngineVersion.VER_2_09_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.3');
/** Version "5.7.mysql_aurora.2.10.0". */
AuroraMysqlEngineVersion.VER_2_10_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.0');
/** Version "5.7.mysql_aurora.2.10.1". */
AuroraMysqlEngineVersion.VER_2_10_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.1');
/** Version "5.7.mysql_aurora.2.10.2". */
AuroraMysqlEngineVersion.VER_2_10_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.2');
/** Version "8.0.mysql_aurora.3.01.0". */
AuroraMysqlEngineVersion.VER_3_01_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.01.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        jsiiDeprecationWarnings.monocdk_aws_rds_AuroraPostgresEngineFeatures(auroraPostgresFeatures);
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
_c = JSII_RTTI_SYMBOL_1;
AuroraPostgresEngineVersion[_c] = { fqn: "monocdk.aws_rds.AuroraPostgresEngineVersion", version: "1.149.0" };
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
AuroraPostgresEngineVersion.VER_9_6_18 = AuroraPostgresEngineVersion.of('9.6.18', '9.6');
/** Version "9.6.19". */
AuroraPostgresEngineVersion.VER_9_6_19 = AuroraPostgresEngineVersion.of('9.6.19', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/** Version "10.13". */
AuroraPostgresEngineVersion.VER_10_13 = AuroraPostgresEngineVersion.of('10.13', '10', { s3Import: true, s3Export: true });
/** Version "10.14". */
AuroraPostgresEngineVersion.VER_10_14 = AuroraPostgresEngineVersion.of('10.14', '10', { s3Import: true, s3Export: true });
/** Version "10.16". */
AuroraPostgresEngineVersion.VER_10_16 = AuroraPostgresEngineVersion.of('10.16', '10', { s3Import: true, s3Export: true });
/** Version "10.18". */
AuroraPostgresEngineVersion.VER_10_18 = AuroraPostgresEngineVersion.of('10.18', '10', { s3Import: true, s3Export: true });
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/** Version "11.8". */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
/** Version "11.9". */
AuroraPostgresEngineVersion.VER_11_9 = AuroraPostgresEngineVersion.of('11.9', '11', { s3Import: true, s3Export: true });
/** Version "11.11". */
AuroraPostgresEngineVersion.VER_11_11 = AuroraPostgresEngineVersion.of('11.11', '11', { s3Import: true, s3Export: true });
/** Version "11.13". */
AuroraPostgresEngineVersion.VER_11_13 = AuroraPostgresEngineVersion.of('11.13', '11', { s3Import: true, s3Export: true });
/** Version "12.4". */
AuroraPostgresEngineVersion.VER_12_4 = AuroraPostgresEngineVersion.of('12.4', '12', { s3Import: true, s3Export: true });
/** Version "12.6". */
AuroraPostgresEngineVersion.VER_12_6 = AuroraPostgresEngineVersion.of('12.6', '12', { s3Import: true, s3Export: true });
/** Version "12.8". */
AuroraPostgresEngineVersion.VER_12_8 = AuroraPostgresEngineVersion.of('12.8', '12', { s3Import: true, s3Export: true });
/** Version "13.3". */
AuroraPostgresEngineVersion.VER_13_3 = AuroraPostgresEngineVersion.of('13.3', '13', { s3Import: true, s3Export: true });
/** Version "13.4". */
AuroraPostgresEngineVersion.VER_13_4 = AuroraPostgresEngineVersion.of('13.4', '13', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _e, _f, _g;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_e = this.engineVersion) === null || _e === void 0 ? void 0 : _e.fullVersion) {
            if (options.s3ImportRole && !((_f = config.features) === null || _f === void 0 ? void 0 : _f.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_g = config.features) === null || _g === void 0 ? void 0 : _g.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        jsiiDeprecationWarnings.monocdk_aws_rds_AuroraClusterEngineProps(props);
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        jsiiDeprecationWarnings.monocdk_aws_rds_AuroraMysqlClusterEngineProps(props);
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        jsiiDeprecationWarnings.monocdk_aws_rds_AuroraPostgresClusterEngineProps(props);
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
_d = JSII_RTTI_SYMBOL_1;
DatabaseClusterEngine[_d] = { fqn: "monocdk.aws_rds.DatabaseClusterEngine", version: "1.149.0" };
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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