"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const secretsmanager = require("../../aws-secretsmanager");
const core = require("../../core");
const cr = require("../../custom-resources");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const msk_generated_1 = require("./msk.generated");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * SASL authentication
     */
    static sasl(props) {
        jsiiDeprecationWarnings.monocdk_aws_msk_SaslAuthProps(props);
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        jsiiDeprecationWarnings.monocdk_aws_msk_TlsAuthProps(props);
        return new ClientAuthentication(undefined, props);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "monocdk.aws_msk.ClientAuthentication", version: "1.149.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        jsiiDeprecationWarnings.monocdk_aws_msk_ClusterProps(props);
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (((_e = (_d = props.clientAuthentication) === null || _d === void 0 ? void 0 : _d.saslProps) === null || _e === void 0 ? void 0 : _e.iam) && ((_g = (_f = props.clientAuthentication) === null || _f === void 0 ? void 0 : _f.saslProps) === null || _g === void 0 ? void 0 : _g.scram)) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (((_h = props.encryptionInTransit) === null || _h === void 0 ? void 0 : _h.clientBroker) ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (((_j = props.encryptionInTransit) === null || _j === void 0 ? void 0 : _j.clientBroker) ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (((_l = (_k = props.clientAuthentication) === null || _k === void 0 ? void 0 : _k.saslProps) === null || _l === void 0 ? void 0 : _l.scram) || ((_o = (_m = props.clientAuthentication) === null || _m === void 0 ? void 0 : _m.saslProps) === null || _o === void 0 ? void 0 : _o.iam))) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = (_q = (_p = props.ebsStorageInfo) === null || _p === void 0 ? void 0 : _p.volumeSize) !== null && _q !== void 0 ? _q : 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = ((_r = props.ebsStorageInfo) === null || _r === void 0 ? void 0 : _r.encryptionKey) ? {
            dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
        }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: (_t = (_s = props.encryptionInTransit) === null || _s === void 0 ? void 0 : _s.clientBroker) !== null && _t !== void 0 ? _t : ClientBrokerEncryption.TLS,
            inCluster: (_v = (_u = props.encryptionInTransit) === null || _u === void 0 ? void 0 : _u.enableInCluster) !== null && _v !== void 0 ? _v : true,
        };
        const openMonitoring = ((_w = props.monitoring) === null || _w === void 0 ? void 0 : _w.enablePrometheusJmxExporter) || ((_x = props.monitoring) === null || _x === void 0 ? void 0 : _x.enablePrometheusNodeExporter)
            ? {
                prometheus: {
                    jmxExporter: ((_y = props.monitoring) === null || _y === void 0 ? void 0 : _y.enablePrometheusJmxExporter) ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: ((_z = props.monitoring) === null || _z === void 0 ? void 0 : _z.enablePrometheusNodeExporter) ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: ((_0 = props.logging) === null || _0 === void 0 ? void 0 : _0.cloudwatchLogGroup) !== undefined,
                    logGroup: (_2 = (_1 = props.logging) === null || _1 === void 0 ? void 0 : _1.cloudwatchLogGroup) === null || _2 === void 0 ? void 0 : _2.logGroupName,
                },
                firehose: {
                    enabled: ((_3 = props.logging) === null || _3 === void 0 ? void 0 : _3.firehoseDeliveryStreamName) !==
                        undefined,
                    deliveryStream: (_4 = props.logging) === null || _4 === void 0 ? void 0 : _4.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: ((_6 = (_5 = props.logging) === null || _5 === void 0 ? void 0 : _5.s3) === null || _6 === void 0 ? void 0 : _6.bucket) !== undefined,
                    bucket: (_8 = (_7 = props.logging) === null || _7 === void 0 ? void 0 : _7.s3) === null || _8 === void 0 ? void 0 : _8.bucket.bucketName,
                    prefix: (_10 = (_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.s3) === null || _10 === void 0 ? void 0 : _10.prefix,
                },
            },
        };
        if (((_12 = (_11 = props.clientAuthentication) === null || _11 === void 0 ? void 0 : _11.saslProps) === null || _12 === void 0 ? void 0 : _12.scram) &&
            ((_14 = (_13 = props.clientAuthentication) === null || _13 === void 0 ? void 0 : _13.saslProps) === null || _14 === void 0 ? void 0 : _14.key) === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if ((_16 = (_15 = props.clientAuthentication) === null || _15 === void 0 ? void 0 : _15.saslProps) === null || _16 === void 0 ? void 0 : _16.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
        }
        else if ((_18 = (_17 = props.clientAuthentication) === null || _17 === void 0 ? void 0 : _17.saslProps) === null || _18 === void 0 ? void 0 : _18.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (((_20 = (_19 = props.clientAuthentication) === null || _19 === void 0 ? void 0 : _19.tlsProps) === null || _20 === void 0 ? void 0 : _20.certificateAuthorities) !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: (_22 = (_21 = props.clientAuthentication) === null || _21 === void 0 ? void 0 : _21.tlsProps) === null || _22 === void 0 ? void 0 : _22.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new msk_generated_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: (_23 = props.monitoring) === null || _23 === void 0 ? void 0 : _23.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        var _c;
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [(_c = this.saslScramAuthenticationKey) === null || _c === void 0 ? void 0 : _c.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "monocdk.aws_msk.Cluster", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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