# AWS Lambda Construct Library


This construct library allows you to define AWS Lambda Functions.

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
});
```

## Handler Code

The `lambda.Code` class includes static convenience methods for various types of
runtime code.

 * `lambda.Code.fromBucket(bucket, key[, objectVersion])` - specify an S3 object
   that contains the archive of your runtime code.
 * `lambda.Code.fromInline(code)` - inline the handle code as a string. This is
   limited to supported runtimes and the code cannot exceed 4KiB.
 * `lambda.Code.fromAsset(path)` - specify a directory or a .zip file in the local
   filesystem which will be zipped and uploaded to S3 before deployment. See also
   [bundling asset code](#bundling-asset-code).
 * `lambda.Code.fromDockerBuild(path, options)` - use the result of a Docker
   build as code. The runtime code is expected to be located at `/asset` in the
   image and will be zipped and uploaded to S3 as an asset.

The following example shows how to define a Python function and deploy the code
from the local directory `my-lambda-handler` to it:

[Example of Lambda Code from Local Assets](test/integ.assets.lit.ts)

When deploying a stack that contains this code, the directory will be zip
archived and then uploaded to an S3 bucket, then the exact location of the S3
objects will be passed when the stack is deployed.

During synthesis, the CDK expects to find a directory on disk at the asset
directory specified. Note that we are referencing the asset directory relatively
to our CDK project directory. This is especially important when we want to share
this construct through a library. Different programming languages will have
different techniques for bundling resources into libraries.

## Docker Images

Lambda functions allow specifying their handlers within docker images. The docker
image can be an image from ECR or a local asset that the CDK will package and load
into ECR.

The following `DockerImageFunction` construct uses a local folder with a
Dockerfile as the asset that will be used as the function handler.

```ts
new lambda.DockerImageFunction(this, 'AssetFunction', {
  code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, 'docker-handler')),
});
```

You can also specify an image that already exists in ECR as the function handler.

```ts
import * as ecr from 'monocdk/aws-ecr';
const repo = new ecr.Repository(this, 'Repository');

new lambda.DockerImageFunction(this, 'ECRFunction', {
  code: lambda.DockerImageCode.fromEcr(repo),
});
```

The props for these docker image resources allow overriding the image's `CMD`, `ENTRYPOINT`, and `WORKDIR`
configurations. See their docs for more information.

## Execution Role

Lambda functions assume an IAM role during execution. In CDK by default, Lambda
functions will use an autogenerated Role if one is not provided.

The autogenerated Role is automatically given permissions to execute the Lambda
function. To reference the autogenerated Role:

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
});

const role = fn.role; // the Role
```

You can also provide your own IAM role. Provided IAM roles will not automatically
be given permissions to execute the Lambda function. To provide a role and grant
it appropriate permissions:

```ts
const myRole = new iam.Role(this, 'My Role', {
  assumedBy: new iam.ServicePrincipal('sns.amazonaws.com'),
});

const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  role: myRole, // user-provided role
});

myRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"));
myRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole")); // only required if your function lives in a VPC
```

## Function Timeout

AWS Lambda functions have a default timeout of 3 seconds, but this can be increased
up to 15 minutes. The timeout is available as a property of `Function` so that
you can reference it elsewhere in your stack. For instance, you could use it to create
a CloudWatch alarm to report when your function timed out:

```ts
import * as cdk from 'monocdk';
import * as cloudwatch from 'monocdk/aws-cloudwatch';

const fn = new lambda.Function(this, 'MyFunction', {
   runtime: lambda.Runtime.NODEJS_12_X,
   handler: 'index.handler',
   code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
   timeout: cdk.Duration.minutes(5),
});

if (fn.timeout) {
   new cloudwatch.Alarm(this, `MyAlarm`, {
      metric: fn.metricDuration().with({
         statistic: 'Maximum',
      }),
      evaluationPeriods: 1,
      datapointsToAlarm: 1,
      threshold: fn.timeout.toMilliseconds(),
      treatMissingData: cloudwatch.TreatMissingData.IGNORE,
      alarmName: 'My Lambda Timeout',
   });
}
```

## Resource-based Policies

AWS Lambda supports resource-based policies for controlling access to Lambda
functions and layers on a per-resource basis. In particular, this allows you to
give permission to AWS services and other AWS accounts to modify and invoke your
functions. You can also restrict permissions given to AWS services by providing
a source account or ARN (representing the account and identifier of the resource
that accesses the function or layer).

```ts
declare const fn: lambda.Function;
const principal = new iam.ServicePrincipal('my-service');

fn.grantInvoke(principal);

// Equivalent to:
fn.addPermission('my-service Invocation', {
  principal: principal,
});
```

For more information, see [Resource-based
policies](https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html)
in the AWS Lambda Developer Guide.

Providing an unowned principal (such as account principals, generic ARN
principals, service principals, and principals in other accounts) to a call to
`fn.grantInvoke` will result in a resource-based policy being created. If the
principal in question has conditions limiting the source account or ARN of the
operation (see above), these conditions will be automatically added to the
resource policy.

```ts
declare const fn: lambda.Function;
const servicePrincipal = new iam.ServicePrincipal('my-service');
const sourceArn = 'arn:aws:s3:::my-bucket';
const sourceAccount = '111122223333';
const servicePrincipalWithConditions = servicePrincipal.withConditions({
  ArnLike: {
    'aws:SourceArn': sourceArn,
  },
  StringEquals: {
    'aws:SourceAccount': sourceAccount,
  },
});

fn.grantInvoke(servicePrincipalWithConditions);

// Equivalent to:
fn.addPermission('my-service Invocation', {
  principal: servicePrincipal,
  sourceArn: sourceArn,
  sourceAccount: sourceAccount,
});
```

## Versions

You can use
[versions](https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html)
to manage the deployment of your AWS Lambda functions. For example, you can
publish a new version of a function for beta testing without affecting users of
the stable production version.

The function version includes the following information:

* The function code and all associated dependencies.
* The Lambda runtime that executes the function.
* All of the function settings, including the environment variables.
* A unique Amazon Resource Name (ARN) to identify this version of the function.

You could create a version to your lambda function using the `Version` construct.

```ts
declare const fn: lambda.Function;
const version = new lambda.Version(this, 'MyVersion', {
  lambda: fn,
});
```

The major caveat to know here is that a function version must always point to a
specific 'version' of the function. When the function is modified, the version
will continue to point to the 'then version' of the function.

One way to ensure that the `lambda.Version` always points to the latest version
of your `lambda.Function` is to set an environment variable which changes at
least as often as your code does. This makes sure the function always has the
latest code. For instance -

```ts
const codeVersion = "stringOrMethodToGetCodeVersion";
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  environment: {
    'CodeVersionString': codeVersion,
  },
});
```

The `fn.latestVersion` property returns a `lambda.IVersion` which represents
the `$LATEST` pseudo-version.

However, most AWS services require a specific AWS Lambda version,
and won't allow you to use `$LATEST`. Therefore, you would normally want
to use `lambda.currentVersion`.

The `fn.currentVersion` property can be used to obtain a `lambda.Version`
resource that represents the AWS Lambda function defined in your application.
Any change to your function's code or configuration will result in the creation
of a new version resource. You can specify options for this version through the
`currentVersionOptions` property.

NOTE: The `currentVersion` property is only supported when your AWS Lambda function
uses either `lambda.Code.fromAsset` or `lambda.Code.fromInline`. Other types
of code providers (such as `lambda.Code.fromBucket`) require that you define a
`lambda.Version` resource directly since the CDK is unable to determine if
their contents had changed.

### `currentVersion`: Updated hashing logic

To produce a new lambda version each time the lambda function is modified, the
`currentVersion` property under the hood, computes a new logical id based on the
properties of the function. This informs CloudFormation that a new
`AWS::Lambda::Version` resource should be created pointing to the updated Lambda
function.

However, a bug was introduced in this calculation that caused the logical id to
change when it was not required (ex: when the Function's `Tags` property, or
when the `DependsOn` clause was modified). This caused the deployment to fail
since the Lambda service does not allow creating duplicate versions.

This has been fixed in the AWS CDK but *existing* users need to opt-in via a
[feature flag]. Users who have run `cdk init` since this fix will be opted in,
by default.

Existing users will need to enable the [feature flag]
`@aws-cdk/aws-lambda:recognizeVersionProps`. Since CloudFormation does not
allow duplicate versions, they will also need to make some modification to
their function so that a new version can be created. Any trivial change such as
a whitespace change in the code or a no-op environment variable will suffice.

When the new logic is in effect, you may rarely come across the following error:
`The following properties are not recognized as version properties`. This will
occur, typically when [property overrides] are used, when a new property
introduced in `AWS::Lambda::Function` is used that CDK is still unaware of.

To overcome this error, use the API `Function.classifyVersionProperty()` to
record whether a new version should be generated when this property is changed.
This can be typically determined by checking whether the property can be
modified using the *[UpdateFunctionConfiguration]* API or not.

[feature flag]: https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html
[property overrides]: https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw
[UpdateFunctionConfiguration]: https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html

## Aliases

You can define one or more
[aliases](https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html)
for your AWS Lambda function. A Lambda alias is like a pointer to a specific
Lambda function version. Users can access the function version using the alias
ARN.

The `version.addAlias()` method can be used to define an AWS Lambda alias that
points to a specific version.

The following example defines an alias named `live` which will always point to a
version that represents the function as defined in your CDK app. When you change
your lambda code or configuration, a new resource will be created. You can
specify options for the current version through the `currentVersionOptions`
property.

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  currentVersionOptions: {
    removalPolicy: RemovalPolicy.RETAIN, // retain old versions
    retryAttempts: 1,                   // async retry attempts
  },
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
});

fn.currentVersion.addAlias('live');
```

## Layers

The `lambda.LayerVersion` class can be used to define Lambda layers and manage
granting permissions to other AWS accounts or organizations.

[Example of Lambda Layer usage](test/integ.layer-version.lit.ts)

By default, updating a layer creates a new layer version, and CloudFormation will delete the old version as part of the stack update.

Alternatively, a removal policy can be used to retain the old version:

```ts
new lambda.LayerVersion(this, 'MyLayer', {
  removalPolicy: RemovalPolicy.RETAIN,
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
});
```

## Architecture

Lambda functions, by default, run on compute systems that have the 64 bit x86 architecture.

The AWS Lambda service also runs compute on the ARM architecture, which can reduce cost
for some workloads.

A lambda function can be configured to be run on one of these platforms:

```ts
new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  architecture: lambda.Architecture.ARM_64,
});
```

Similarly, lambda layer versions can also be tagged with architectures it is compatible with.

```ts
new lambda.LayerVersion(this, 'MyLayer', {
  removalPolicy: RemovalPolicy.RETAIN,
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  compatibleArchitectures: [lambda.Architecture.X86_64, lambda.Architecture.ARM_64],
});
```

## Lambda Insights

Lambda functions can be configured to use CloudWatch [Lambda Insights](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights.html)
which provides low-level runtime metrics for a Lambda functions.

```ts
new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  insightsVersion: lambda.LambdaInsightsVersion.VERSION_1_0_98_0,
});
```

If the version of insights is not yet available in the CDK, you can also provide the ARN directly as so -

```ts
const layerArn = 'arn:aws:lambda:us-east-1:580247275435:layer:LambdaInsightsExtension:14';
new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  insightsVersion: lambda.LambdaInsightsVersion.fromInsightVersionArn(layerArn),
});
```

If you are deploying an ARM_64 Lambda Function, you must specify a
Lambda Insights Version >= `1_0_119_0`.

```ts
new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  architecture: lambda.Architecture.ARM_64,
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  insightsVersion: lambda.LambdaInsightsVersion.VERSION_1_0_119_0,
});
```

## Event Rule Target

You can use an AWS Lambda function as a target for an Amazon CloudWatch event
rule:

```ts
import * as events from 'monocdk/aws-events';
import * as targets from 'monocdk/aws-events-targets';

declare const fn: lambda.Function;
const rule = new events.Rule(this, 'Schedule Rule', {
 schedule: events.Schedule.cron({ minute: '0', hour: '4' }),
});
rule.addTarget(new targets.LambdaFunction(fn));
```

## Event Sources

AWS Lambda supports a [variety of event sources](https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html).

In most cases, it is possible to trigger a function as a result of an event by
using one of the `add<Event>Notification` methods on the source construct. For
example, the `s3.Bucket` construct has an `onEvent` method which can be used to
trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.

An alternative way to add event sources to a function is to use `function.addEventSource(source)`.
This method accepts an `IEventSource` object. The module __@aws-cdk/aws-lambda-event-sources__
includes classes for the various event sources supported by AWS Lambda.

For example, the following code adds an SQS queue as an event source for a function:

```ts
import * as eventsources from 'monocdk/aws-lambda-event-sources';
import * as sqs from 'monocdk/aws-sqs';

declare const fn: lambda.Function;
const queue = new sqs.Queue(this, 'Queue');
fn.addEventSource(new eventsources.SqsEventSource(queue));
```

The following code adds an S3 bucket notification as an event source:

```ts
import * as eventsources from 'monocdk/aws-lambda-event-sources';
import * as s3 from 'monocdk/aws-s3';

declare const fn: lambda.Function;
const bucket = new s3.Bucket(this, 'Bucket');
fn.addEventSource(new eventsources.S3EventSource(bucket, {
  events: [ s3.EventType.OBJECT_CREATED, s3.EventType.OBJECT_REMOVED ],
  filters: [ { prefix: 'subdir/' } ] // optional
}));
```

See the documentation for the __@aws-cdk/aws-lambda-event-sources__ module for more details.

## Imported Lambdas

When referencing an imported lambda in the CDK, use `fromFunctionArn()` for most use cases:

```ts
const fn = lambda.Function.fromFunctionArn(
  this,
  'Function',
  'arn:aws:lambda:us-east-1:123456789012:function:MyFn',
);
```

The `fromFunctionAttributes()` API is available for more specific use cases:

```ts
const fn = lambda.Function.fromFunctionAttributes(this, 'Function', {
  functionArn: 'arn:aws:lambda:us-east-1:123456789012:function:MyFn',
  // The following are optional properties for specific use cases and should be used with caution:

  // Use Case: imported function is in the same account as the stack. This tells the CDK that it
  // can modify the function's permissions.
  sameEnvironment: true,

  // Use Case: imported function is in a different account and user commits to ensuring that the
  // imported function has the correct permissions outside the CDK.
  skipPermissions: true,
});
```

If `fromFunctionArn()` causes an error related to having to provide an account and/or region in a different construct,
and the lambda is in the same account and region as the stack you're importing it into,
you can use `Function.fromFunctionName()` instead:

```ts
const fn = lambda.Function.fromFunctionName(this, 'Function', 'MyFn');
```

## Lambda with DLQ

A dead-letter queue can be automatically created for a Lambda function by
setting the `deadLetterQueueEnabled: true` configuration. In such case CDK creates
a `sqs.Queue` as `deadLetterQueue`.

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, "hi"); }'),
  deadLetterQueueEnabled: true,
});
```

It is also possible to provide a dead-letter queue instead of getting a new queue created:

```ts
import * as sqs from 'monocdk/aws-sqs';

const dlq = new sqs.Queue(this, 'DLQ');
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, "hi"); }'),
  deadLetterQueue: dlq,
});
```

You can also use a `sns.Topic` instead of an `sqs.Queue` as dead-letter queue:

```ts
import * as sns from 'monocdk/aws-sns';

const dlt = new sns.Topic(this, 'DLQ');
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromInline('// your code here'),
  deadLetterTopic: dlt,
});
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/dlq.html)
to learn more about AWS Lambdas and DLQs.

## Lambda with X-Ray Tracing

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, "hi"); }'),
  tracing: lambda.Tracing.ACTIVE,
});
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html)
to learn more about AWS Lambda's X-Ray support.

## Lambda with Profiling

The following code configures the lambda function with CodeGuru profiling. By default, this creates a new CodeGuru
profiling group -

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.PYTHON_3_6,
  handler: 'index.handler',
  code: lambda.Code.fromAsset('lambda-handler'),
  profiling: true,
});
```

The `profilingGroup` property can be used to configure an existing CodeGuru profiler group.

CodeGuru profiling is supported for all Java runtimes and Python3.6+ runtimes.

See [the AWS documentation](https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html)
to learn more about AWS Lambda's Profiling support.

## Lambda with Reserved Concurrent Executions

```ts
const fn = new lambda.Function(this, 'MyFunction', {
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, "hi"); }'),
  reservedConcurrentExecutions: 100,
});
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html)
managing concurrency.

## AutoScaling

You can use Application AutoScaling to automatically configure the provisioned concurrency for your functions. AutoScaling can be set to track utilization or be based on a schedule. To configure AutoScaling on a function alias:

```ts
import * as autoscaling from 'monocdk/aws-autoscaling';

declare const fn: lambda.Function;
const alias = new lambda.Alias(this, 'Alias', {
  aliasName: 'prod',
  version: fn.latestVersion,
});

// Create AutoScaling target
const as = alias.addAutoScaling({ maxCapacity: 50 });

// Configure Target Tracking
as.scaleOnUtilization({
  utilizationTarget: 0.5,
});

// Configure Scheduled Scaling
as.scaleOnSchedule('ScaleUpInTheMorning', {
  schedule: autoscaling.Schedule.cron({ hour: '8', minute: '0'}),
  minCapacity: 20,
});
```

[Example of Lambda AutoScaling usage](test/integ.autoscaling.lit.ts)

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/invocation-scaling.html) on autoscaling lambda functions.

## Log Group

Lambda functions automatically create a log group with the name `/aws/lambda/<function-name>` upon first execution with
log data set to never expire.

The `logRetention` property can be used to set a different expiration period.

It is possible to obtain the function's log group as a `logs.ILogGroup` by calling the `logGroup` property of the
`Function` construct.

By default, CDK uses the AWS SDK retry options when creating a log group. The `logRetentionRetryOptions` property
allows you to customize the maximum number of retries and base backoff duration.

*Note* that, if either `logRetention` is set or `logGroup` property is called, a [CloudFormation custom
resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html) is added
to the stack that pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the
correct log retention period (never expire, by default).

*Further note* that, if the log group already exists and the `logRetention` is not set, the custom resource will reset
the log retention to never expire even if it was configured with a different value.

## FileSystem Access

You can configure a function to mount an Amazon Elastic File System (Amazon EFS) to a
directory in your runtime environment with the `filesystem` property. To access Amazon EFS
from lambda function, the Amazon EFS access point will be required.

The following sample allows the lambda function to mount the Amazon EFS access point to `/mnt/msg` in the runtime environment and access the filesystem with the POSIX identity defined in `posixUser`.

```ts
import * as ec2 from 'monocdk/aws-ec2';
import * as efs from 'monocdk/aws-efs';

// create a new VPC
const vpc = new ec2.Vpc(this, 'VPC');

// create a new Amazon EFS filesystem
const fileSystem = new efs.FileSystem(this, 'Efs', { vpc });

// create a new access point from the filesystem
const accessPoint = fileSystem.addAccessPoint('AccessPoint', {
  // set /export/lambda as the root of the access point
  path: '/export/lambda',
  // as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
  createAcl: {
    ownerUid: '1001',
    ownerGid: '1001',
    permissions: '750',
  },
  // enforce the POSIX identity so lambda function will access with this identity
  posixUser: {
    uid: '1001',
    gid: '1001',
  },
});

const fn = new lambda.Function(this, 'MyLambda', {
  // mount the access point to /mnt/msg in the lambda runtime environment
  filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/msg'),
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
  vpc,
});
```


## Singleton Function

The `SingletonFunction` construct is a way to guarantee that a lambda function will be guaranteed to be part of the stack,
once and only once, irrespective of how many times the construct is declared to be part of the stack. This is guaranteed
as long as the `uuid` property and the optional `lambdaPurpose` property stay the same whenever they're declared into the
stack.

A typical use case of this function is when a higher level construct needs to declare a Lambda function as part of it but
needs to guarantee that the function is declared once. However, a user of this higher level construct can declare it any
number of times and with different properties. Using `SingletonFunction` here with a fixed `uuid` will guarantee this.

For example, the `LogRetention` construct requires only one single lambda function for all different log groups whose
retention it seeks to manage.

## Bundling Asset Code

When using `lambda.Code.fromAsset(path)` it is possible to bundle the code by running a
command in a Docker container. The asset path will be mounted at `/asset-input`. The
Docker container is responsible for putting content at `/asset-output`. The content at
`/asset-output` will be zipped and used as Lambda code.

Example with Python:

```ts
new lambda.Function(this, 'Function', {
  code: lambda.Code.fromAsset(path.join(__dirname, 'my-python-handler'), {
    bundling: {
      image: lambda.Runtime.PYTHON_3_9.bundlingImage,
      command: [
        'bash', '-c',
        'pip install -r requirements.txt -t /asset-output && cp -au . /asset-output'
      ],
    },
  }),
  runtime: lambda.Runtime.PYTHON_3_9,
  handler: 'index.handler',
});
```

Runtimes expose a `bundlingImage` property that points to the [AWS SAM](https://github.com/awslabs/aws-sam-cli) build image.

Use `cdk.DockerImage.fromRegistry(image)` to use an existing image or
`cdk.DockerImage.fromBuild(path)` to build a specific image:

```ts
new lambda.Function(this, 'Function', {
  code: lambda.Code.fromAsset('/path/to/handler', {
    bundling: {
      image: DockerImage.fromBuild('/path/to/dir/with/DockerFile', {
        buildArgs: {
          ARG1: 'value1',
        },
      }),
      command: ['my', 'cool', 'command'],
    },
  }),
  runtime: lambda.Runtime.PYTHON_3_9,
  handler: 'index.handler',
});
```

## Language-specific APIs

Language-specific higher level constructs are provided in separate modules:

* `@aws-cdk/aws-lambda-nodejs`: [Github](https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-nodejs) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-nodejs-readme.html)
* `@aws-cdk/aws-lambda-python`: [Github](https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-python) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-python-readme.html)

## Code Signing

Code signing for AWS Lambda helps to ensure that only trusted code runs in your Lambda functions.
When enabled, AWS Lambda checks every code deployment and verifies that the code package is signed by a trusted source.
For more information, see [Configuring code signing for AWS Lambda](https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html).
The following code configures a function with code signing.

```ts
import * as signer from 'monocdk/aws-signer';

const signingProfile = new signer.SigningProfile(this, 'SigningProfile', {
  platform: signer.Platform.AWS_LAMBDA_SHA384_ECDSA,
});

const codeSigningConfig = new lambda.CodeSigningConfig(this, 'CodeSigningConfig', {
  signingProfiles: [signingProfile],
});

new lambda.Function(this, 'Function', {
  codeSigningConfig,
  runtime: lambda.Runtime.NODEJS_12_X,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
});
```
