"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile = exports.DependenciesFile || (exports.DependenciesFile = {}));
class Packaging {
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
    static fromEntry(entry) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return Packaging.PIPENV;
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return Packaging.POETRY;
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return Packaging.PIP;
        }
        else {
            return Packaging.NONE;
        }
    }
}
exports.Packaging = Packaging;
/**
 * Standard packaging with `pip`.
 */
Packaging.PIP = new Packaging({
    dependenciesFile: DependenciesFile.PIP,
});
/**
 * Packaging with `pipenv`.
 */
Packaging.PIPENV = new Packaging({
    dependenciesFile: DependenciesFile.PIPENV,
    // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
    // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
    exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv lock -r > ${DependenciesFile.PIP} && rm -rf .venv`,
});
/**
 * Packaging with `poetry`.
 */
Packaging.POETRY = new Packaging({
    dependenciesFile: DependenciesFile.POETRY,
    // Export dependencies with credentials avaiable in the bundling image.
    exportCommand: `poetry export --with-credentials --format ${DependenciesFile.PIP} --output ${DependenciesFile.PIP}`,
});
/**
 * No dependencies or packaging.
 */
Packaging.NONE = new Packaging({ dependenciesFile: DependenciesFile.NONE });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGFja2FnaW5nLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicGFja2FnaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFFN0IsSUFBWSxnQkFLWDtBQUxELFdBQVksZ0JBQWdCO0lBQzFCLDRDQUF3QixDQUFBO0lBQ3hCLDBDQUFzQixDQUFBO0lBQ3RCLDJDQUF1QixDQUFBO0lBQ3ZCLDZCQUFTLENBQUE7QUFDWCxDQUFDLEVBTFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFLM0I7QUFlRCxNQUFhLFNBQVM7SUErQ3BCLFlBQVksS0FBcUI7UUFDL0IsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUMvQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7S0FDMUM7SUFqQk0sTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFhO1FBQ25DLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFO1lBQzVELE9BQU8sU0FBUyxDQUFDLE1BQU0sQ0FBQztTQUN6QjtRQUFDLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFO1lBQzlELE9BQU8sU0FBUyxDQUFDLE1BQU0sQ0FBQztTQUN6QjthQUFNLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO1lBQ2hFLE9BQU8sU0FBUyxDQUFDLEdBQUcsQ0FBQztTQUN0QjthQUFNO1lBQ0wsT0FBTyxTQUFTLENBQUMsSUFBSSxDQUFDO1NBQ3ZCO0tBQ0Y7O0FBM0NILDhCQW1EQztBQWpEQzs7R0FFRztBQUNvQixhQUFHLEdBQUcsSUFBSSxTQUFTLENBQUM7SUFDekMsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsR0FBRztDQUN2QyxDQUFDLENBQUM7QUFFSDs7R0FFRztBQUNvQixnQkFBTSxHQUFHLElBQUksU0FBUyxDQUFDO0lBQzVDLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLE1BQU07SUFDekMsK0dBQStHO0lBQy9HLGlHQUFpRztJQUNqRyxhQUFhLEVBQUUsNkNBQTZDLGdCQUFnQixDQUFDLEdBQUcsa0JBQWtCO0NBQ25HLENBQUMsQ0FBQztBQUVIOztHQUVHO0FBQ29CLGdCQUFNLEdBQUcsSUFBSSxTQUFTLENBQUM7SUFDNUMsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsTUFBTTtJQUN6Qyx1RUFBdUU7SUFDdkUsYUFBYSxFQUFFLDZDQUE2QyxnQkFBZ0IsQ0FBQyxHQUFHLGFBQWEsZ0JBQWdCLENBQUMsR0FBRyxFQUFFO0NBQ3BILENBQUMsQ0FBQztBQUVIOztHQUVHO0FBQ29CLGNBQUksR0FBRyxJQUFJLFNBQVMsQ0FBQyxFQUFFLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLElBQUksRUFBRSxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuXG5leHBvcnQgZW51bSBEZXBlbmRlbmNpZXNGaWxlIHtcbiAgUElQID0gJ3JlcXVpcmVtZW50cy50eHQnLFxuICBQT0VUUlkgPSAncG9ldHJ5LmxvY2snLFxuICBQSVBFTlYgPSAnUGlwZmlsZS5sb2NrJyxcbiAgTk9ORSA9ICcnXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUGFja2FnaW5nUHJvcHMge1xuICAvKipcbiAgICogRGVwZW5kZW5jeSBmaWxlIGZvciB0aGUgdHlwZSBvZiBwYWNrYWdpbmcuXG4gICAqL1xuICByZWFkb25seSBkZXBlbmRlbmNpZXNGaWxlOiBEZXBlbmRlbmNpZXNGaWxlO1xuICAvKipcbiAgICogQ29tbWFuZCB0byBleHBvcnQgdGhlIGRlcGVuZGVuY2llcyBpbnRvIGEgcGlwLWNvbXBhdGlibGUgYHJlcXVpcmVtZW50cy50eHRgIGZvcm1hdC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBkZXBlbmRlbmNpZXMgYXJlIGV4cG9ydGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhwb3J0Q29tbWFuZD86IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIFBhY2thZ2luZyB7XG5cbiAgLyoqXG4gICAqIFN0YW5kYXJkIHBhY2thZ2luZyB3aXRoIGBwaXBgLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQSVAgPSBuZXcgUGFja2FnaW5nKHtcbiAgICBkZXBlbmRlbmNpZXNGaWxlOiBEZXBlbmRlbmNpZXNGaWxlLlBJUCxcbiAgfSk7XG5cbiAgLyoqXG4gICAqIFBhY2thZ2luZyB3aXRoIGBwaXBlbnZgLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQSVBFTlYgPSBuZXcgUGFja2FnaW5nKHtcbiAgICBkZXBlbmRlbmNpZXNGaWxlOiBEZXBlbmRlbmNpZXNGaWxlLlBJUEVOVixcbiAgICAvLyBCeSBkZWZhdWx0LCBwaXBlbnYgY3JlYXRlcyBhIHZpcnR1YWxlbnYgaW4gYC8ubG9jYWxgLCBzbyB3ZSBmb3JjZSBpdCB0byBjcmVhdGUgb25lIGluIHRoZSBwYWNrYWdlIGRpcmVjdG9yeS5cbiAgICAvLyBBdCB0aGUgZW5kLCB3ZSByZW1vdmUgdGhlIHZpcnR1YWxlbnYgdG8gYXZvaWQgY3JlYXRpbmcgYSBkdXBsaWNhdGUgY29weSBpbiB0aGUgTGFtYmRhIHBhY2thZ2UuXG4gICAgZXhwb3J0Q29tbWFuZDogYFBJUEVOVl9WRU5WX0lOX1BST0pFQ1Q9MSBwaXBlbnYgbG9jayAtciA+ICR7RGVwZW5kZW5jaWVzRmlsZS5QSVB9ICYmIHJtIC1yZiAudmVudmAsXG4gIH0pO1xuXG4gIC8qKlxuICAgKiBQYWNrYWdpbmcgd2l0aCBgcG9ldHJ5YC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUE9FVFJZID0gbmV3IFBhY2thZ2luZyh7XG4gICAgZGVwZW5kZW5jaWVzRmlsZTogRGVwZW5kZW5jaWVzRmlsZS5QT0VUUlksXG4gICAgLy8gRXhwb3J0IGRlcGVuZGVuY2llcyB3aXRoIGNyZWRlbnRpYWxzIGF2YWlhYmxlIGluIHRoZSBidW5kbGluZyBpbWFnZS5cbiAgICBleHBvcnRDb21tYW5kOiBgcG9ldHJ5IGV4cG9ydCAtLXdpdGgtY3JlZGVudGlhbHMgLS1mb3JtYXQgJHtEZXBlbmRlbmNpZXNGaWxlLlBJUH0gLS1vdXRwdXQgJHtEZXBlbmRlbmNpZXNGaWxlLlBJUH1gLFxuICB9KTtcblxuICAvKipcbiAgICogTm8gZGVwZW5kZW5jaWVzIG9yIHBhY2thZ2luZy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTk9ORSA9IG5ldyBQYWNrYWdpbmcoeyBkZXBlbmRlbmNpZXNGaWxlOiBEZXBlbmRlbmNpZXNGaWxlLk5PTkUgfSk7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tRW50cnkoZW50cnk6IHN0cmluZyk6IFBhY2thZ2luZyB7XG4gICAgaWYgKGZzLmV4aXN0c1N5bmMocGF0aC5qb2luKGVudHJ5LCBEZXBlbmRlbmNpZXNGaWxlLlBJUEVOVikpKSB7XG4gICAgICByZXR1cm4gUGFja2FnaW5nLlBJUEVOVjtcbiAgICB9IGlmIChmcy5leGlzdHNTeW5jKHBhdGguam9pbihlbnRyeSwgRGVwZW5kZW5jaWVzRmlsZS5QT0VUUlkpKSkge1xuICAgICAgcmV0dXJuIFBhY2thZ2luZy5QT0VUUlk7XG4gICAgfSBlbHNlIGlmIChmcy5leGlzdHNTeW5jKHBhdGguam9pbihlbnRyeSwgRGVwZW5kZW5jaWVzRmlsZS5QSVApKSkge1xuICAgICAgcmV0dXJuIFBhY2thZ2luZy5QSVA7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBQYWNrYWdpbmcuTk9ORTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgZGVwZW5kZW5jaWVzRmlsZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZXhwb3J0Q29tbWFuZD86IHN0cmluZztcbiAgY29uc3RydWN0b3IocHJvcHM6IFBhY2thZ2luZ1Byb3BzKSB7XG4gICAgdGhpcy5kZXBlbmRlbmNpZXNGaWxlID0gcHJvcHMuZGVwZW5kZW5jaWVzRmlsZTtcbiAgICB0aGlzLmV4cG9ydENvbW1hbmQgPSBwcm9wcy5leHBvcnRDb21tYW5kO1xuICB9XG59Il19