"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda");
const cdk = require("../../core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath, props.logLevel);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('typescript');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        const pathJoin = osPathJoin(options.osPlatform);
        let relativeEntryPath = pathJoin(options.inputDir, this.relativeEntryPath);
        let tscCommand = '';
        if (this.props.preCompilation) {
            const tsconfig = (_a = this.props.tsconfig) !== null && _a !== void 0 ? _a : util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
            if (!tsconfig) {
                throw new Error('Cannot find a `tsconfig.json` but `preCompilation` is set to `true`, please specify it via `tsconfig`');
            }
            const compilerOptions = util_1.getTsconfigCompilerOptions(tsconfig);
            tscCommand = `${options.tscRunner} "${relativeEntryPath}" ${compilerOptions}`;
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
        }
        const loaders = Object.entries((_b = this.props.loader) !== null && _b !== void 0 ? _b : {});
        const defines = Object.entries((_c = this.props.define) !== null && _c !== void 0 ? _c : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : this.props.sourceMap;
        const sourceMapMode = (_e = this.props.sourceMapMode) !== null && _e !== void 0 ? _e : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_f = this.props.sourcesContent) !== null && _f !== void 0 ? _f : true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${relativeEntryPath}"`,
            `--target=${(_g = this.props.target) !== null && _g !== void 0 ? _g : toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
            ...this.props.inject ? this.props.inject.map(i => `--inject:${i}`) : [],
            ...this.props.esbuildArgs ? [toCliArgs(this.props.esbuildArgs)] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_h = this.relativeDepsLockFilePath) !== null && _h !== void 0 ? _h : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_k = (_j = this.props.commandHooks) === null || _j === void 0 ? void 0 : _j.beforeBundling(options.inputDir, options.outputDir)) !== null && _k !== void 0 ? _k : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_m = (this.props.nodeModules && ((_l = this.props.commandHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall(options.inputDir, options.outputDir)))) !== null && _m !== void 0 ? _m : [],
            depsCommand,
            ...(_p = (_o = this.props.commandHooks) === null || _o === void 0 ? void 0 : _o.afterBundling(options.inputDir, options.outputDir)) !== null && _p !== void 0 ? _p : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > "${filePath}"`;
        }
        return `echo '${stringifiedData}' > "${filePath}"`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
function toCliArgs(esbuildArgs) {
    const args = [];
    for (const [key, value] of Object.entries(esbuildArgs)) {
        if (value === true || value === '') {
            args.push(key);
        }
        else if (value) {
            args.push(`${key}="${value}"`);
        }
    }
    return args.join(' ');
}
//# sourceMappingURL=data:application/json;base64,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