"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    /**
     * BASIC_AUTH (SASL/PLAIN) authentication method for your Kafka cluster
     */
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
    /**
     * CLIENT_CERTIFICATE_TLS_AUTH (mTLS) authentication method for your Kafka cluster
     */
    AuthenticationMethod["CLIENT_CERTIFICATE_TLS_AUTH"] = "CLIENT_CERTIFICATE_TLS_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        this._eventSourceMappingId = undefined;
        jsiiDeprecationWarnings.monocdk_aws_lambda_event_sources_ManagedKafkaEventSourceProps(props);
        this.innerProps = props;
    }
    bind(target) {
        var _c;
        jsiiDeprecationWarnings.monocdk_aws_lambda_IFunction(target);
        const eventSourceMapping = target.addEventSourceMapping(`KafkaEventSource:${core_1.Names.nodeUniqueId(target.node)}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        (_c = target.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
    /**
    * The identifier for this EventSourceMapping
    */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KafkaEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "monocdk.aws_lambda_event_sources.ManagedKafkaEventSource", version: "1.149.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        jsiiDeprecationWarnings.monocdk_aws_lambda_event_sources_SelfManagedKafkaEventSourceProps(props);
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    bind(target) {
        jsiiDeprecationWarnings.monocdk_aws_lambda_IFunction(target);
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        var _c;
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.CLIENT_CERTIFICATE_TLS_AUTH:
                authType = lambda.SourceAccessConfigurationType.CLIENT_CERTIFICATE_TLS_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            (_c = this.innerProps.vpc) === null || _c === void 0 ? void 0 : _c.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "monocdk.aws_lambda_event_sources.SelfManagedKafkaEventSource", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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