"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Key = exports.KeyUsage = exports.KeySpec = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const cxschema = require("../../cloud-assembly-schema");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
const perms = require("./private/perms");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return { statementAdded: false };
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    validate() {
        var _b;
        const errors = super.validate();
        errors.push(...((_b = this.policy) === null || _b === void 0 ? void 0 : _b.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (this.trustAccountIdentities && !crossEnvironment) {
            return iam.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return iam.Grant.addToPrincipalAndResource({
                ...grantOptions,
                // if the key is used in a cross-environment matter,
                // we can't access the Key ARN (they don't have physical names),
                // so fall back to using '*'. ToDo we need to make this better... somehow
                resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
                resourcePolicyPrincipal: principal,
            });
        }
    }
    /**
     * Grant decryption permissions using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, ...perms.DECRYPT_ACTIONS);
    }
    /**
     * Grant encryption permissions using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, ...perms.ENCRYPT_ACTIONS);
    }
    /**
     * Grant encryption and decryption permissions using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, ...[...perms.DECRYPT_ACTIONS, ...perms.ENCRYPT_ACTIONS]);
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        const grantPrincipal = grantee.grantPrincipal;
        if (!isConstruct(grantPrincipal)) {
            return undefined;
        }
        // this logic should only apply to newly created
        // (= not imported) resources
        if (!this.principalIsANewlyCreatedResource(grantPrincipal)) {
            return undefined;
        }
        // return undefined;
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantPrincipal);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    principalIsANewlyCreatedResource(principal) {
        // yes, this sucks
        // this is just a temporary stopgap to stem the bleeding while we work on a proper fix
        return principal instanceof iam.Role ||
            principal instanceof iam.User ||
            principal instanceof iam.Group;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!isConstruct(grantee)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!isConstruct(grantee)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * The key spec, represents the cryptographic configuration of keys.
 */
var KeySpec;
(function (KeySpec) {
    /**
     * The default key spec.
     *
     * Valid usage: ENCRYPT_DECRYPT
     */
    KeySpec["SYMMETRIC_DEFAULT"] = "SYMMETRIC_DEFAULT";
    /**
     * RSA with 2048 bits of key.
     *
     * Valid usage: ENCRYPT_DECRYPT and SIGN_VERIFY
     */
    KeySpec["RSA_2048"] = "RSA_2048";
    /**
     * RSA with 3072 bits of key.
     *
     * Valid usage: ENCRYPT_DECRYPT and SIGN_VERIFY
     */
    KeySpec["RSA_3072"] = "RSA_3072";
    /**
     * RSA with 4096 bits of key.
     *
     * Valid usage: ENCRYPT_DECRYPT and SIGN_VERIFY
     */
    KeySpec["RSA_4096"] = "RSA_4096";
    /**
     * NIST FIPS 186-4, Section 6.4, ECDSA signature using the curve specified by the key and
     * SHA-256 for the message digest.
     *
     * Valid usage: SIGN_VERIFY
     */
    KeySpec["ECC_NIST_P256"] = "ECC_NIST_P256";
    /**
     * NIST FIPS 186-4, Section 6.4, ECDSA signature using the curve specified by the key and
     * SHA-384 for the message digest.
     *
     * Valid usage: SIGN_VERIFY
     */
    KeySpec["ECC_NIST_P384"] = "ECC_NIST_P384";
    /**
     * NIST FIPS 186-4, Section 6.4, ECDSA signature using the curve specified by the key and
     * SHA-512 for the message digest.
     *
     * Valid usage: SIGN_VERIFY
     */
    KeySpec["ECC_NIST_P521"] = "ECC_NIST_P521";
    /**
     * Standards for Efficient Cryptography 2, Section 2.4.1, ECDSA signature on the Koblitz curve.
     *
     * Valid usage: SIGN_VERIFY
     */
    KeySpec["ECC_SECG_P256K1"] = "ECC_SECG_P256K1";
})(KeySpec = exports.KeySpec || (exports.KeySpec = {}));
/**
 * The key usage, represents the cryptographic operations of keys.
 */
var KeyUsage;
(function (KeyUsage) {
    /**
     * Encryption and decryption.
     */
    KeyUsage["ENCRYPT_DECRYPT"] = "ENCRYPT_DECRYPT";
    /**
     * Signing and verification
     */
    KeyUsage["SIGN_VERIFY"] = "SIGN_VERIFY";
})(KeyUsage = exports.KeyUsage || (exports.KeyUsage = {}));
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_kms_KeyProps(props);
        const denyLists = {
            [KeyUsage.ENCRYPT_DECRYPT]: [
                KeySpec.ECC_NIST_P256,
                KeySpec.ECC_NIST_P384,
                KeySpec.ECC_NIST_P521,
                KeySpec.ECC_SECG_P256K1,
            ],
            [KeyUsage.SIGN_VERIFY]: [
                KeySpec.SYMMETRIC_DEFAULT,
            ],
        };
        const keySpec = (_b = props.keySpec) !== null && _b !== void 0 ? _b : KeySpec.SYMMETRIC_DEFAULT;
        const keyUsage = (_c = props.keyUsage) !== null && _c !== void 0 ? _c : KeyUsage.ENCRYPT_DECRYPT;
        if (denyLists[keyUsage].includes(keySpec)) {
            throw new Error(`key spec '${keySpec}' is not valid with usage '${keyUsage}'`);
        }
        if (keySpec !== KeySpec.SYMMETRIC_DEFAULT && props.enableKeyRotation) {
            throw new Error('key rotation cannot be enabled on asymmetric keys');
        }
        const defaultKeyPoliciesFeatureEnabled = core_1.FeatureFlags.of(this).isEnabled(cxapi.KMS_DEFAULT_KEY_POLICIES);
        this.policy = (_d = props.policy) !== null && _d !== void 0 ? _d : new iam.PolicyDocument();
        if (defaultKeyPoliciesFeatureEnabled) {
            if (props.trustAccountIdentities === false) {
                throw new Error('`trustAccountIdentities` cannot be false if the @aws-cdk/aws-kms:defaultKeyPolicies feature flag is set');
            }
            this.trustAccountIdentities = true;
            // Set the default key policy if one hasn't been provided by the user.
            if (!props.policy) {
                this.addDefaultAdminPolicy();
            }
        }
        else {
            this.trustAccountIdentities = (_e = props.trustAccountIdentities) !== null && _e !== void 0 ? _e : false;
            if (this.trustAccountIdentities) {
                this.addDefaultAdminPolicy();
            }
            else {
                this.addLegacyAdminPolicy();
            }
        }
        let pendingWindowInDays;
        if (props.pendingWindow) {
            pendingWindowInDays = props.pendingWindow.toDays();
            if (pendingWindowInDays < 7 || pendingWindowInDays > 30) {
                throw new Error(`'pendingWindow' value must between 7 and 30 days. Received: ${pendingWindowInDays}`);
            }
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keySpec: props.keySpec,
            keyUsage: props.keyUsage,
            keyPolicy: this.policy,
            pendingWindowInDays: pendingWindowInDays,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        ((_f = props.admins) !== null && _f !== void 0 ? _f : []).forEach((p) => this.grantAdmin(p));
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).splitArn(keyArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    /**
     * Create a mutable {@link IKey} based on a low-level {@link CfnKey}.
     * This is most useful when combined with the cloudformation-include module.
     * This method is different than {@link fromKeyArn()} because the {@link IKey}
     * returned from this method is mutable;
     * meaning, calling any mutating methods on it,
     * like {@link IKey.addToResourcePolicy()},
     * will actually be reflected in the resulting template,
     * as opposed to the object returned from {@link fromKeyArn()},
     * on which calling those methods would have no effect.
     */
    static fromCfnKey(cfnKey) {
        jsiiDeprecationWarnings.monocdk_aws_kms_CfnKey(cfnKey);
        // use a "weird" id that has a higher chance of being unique
        const id = '@FromCfnKey';
        // if fromCfnKey() was already called on this cfnKey,
        // return the same L2
        // (as different L2s would conflict, because of the mutation of the keyPolicy property of the L1 below)
        const existing = cfnKey.node.tryFindChild(id);
        if (existing) {
            return existing;
        }
        let keyPolicy;
        try {
            keyPolicy = iam.PolicyDocument.fromJson(cfnKey.keyPolicy);
        }
        catch (e) {
            // If the KeyPolicy contains any CloudFormation functions,
            // PolicyDocument.fromJson() throws an exception.
            // In that case, because we would have to effectively make the returned IKey immutable,
            // throw an exception suggesting to use the other importing methods instead.
            // We might make this parsing logic smarter later,
            // but let's start by erroring out.
            throw new Error('Could not parse the PolicyDocument of the passed AWS::KMS::Key resource because it contains CloudFormation functions. ' +
                'This makes it impossible to create a mutable IKey from that Policy. ' +
                'You have to use fromKeyArn instead, passing it the ARN attribute property of the low-level CfnKey');
        }
        // change the key policy of the L1, so that all changes done in the L2 are reflected in the resulting template
        cfnKey.keyPolicy = core_1.Lazy.any({ produce: () => keyPolicy.toJSON() });
        return new class extends KeyBase {
            constructor() {
                super(...arguments);
                this.keyArn = cfnKey.attrArn;
                this.keyId = cfnKey.ref;
                this.policy = keyPolicy;
                this.trustAccountIdentities = false;
            }
        }(cfnKey, id);
    }
    /**
     * Import an existing Key by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use Keys not defined in your CDK
     * application. If you are looking to share a Key between stacks, you can
     * pass the `Key` object between stacks and use it as normal. In addition,
     * it's not necessary to use this method if an interface accepts an `IKey`.
     * In this case, `Alias.fromAliasName()` can be used which returns an alias
     * that extends `IKey`.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The Key information will be cached in `cdk.context.json` and the same Key
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        jsiiDeprecationWarnings.monocdk_aws_kms_KeyLookupOptions(options);
        class Import extends KeyBase {
            constructor(keyId, keyArn) {
                super(scope, id);
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
                this.keyArn = keyArn;
            }
        }
        if (core_1.Token.isUnresolved(options.aliasName)) {
            throw new Error('All arguments to Key.fromLookup() must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.KEY_PROVIDER,
            props: {
                aliasName: options.aliasName,
            },
            dummyValue: {
                keyId: '1234abcd-12ab-34cd-56ef-1234567890ab',
            },
        }).value;
        return new Import(attributes.keyId, core_1.Arn.format({ resource: 'key', service: 'kms', resourceName: attributes.keyId }, core_1.Stack.of(scope)));
    }
    /**
     * Grant admins permissions using this key to the given principal
     *
     * Key administrators have permissions to manage the key (e.g., change permissions, revoke), but do not have permissions
     * to use the key in cryptographic operations (e.g., encrypt, decrypt).
     */
    grantAdmin(grantee) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        return this.grant(grantee, ...perms.ADMIN_ACTIONS);
    }
    /**
     * Adds the default key policy to the key. This policy gives the AWS account (root user) full access to the CMK,
     * which reduces the risk of the CMK becoming unmanageable and enables IAM policies to allow access to the CMK.
     * This is the same policy that is default when creating a Key via the KMS API or Console.
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     */
    addDefaultAdminPolicy() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Grants the account admin privileges -- not full account access -- plus the GenerateDataKey action.
     * The GenerateDataKey action was added for interop with S3 in https://github.com/aws/aws-cdk/issues/3458.
     *
     * This policy is discouraged and deprecated by the `@aws-cdk/aws-kms:defaultKeyPolicies` feature flag.
     *
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @deprecated
     */
    addLegacyAdminPolicy() {
        // This is equivalent to `[...perms.ADMIN_ACTIONS, 'kms:GenerateDataKey']`,
        // but keeping this explicit ordering for backwards-compatibility (changing the ordering causes resource updates)
        const actions = [
            'kms:Create*',
            'kms:Describe*',
            'kms:Enable*',
            'kms:List*',
            'kms:Put*',
            'kms:Update*',
            'kms:Revoke*',
            'kms:Disable*',
            'kms:Get*',
            'kms:Delete*',
            'kms:ScheduleKeyDeletion',
            'kms:CancelKeyDeletion',
            'kms:GenerateDataKey',
            'kms:TagResource',
            'kms:UntagResource',
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
exports.Key = Key;
_a = JSII_RTTI_SYMBOL_1;
Key[_a] = { fqn: "monocdk.aws_kms.Key", version: "1.149.0" };
/**
 * Whether the given object is a Construct
 *
 * Normally we'd do `x instanceof Construct`, but that is not robust against
 * multiple copies of the `constructs` library on disk. This can happen
 * when upgrading and downgrading between v2 and v1, and in the use of CDK
 * Pipelines is going to an error that says "Can't use Pipeline/Pipeline/Role in
 * a cross-environment fashion", which is very confusing.
 */
function isConstruct(x) {
    const sym = Symbol.for('constructs.Construct.node');
    return (typeof x === 'object' && x &&
        (x instanceof constructs_1.Construct // happy fast case
            || !!x.node // constructs v10
            || !!x[sym])); // constructs v3
}
//# sourceMappingURL=data:application/json;base64,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