"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeliveryStream = exports.StreamEncryption = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const cdk = require("../../core");
const region_info_1 = require("../../region-info");
const constructs_1 = require("constructs");
const kinesisfirehose_canned_metrics_generated_1 = require("./kinesisfirehose-canned-metrics.generated");
const kinesisfirehose_generated_1 = require("./kinesisfirehose.generated");
const PUT_RECORD_ACTIONS = [
    'firehose:PutRecord',
    'firehose:PutRecordBatch',
];
/**
 * Base class for new and imported Kinesis Data Firehose delivery streams.
 */
class DeliveryStreamBase extends cdk.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.connections = setConnections(this);
    }
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            resourceArns: [this.deliveryStreamArn],
            grantee: grantee,
            actions: actions,
        });
    }
    grantPutRecords(grantee) {
        return this.grant(grantee, ...PUT_RECORD_ACTIONS);
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Firehose',
            metricName: metricName,
            dimensionsMap: {
                DeliveryStreamName: this.deliveryStreamName,
            },
            ...props,
        }).attachTo(this);
    }
    metricIncomingBytes(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.incomingBytesSum, props);
    }
    metricIncomingRecords(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.incomingRecordsSum, props);
    }
    metricBackupToS3Bytes(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3BytesSum, props);
    }
    metricBackupToS3DataFreshness(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3DataFreshnessAverage, props);
    }
    metricBackupToS3Records(props) {
        return this.cannedMetric(kinesisfirehose_canned_metrics_generated_1.FirehoseMetrics.backupToS3RecordsSum, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ DeliveryStreamName: this.deliveryStreamName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Options for server-side encryption of a delivery stream.
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Data in the stream is stored unencrypted.
     */
    StreamEncryption[StreamEncryption["UNENCRYPTED"] = 0] = "UNENCRYPTED";
    /**
     * Data in the stream is stored encrypted by a KMS key managed by the customer.
     */
    StreamEncryption[StreamEncryption["CUSTOMER_MANAGED"] = 1] = "CUSTOMER_MANAGED";
    /**
     * Data in the stream is stored encrypted by a KMS key owned by AWS and managed for use in multiple AWS accounts.
     */
    StreamEncryption[StreamEncryption["AWS_OWNED"] = 2] = "AWS_OWNED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
/**
 * Create a Kinesis Data Firehose delivery stream
 *
 * @resource AWS::KinesisFirehose::DeliveryStream
 */
class DeliveryStream extends DeliveryStreamBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.deliveryStreamName,
        });
        jsiiDeprecationWarnings.monocdk_aws_kinesisfirehose_DeliveryStreamProps(props);
        if (props.destinations.length !== 1) {
            throw new Error(`Only one destination is allowed per delivery stream, given ${props.destinations.length}`);
        }
        const role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Service Role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        this.grantPrincipal = role;
        if (props.sourceStream &&
            (props.encryption === StreamEncryption.AWS_OWNED || props.encryption === StreamEncryption.CUSTOMER_MANAGED || props.encryptionKey)) {
            throw new Error('Requested server-side encryption but delivery stream source is a Kinesis data stream. Specify server-side encryption on the data stream instead.');
        }
        if ((props.encryption === StreamEncryption.AWS_OWNED || props.encryption === StreamEncryption.UNENCRYPTED) && props.encryptionKey) {
            throw new Error(`Specified stream encryption as ${StreamEncryption[props.encryption]} but provided a customer-managed key`);
        }
        const encryptionKey = (_c = props.encryptionKey) !== null && _c !== void 0 ? _c : (props.encryption === StreamEncryption.CUSTOMER_MANAGED ? new kms.Key(this, 'Key') : undefined);
        const encryptionConfig = (encryptionKey || (props.encryption === StreamEncryption.AWS_OWNED)) ? {
            keyArn: encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.keyArn,
            keyType: encryptionKey ? 'CUSTOMER_MANAGED_CMK' : 'AWS_OWNED_CMK',
        } : undefined;
        /*
         * In order for the service role to have access to the encryption key before the delivery stream is created, the
         * CfnDeliveryStream below should have a dependency on the grant returned by the function call below:
         * > `keyGrant?.applyBefore(resource)`
         * However, an error during synthesis is thrown if this is added:
         * > ${Token[PolicyDocument.###]} does not implement DependableTrait
         * Data will not be lost if the permissions are not granted to the service role immediately; Firehose has a 24 hour
         * period where data will be buffered and retried if access is denied to the encryption key. For that reason, it is
         * acceptable to omit the dependency for now. See: https://github.com/aws/aws-cdk/issues/15790
         */
        encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.grantEncryptDecrypt(role);
        const sourceStreamConfig = props.sourceStream ? {
            kinesisStreamArn: props.sourceStream.streamArn,
            roleArn: role.roleArn,
        } : undefined;
        const readStreamGrant = (_d = props.sourceStream) === null || _d === void 0 ? void 0 : _d.grantRead(role);
        const destinationConfig = props.destinations[0].bind(this, {});
        const resource = new kinesisfirehose_generated_1.CfnDeliveryStream(this, 'Resource', {
            deliveryStreamEncryptionConfigurationInput: encryptionConfig,
            deliveryStreamName: props.deliveryStreamName,
            deliveryStreamType: props.sourceStream ? 'KinesisStreamAsSource' : 'DirectPut',
            kinesisStreamSourceConfiguration: sourceStreamConfig,
            ...destinationConfig,
        });
        (_e = destinationConfig.dependables) === null || _e === void 0 ? void 0 : _e.forEach(dependable => resource.node.addDependency(dependable));
        if (readStreamGrant) {
            resource.node.addDependency(readStreamGrant);
        }
        this.deliveryStreamArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'kinesis',
            resource: 'deliverystream',
            resourceName: this.physicalName,
        });
        this.deliveryStreamName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import an existing delivery stream from its name.
     */
    static fromDeliveryStreamName(scope, id, deliveryStreamName) {
        return this.fromDeliveryStreamAttributes(scope, id, { deliveryStreamName });
    }
    /**
     * Import an existing delivery stream from its ARN.
     */
    static fromDeliveryStreamArn(scope, id, deliveryStreamArn) {
        return this.fromDeliveryStreamAttributes(scope, id, { deliveryStreamArn });
    }
    /**
     * Import an existing delivery stream from its attributes.
     */
    static fromDeliveryStreamAttributes(scope, id, attrs) {
        var _b, _c;
        jsiiDeprecationWarnings.monocdk_aws_kinesisfirehose_DeliveryStreamAttributes(attrs);
        if (!attrs.deliveryStreamName && !attrs.deliveryStreamArn) {
            throw new Error('Either deliveryStreamName or deliveryStreamArn must be provided in DeliveryStreamAttributes');
        }
        const deliveryStreamName = (_b = attrs.deliveryStreamName) !== null && _b !== void 0 ? _b : cdk.Stack.of(scope).splitArn(attrs.deliveryStreamArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!deliveryStreamName) {
            throw new Error(`No delivery stream name found in ARN: '${attrs.deliveryStreamArn}'`);
        }
        const deliveryStreamArn = (_c = attrs.deliveryStreamArn) !== null && _c !== void 0 ? _c : cdk.Stack.of(scope).formatArn({
            service: 'firehose',
            resource: 'deliverystream',
            resourceName: attrs.deliveryStreamName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends DeliveryStreamBase {
            constructor() {
                var _b;
                super(...arguments);
                this.deliveryStreamName = deliveryStreamName;
                this.deliveryStreamArn = deliveryStreamArn;
                this.grantPrincipal = (_b = attrs.role) !== null && _b !== void 0 ? _b : new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
}
exports.DeliveryStream = DeliveryStream;
_a = JSII_RTTI_SYMBOL_1;
DeliveryStream[_a] = { fqn: "monocdk.aws_kinesisfirehose.DeliveryStream", version: "1.149.0" };
function setConnections(scope) {
    const stack = cdk.Stack.of(scope);
    const mappingId = '@aws-cdk/aws-kinesisfirehose.CidrBlocks';
    let cfnMapping = constructs_1.Node.of(stack).tryFindChild(mappingId);
    if (!cfnMapping) {
        const mapping = {};
        region_info_1.RegionInfo.regions.forEach((regionInfo) => {
            if (regionInfo.firehoseCidrBlock) {
                mapping[regionInfo.name] = {
                    FirehoseCidrBlock: regionInfo.firehoseCidrBlock,
                };
            }
        });
        cfnMapping = new cdk.CfnMapping(stack, mappingId, {
            mapping,
            lazy: true,
        });
    }
    const cidrBlock = cfnMapping.findInMap(stack.region, 'FirehoseCidrBlock');
    return new ec2.Connections({
        peer: ec2.Peer.ipv4(cidrBlock),
    });
}
//# sourceMappingURL=data:application/json;base64,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