"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const aws_kinesisanalytics_1 = require("../../aws-kinesisanalytics");
const logs = require("../../aws-logs");
const core = require("../../core");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience {@link IApplication.addToPrincipalPolicy} method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        this.applicationName = attrs.applicationName;
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, { physicalName: props.applicationName });
        jsiiDeprecationWarnings.monocdk_aws_kinesisanalytics_flink_ApplicationProps(props);
        validation_1.validateFlinkApplicationProps(props);
        this.role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: (_c = props.snapshotsEnabled) !== null && _c !== void 0 ? _c : true,
                },
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn, applicationName });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const applicationName = core.Stack.of(scope).splitArn(applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${applicationArn}) must include resource name`);
        }
        return new Import(scope, id, { applicationArn, applicationName });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "monocdk.aws_kinesisanalytics_flink.Application", version: "1.149.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFDckMscUVBQXVHO0FBQ3ZHLHVDQUF1QztBQUN2QyxtQ0FBbUM7QUFHbkMsNkVBQXlFO0FBQ3pFLCtGQUEwRjtBQUMxRixxREFBaUc7QUFpQ2pHOzs7R0FHRztBQUNILE1BQWUsZUFBZ0IsU0FBUSxJQUFJLENBQUMsUUFBUTtJQVFsRCxrRkFBa0Y7SUFDM0UsZUFBZSxDQUFDLGVBQW9DO1FBQ3pELElBQUksSUFBSSxDQUFDLElBQUksRUFBRTtZQUNiLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDaEQsT0FBTyxJQUFJLENBQUM7U0FDYjtRQUVELE9BQU8sS0FBSyxDQUFDO0tBQ2Q7Q0FDRjtBQTRIRDs7R0FFRztBQUNILE1BQU0sTUFBTyxTQUFRLGVBQWU7SUFNbEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwRDtRQUNsRyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLGtFQUFrRTtRQUNsRSxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxDQUFDLElBQUksR0FBRyxTQUFTLENBQUM7UUFFdEIsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQzNDLElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztLQUM5QztDQUNGO0FBRUQ7Ozs7O0dBS0c7QUFDSCxNQUFhLFdBQVksU0FBUSxlQUFlO0lBZ0M5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCOztRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFlBQVksRUFBRSxLQUFLLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQzs7UUFDMUQsMENBQXdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFaEMsSUFBSSxDQUFDLElBQUksU0FBRyxLQUFLLENBQUMsSUFBSSxtQ0FBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUNuRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsZ0NBQWdDLENBQUM7U0FDdEUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBRWhDLHlDQUF5QztRQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxPQUFPLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQztZQUNyQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1QixNQUFNLFFBQVEsR0FBRyxJQUFJLHVDQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDdEQsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSztZQUN2QyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87WUFDdkMsd0JBQXdCLEVBQUU7Z0JBQ3hCLEdBQUcsSUFBSSxDQUFDLG9DQUFvQztnQkFDNUMscUJBQXFCLEVBQUUsOENBQXFCLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQkFDbEUsNkJBQTZCLEVBQUUsK0RBQTZCLENBQUM7b0JBQzNELG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7b0JBQ2hELGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7b0JBQzVDLDBCQUEwQixFQUFFLEtBQUssQ0FBQywwQkFBMEI7b0JBQzVELFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtvQkFDeEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO29CQUNoQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO29CQUM1QyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7b0JBQzlCLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7aUJBQzNDLENBQUM7Z0JBQ0YsZ0NBQWdDLEVBQUU7b0JBQ2hDLGdCQUFnQixRQUFFLEtBQUssQ0FBQyxnQkFBZ0IsbUNBQUksSUFBSTtpQkFDakQ7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV2QyxNQUFNLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUV0RSxvQkFBb0I7UUFFcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDckQsT0FBTyxFQUFFLENBQUMsd0JBQXdCLENBQUM7WUFDbkMsU0FBUyxFQUFFO2dCQUNULElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztvQkFDNUIsT0FBTyxFQUFFLE1BQU07b0JBQ2YsUUFBUSxFQUFFLFdBQVc7b0JBQ3JCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtvQkFDN0MsWUFBWSxFQUFFLEdBQUc7aUJBQ2xCLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDckQsT0FBTyxFQUFFLENBQUMseUJBQXlCLENBQUM7WUFDcEMsU0FBUyxFQUFFLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztTQUNsQyxDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sWUFBWSxHQUFHLE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLFNBQVMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLEdBQUcsQ0FBQyxVQUFVLGNBQWMsUUFBUSxDQUFDLFlBQVksZUFBZSxTQUFTLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDekssSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDckQsT0FBTyxFQUFFLENBQUMsbUJBQW1CLENBQUM7WUFDOUIsU0FBUyxFQUFFLENBQUMsWUFBWSxDQUFDO1NBQzFCLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSw4REFBdUMsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ2pFLGVBQWUsRUFBRSxRQUFRLENBQUMsR0FBRztZQUM3Qix1QkFBdUIsRUFBRTtnQkFDdkIsWUFBWTthQUNiO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUNoRCxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQ3JFLHdCQUF3QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FDNUMsQ0FBQztRQUVGLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQy9DLE9BQU8sRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDcEMsQ0FBQyxDQUFDO0tBQ0o7SUFySEQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGVBQXVCO1FBQ3JGLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1FBRWpHLE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxlQUFlLEVBQUUsQ0FBQyxDQUFDO0tBQ25FO0lBRUQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGNBQXNCO1FBQ25GLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxjQUFjLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQVksQ0FBQztRQUN2SCxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLGNBQWMsOEJBQThCLENBQUMsQ0FBQztTQUN6RztRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxlQUFlLEVBQUUsQ0FBQyxDQUFDO0tBQ25FOztBQXRCSCxrQ0F1SEM7OztBQUVELFNBQVMsd0JBQXdCLENBQUMsWUFBb0I7SUFDcEQsT0FBTztRQUNMLE9BQU8sRUFBRSxrQkFBa0I7UUFDM0IsUUFBUSxFQUFFLGFBQWE7UUFDdkIsWUFBWTtLQUNiLENBQUM7QUFDSixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ2ZuQXBwbGljYXRpb25DbG91ZFdhdGNoTG9nZ2luZ09wdGlvblYyLCBDZm5BcHBsaWNhdGlvblYyIH0gZnJvbSAnLi4vLi4vYXdzLWtpbmVzaXNhbmFseXRpY3MnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICcuLi8uLi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBjb3JlIGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBBcHBsaWNhdGlvbkNvZGUgfSBmcm9tICcuL2FwcGxpY2F0aW9uLWNvZGUnO1xuaW1wb3J0IHsgZW52aXJvbm1lbnRQcm9wZXJ0aWVzIH0gZnJvbSAnLi9wcml2YXRlL2Vudmlyb25tZW50LXByb3BlcnRpZXMnO1xuaW1wb3J0IHsgZmxpbmtBcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb24gfSBmcm9tICcuL3ByaXZhdGUvZmxpbmstYXBwbGljYXRpb24tY29uZmlndXJhdGlvbic7XG5pbXBvcnQgeyB2YWxpZGF0ZUZsaW5rQXBwbGljYXRpb25Qcm9wcyBhcyB2YWxpZGF0ZUFwcGxpY2F0aW9uUHJvcHMgfSBmcm9tICcuL3ByaXZhdGUvdmFsaWRhdGlvbic7XG5pbXBvcnQgeyBMb2dMZXZlbCwgTWV0cmljc0xldmVsLCBQcm9wZXJ0eUdyb3VwcywgUnVudGltZSB9IGZyb20gJy4vdHlwZXMnO1xuXG4vKipcbiAqIEFuIGludGVyZmFjZSBleHByZXNzaW5nIHRoZSBwdWJsaWMgcHJvcGVydGllcyBvbiBib3RoIGFuIGltcG9ydGVkIGFuZFxuICogQ0RLLWNyZWF0ZWQgRmxpbmsgYXBwbGljYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUFwcGxpY2F0aW9uIGV4dGVuZHMgY29yZS5JUmVzb3VyY2UsIGlhbS5JR3JhbnRhYmxlIHtcbiAgLyoqXG4gICAqIFRoZSBhcHBsaWNhdGlvbiBBUk4uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBGbGluayBhcHBsaWNhdGlvbi5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBhcHBsaWNhdGlvbiBJQU0gcm9sZS5cbiAgICovXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIENvbnZlbmllbmNlIG1ldGhvZCBmb3IgYWRkaW5nIGEgcG9saWN5IHN0YXRlbWVudCB0byB0aGUgYXBwbGljYXRpb24gcm9sZS5cbiAgICovXG4gIGFkZFRvUm9sZVBvbGljeShwb2xpY3lTdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpOiBib29sZWFuO1xufVxuXG4vKipcbiAqIEltcGxlbWVudHMgdGhlIGZ1bmN0aW9uYWxpdHkgc2hhcmVkIGJldHdlZW4gQ0RLIGNyZWF0ZWQgYW5kIGltcG9ydGVkXG4gKiBJQXBwbGljYXRpb25zLlxuICovXG5hYnN0cmFjdCBjbGFzcyBBcHBsaWNhdGlvbkJhc2UgZXh0ZW5kcyBjb3JlLlJlc291cmNlIGltcGxlbWVudHMgSUFwcGxpY2F0aW9uIHtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLy8gSW1wbGVtZW50IGlhbS5JR3JhbnRhYmxlIGludGVyZmFjZVxuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuXG4gIC8qKiBJbXBsZW1lbnQgdGhlIGNvbnZlbmllbmNlIHtAbGluayBJQXBwbGljYXRpb24uYWRkVG9QcmluY2lwYWxQb2xpY3l9IG1ldGhvZC4gKi9cbiAgcHVibGljIGFkZFRvUm9sZVBvbGljeShwb2xpY3lTdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpOiBib29sZWFuIHtcbiAgICBpZiAodGhpcy5yb2xlKSB7XG4gICAgICB0aGlzLnJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3kocG9saWN5U3RhdGVtZW50KTtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxufVxuXG4vKipcbiAqIFByb3BzIGZvciBjcmVhdGluZyBhbiBBcHBsaWNhdGlvbiBjb25zdHJ1Y3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBBIG5hbWUgZm9yIHlvdXIgQXBwbGljYXRpb24gdGhhdCBpcyB1bmlxdWUgdG8gYW4gQVdTIGFjY291bnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ2xvdWRGb3JtYXRpb24tZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEZsaW5rIHZlcnNpb24gdG8gdXNlIGZvciB0aGlzIGFwcGxpY2F0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcblxuICAvKipcbiAgICogVGhlIEZsaW5rIGNvZGUgYXNzZXQgdG8gcnVuLlxuICAgKi9cbiAgcmVhZG9ubHkgY29kZTogQXBwbGljYXRpb25Db2RlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIGNoZWNrcG9pbnRpbmcgaXMgZW5hYmxlZCB3aGlsZSB5b3VyIGFwcGxpY2F0aW9uIHJ1bnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNoZWNrcG9pbnRpbmdFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGludGVydmFsIGJldHdlZW4gY2hlY2twb2ludHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IDEgbWludXRlXG4gICAqL1xuICByZWFkb25seSBjaGVja3BvaW50SW50ZXJ2YWw/OiBjb3JlLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBhbW91bnQgb2YgdGltZSBpbiB0byB3YWl0IGFmdGVyIGEgY2hlY2twb2ludCBmaW5pc2hlcyB0byBzdGFydFxuICAgKiBhIG5ldyBjaGVja3BvaW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCA1IHNlY29uZHNcbiAgICovXG4gIHJlYWRvbmx5IG1pblBhdXNlQmV0d2VlbkNoZWNrcG9pbnRzPzogY29yZS5EdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGxldmVsIG9mIGxvZyB2ZXJib3NpdHkgZnJvbSB0aGUgRmxpbmsgYXBwbGljYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IEZsaW5rTG9nTGV2ZWwuSU5GT1xuICAgKi9cbiAgcmVhZG9ubHkgbG9nTGV2ZWw/OiBMb2dMZXZlbDtcblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBncmFudWxhcml0eSBvZiB0aGUgQ2xvdWRXYXRjaCBtZXRyaWNzIGZvciBhbiBhcHBsaWNhdGlvbi5cbiAgICogVXNlIGNhdXRpb24gd2l0aCBQYXJhbGxlbGlzbSBsZXZlbCBtZXRyaWNzLiBQYXJhbGxlbGlzbSBncmFudWxhcml0eSBsb2dzXG4gICAqIG1ldHJpY3MgZm9yIGVhY2ggcGFyYWxsZWwgdGhyZWFkIGFuZCBjYW4gcXVpY2tseSBiZWNvbWUgZXhwZW5zaXZlIHdoZW5cbiAgICogcGFyYWxsZWxpc20gaXMgaGlnaCAoZS5nLiA+IDY0KS5cbiAgICpcbiAgICogQGRlZmF1bHQgTWV0cmljc0xldmVsLkFQUExJQ0FUSU9OXG4gICAqL1xuICByZWFkb25seSBtZXRyaWNzTGV2ZWw/OiBNZXRyaWNzTGV2ZWw7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIEtpbmVzaXMgRGF0YSBBbmFseXRpY3Mgc2VydmljZSBjYW4gaW5jcmVhc2UgdGhlIHBhcmFsbGVsaXNtIG9mXG4gICAqIHRoZSBhcHBsaWNhdGlvbiBpbiByZXNwb25zZSB0byByZXNvdXJjZSB1c2FnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b1NjYWxpbmdFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGluaXRpYWwgcGFyYWxsZWxpc20gZm9yIHRoZSBhcHBsaWNhdGlvbi4gS2luZXNpcyBEYXRhIEFuYWx5dGljcyBjYW5cbiAgICogc3RvcCB0aGUgYXBwLCBpbmNyZWFzZSB0aGUgcGFyYWxsZWxpc20sIGFuZCBzdGFydCB0aGUgYXBwIGFnYWluIGlmXG4gICAqIGF1dG9TY2FsaW5nRW5hYmxlZCBpcyB0cnVlICh0aGUgZGVmYXVsdCB2YWx1ZSkuXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFsbGVsaXNtPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgRmxpbmsgcGFyYWxsZWxpc20gYWxsb3dlZCBwZXIgS2luZXNpcyBQcm9jZXNzaW5nIFVuaXQgKEtQVSkuXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFsbGVsaXNtUGVyS3B1PzogbnVtYmVyXG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgaWYgRmxpbmsgc25hcHNob3RzIGFyZSBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBzbmFwc2hvdHNFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBQcm9wZXJ0eUdyb3Vwcy4gWW91IGNhbiB1c2UgdGhlc2UgcHJvcGVydHkgZ3JvdXBzIHRvIHBhc3NcbiAgICogYXJiaXRyYXJ5IHJ1bnRpbWUgY29uZmlndXJhdGlvbiB2YWx1ZXMgdG8geW91ciBGbGluayBhcHAuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIHByb3BlcnR5IGdyb3VwIGNvbmZpZ3VyYXRpb24gcHJvdmlkZWQgdG8gdGhlIEZsaW5rIGFwcFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvcGVydHlHcm91cHM/OiBQcm9wZXJ0eUdyb3VwcztcblxuICAvKipcbiAgICogQSByb2xlIHRvIHVzZSB0byBncmFudCBwZXJtaXNzaW9ucyB0byB5b3VyIGFwcGxpY2F0aW9uLiBQcmVmZXIgb21pdHRpbmdcbiAgICogdGhpcyBwcm9wZXJ0eSBhbmQgdXNpbmcgdGhlIGRlZmF1bHQgcm9sZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyBSb2xlIHdpbGwgYmUgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogUHJvdmlkZSBhIFJlbW92YWxQb2xpY3kgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IGNvcmUuUmVtb3ZhbFBvbGljeTtcblxuICAvKipcbiAgICogVGhlIGxvZyBncm91cCB0byBzZW5kIGxvZyBlbnRyaWVzIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCBDREsncyBkZWZhdWx0IExvZ0dyb3VwXG4gICAqL1xuICByZWFkb25seSBsb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xufVxuXG4vKipcbiAqIEFuIGltcG9ydGVkIEZsaW5rIGFwcGxpY2F0aW9uLlxuICovXG5jbGFzcyBJbXBvcnQgZXh0ZW5kcyBBcHBsaWNhdGlvbkJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuICBwdWJsaWMgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbiAgcHVibGljIHJlYWRvbmx5IGFwcGxpY2F0aW9uTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb25Bcm46IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogeyBhcHBsaWNhdGlvbkFybjogc3RyaW5nLCBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZyB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vIEltcG9ydGVkIGFwcGxpY2F0aW9ucyBoYXZlIG5vIGFzc29jaWF0ZWQgcm9sZSBvciBncmFudFByaW5jaXBhbFxuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSBuZXcgaWFtLlVua25vd25QcmluY2lwYWwoeyByZXNvdXJjZTogdGhpcyB9KTtcbiAgICB0aGlzLnJvbGUgPSB1bmRlZmluZWQ7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uQXJuID0gYXR0cnMuYXBwbGljYXRpb25Bcm47XG4gICAgdGhpcy5hcHBsaWNhdGlvbk5hbWUgPSBhdHRycy5hcHBsaWNhdGlvbk5hbWU7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGUgTDIgY29uc3RydWN0IGZvciBGbGluayBLaW5lc2lzIERhdGEgQXBwbGljYXRpb25zLlxuICpcbiAqIEByZXNvdXJjZSBBV1M6OktpbmVzaXNBbmFseXRpY3NWMjo6QXBwbGljYXRpb25cbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBBcHBsaWNhdGlvbiBleHRlbmRzIEFwcGxpY2F0aW9uQmFzZSB7XG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgRmxpbmsgYXBwbGljYXRpb24gZGVmaW5lZCBvdXRzaWRlIG9mIENESyBjb2RlIGJ5XG4gICAqIGFwcGxpY2F0aW9uTmFtZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFwcGxpY2F0aW9uTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhcHBsaWNhdGlvbk5hbWU6IHN0cmluZyk6IElBcHBsaWNhdGlvbiB7XG4gICAgY29uc3QgYXBwbGljYXRpb25Bcm4gPSBjb3JlLlN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oYXBwbGljYXRpb25Bcm5Db21wb25lbnRzKGFwcGxpY2F0aW9uTmFtZSkpO1xuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkLCB7IGFwcGxpY2F0aW9uQXJuLCBhcHBsaWNhdGlvbk5hbWUgfSk7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIGFwcGxpY2F0aW9uIGRlZmluZWQgb3V0c2lkZSBvZiBDREsgY29kZSBieVxuICAgKiBhcHBsaWNhdGlvbkFybi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFwcGxpY2F0aW9uQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGFwcGxpY2F0aW9uQXJuOiBzdHJpbmcpOiBJQXBwbGljYXRpb24ge1xuICAgIGNvbnN0IGFwcGxpY2F0aW9uTmFtZSA9IGNvcmUuU3RhY2sub2Yoc2NvcGUpLnNwbGl0QXJuKGFwcGxpY2F0aW9uQXJuLCBjb3JlLkFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FKS5yZXNvdXJjZU5hbWU7XG4gICAgaWYgKCFhcHBsaWNhdGlvbk5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgYXBwbGljYXRpb25Bcm4gZm9yIGZyb21BcHBsaWNhdGlvbkFybiAoJHthcHBsaWNhdGlvbkFybn0pIG11c3QgaW5jbHVkZSByZXNvdXJjZSBuYW1lYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkLCB7IGFwcGxpY2F0aW9uQXJuLCBhcHBsaWNhdGlvbk5hbWUgfSk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb25Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGFwcGxpY2F0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8vIFJvbGUgbXVzdCBiZSBvcHRpb25hbCBmb3IgSlNJSSBjb21wYXRpYmlsaXR5XG4gIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFwcGxpY2F0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHsgcGh5c2ljYWxOYW1lOiBwcm9wcy5hcHBsaWNhdGlvbk5hbWUgfSk7XG4gICAgdmFsaWRhdGVBcHBsaWNhdGlvblByb3BzKHByb3BzKTtcblxuICAgIHRoaXMucm9sZSA9IHByb3BzLnJvbGUgPz8gbmV3IGlhbS5Sb2xlKHRoaXMsICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2tpbmVzaXNhbmFseXRpY3MuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLnJvbGU7XG5cbiAgICAvLyBQZXJtaXQgbWV0cmljIHB1Ymxpc2hpbmcgdG8gQ2xvdWRXYXRjaFxuICAgIHRoaXMucm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2Nsb3Vkd2F0Y2g6UHV0TWV0cmljRGF0YSddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBjb2RlID0gcHJvcHMuY29kZS5iaW5kKHRoaXMpO1xuICAgIGNvZGUuYnVja2V0LmdyYW50UmVhZCh0aGlzKTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkFwcGxpY2F0aW9uVjIodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXBwbGljYXRpb25OYW1lOiBwcm9wcy5hcHBsaWNhdGlvbk5hbWUsXG4gICAgICBydW50aW1lRW52aXJvbm1lbnQ6IHByb3BzLnJ1bnRpbWUudmFsdWUsXG4gICAgICBzZXJ2aWNlRXhlY3V0aW9uUm9sZTogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBhcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uY29kZS5hcHBsaWNhdGlvbkNvZGVDb25maWd1cmF0aW9uUHJvcGVydHksXG4gICAgICAgIGVudmlyb25tZW50UHJvcGVydGllczogZW52aXJvbm1lbnRQcm9wZXJ0aWVzKHByb3BzLnByb3BlcnR5R3JvdXBzKSxcbiAgICAgICAgZmxpbmtBcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb246IGZsaW5rQXBwbGljYXRpb25Db25maWd1cmF0aW9uKHtcbiAgICAgICAgICBjaGVja3BvaW50aW5nRW5hYmxlZDogcHJvcHMuY2hlY2twb2ludGluZ0VuYWJsZWQsXG4gICAgICAgICAgY2hlY2twb2ludEludGVydmFsOiBwcm9wcy5jaGVja3BvaW50SW50ZXJ2YWwsXG4gICAgICAgICAgbWluUGF1c2VCZXR3ZWVuQ2hlY2twb2ludHM6IHByb3BzLm1pblBhdXNlQmV0d2VlbkNoZWNrcG9pbnRzLFxuICAgICAgICAgIGxvZ0xldmVsOiBwcm9wcy5sb2dMZXZlbCxcbiAgICAgICAgICBtZXRyaWNzTGV2ZWw6IHByb3BzLm1ldHJpY3NMZXZlbCxcbiAgICAgICAgICBhdXRvU2NhbGluZ0VuYWJsZWQ6IHByb3BzLmF1dG9TY2FsaW5nRW5hYmxlZCxcbiAgICAgICAgICBwYXJhbGxlbGlzbTogcHJvcHMucGFyYWxsZWxpc20sXG4gICAgICAgICAgcGFyYWxsZWxpc21QZXJLcHU6IHByb3BzLnBhcmFsbGVsaXNtUGVyS3B1LFxuICAgICAgICB9KSxcbiAgICAgICAgYXBwbGljYXRpb25TbmFwc2hvdENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBzbmFwc2hvdHNFbmFibGVkOiBwcm9wcy5zbmFwc2hvdHNFbmFibGVkID8/IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLnJvbGUpO1xuXG4gICAgY29uc3QgbG9nR3JvdXAgPSBwcm9wcy5sb2dHcm91cCA/PyBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnTG9nR3JvdXAnKTtcbiAgICBjb25zdCBsb2dTdHJlYW0gPSBuZXcgbG9ncy5Mb2dTdHJlYW0odGhpcywgJ0xvZ1N0cmVhbScsIHsgbG9nR3JvdXAgfSk7XG5cbiAgICAvKiBQZXJtaXQgbG9nZ2luZyAqL1xuXG4gICAgdGhpcy5yb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnbG9nczpEZXNjcmliZUxvZ0dyb3VwcyddLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIGNvcmUuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgICBzZXJ2aWNlOiAnbG9ncycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdsb2ctZ3JvdXAnLFxuICAgICAgICAgIGFybkZvcm1hdDogY29yZS5Bcm5Gb3JtYXQuQ09MT05fUkVTT1VSQ0VfTkFNRSxcbiAgICAgICAgICByZXNvdXJjZU5hbWU6ICcqJyxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRoaXMucm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2xvZ3M6RGVzY3JpYmVMb2dTdHJlYW1zJ10sXG4gICAgICByZXNvdXJjZXM6IFtsb2dHcm91cC5sb2dHcm91cEFybl0sXG4gICAgfSkpO1xuXG4gICAgY29uc3QgbG9nU3RyZWFtQXJuID0gYGFybjoke2NvcmUuQXdzLlBBUlRJVElPTn06bG9nczoke2NvcmUuQXdzLlJFR0lPTn06JHtjb3JlLkF3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6JHtsb2dHcm91cC5sb2dHcm91cE5hbWV9OmxvZy1zdHJlYW06JHtsb2dTdHJlYW0ubG9nU3RyZWFtTmFtZX1gO1xuICAgIHRoaXMucm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ2xvZ3M6UHV0TG9nRXZlbnRzJ10sXG4gICAgICByZXNvdXJjZXM6IFtsb2dTdHJlYW1Bcm5dLFxuICAgIH0pKTtcblxuICAgIG5ldyBDZm5BcHBsaWNhdGlvbkNsb3VkV2F0Y2hMb2dnaW5nT3B0aW9uVjIodGhpcywgJ0xvZ2dpbmdPcHRpb24nLCB7XG4gICAgICBhcHBsaWNhdGlvbk5hbWU6IHJlc291cmNlLnJlZixcbiAgICAgIGNsb3VkV2F0Y2hMb2dnaW5nT3B0aW9uOiB7XG4gICAgICAgIGxvZ1N0cmVhbUFybixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLnJlZik7XG4gICAgdGhpcy5hcHBsaWNhdGlvbkFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoXG4gICAgICBjb3JlLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybihhcHBsaWNhdGlvbkFybkNvbXBvbmVudHMocmVzb3VyY2UucmVmKSksXG4gICAgICBhcHBsaWNhdGlvbkFybkNvbXBvbmVudHModGhpcy5waHlzaWNhbE5hbWUpLFxuICAgICk7XG5cbiAgICByZXNvdXJjZS5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSwge1xuICAgICAgZGVmYXVsdDogY29yZS5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gYXBwbGljYXRpb25Bcm5Db21wb25lbnRzKHJlc291cmNlTmFtZTogc3RyaW5nKTogY29yZS5Bcm5Db21wb25lbnRzIHtcbiAgcmV0dXJuIHtcbiAgICBzZXJ2aWNlOiAna2luZXNpc2FuYWx5dGljcycsXG4gICAgcmVzb3VyY2U6ICdhcHBsaWNhdGlvbicsXG4gICAgcmVzb3VyY2VOYW1lLFxuICB9O1xufVxuIl19