import * as iam from '../../aws-iam';
import * as logs from '../../aws-logs';
import * as core from '../../core';
import { Construct } from 'constructs';
import { ApplicationCode } from './application-code';
import { LogLevel, MetricsLevel, PropertyGroups, Runtime } from './types';
/**
 * An interface expressing the public properties on both an imported and
 * CDK-created Flink application.
 */
export interface IApplication extends core.IResource, iam.IGrantable {
    /**
     * The application ARN.
     *
     * @attribute
     */
    readonly applicationArn: string;
    /**
     * The name of the Flink application.
     *
     * @attribute
     */
    readonly applicationName: string;
    /**
     * The application IAM role.
     */
    readonly role?: iam.IRole;
    /**
     * Convenience method for adding a policy statement to the application role.
     */
    addToRolePolicy(policyStatement: iam.PolicyStatement): boolean;
}
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
declare abstract class ApplicationBase extends core.Resource implements IApplication {
    abstract readonly applicationArn: string;
    abstract readonly applicationName: string;
    abstract readonly role?: iam.IRole;
    abstract readonly grantPrincipal: iam.IPrincipal;
    /** Implement the convenience {@link IApplication.addToPrincipalPolicy} method. */
    addToRolePolicy(policyStatement: iam.PolicyStatement): boolean;
}
/**
 * Props for creating an Application construct.
 */
export interface ApplicationProps {
    /**
     * A name for your Application that is unique to an AWS account.
     *
     * @default - CloudFormation-generated name
     */
    readonly applicationName?: string;
    /**
     * The Flink version to use for this application.
     */
    readonly runtime: Runtime;
    /**
     * The Flink code asset to run.
     */
    readonly code: ApplicationCode;
    /**
     * Whether checkpointing is enabled while your application runs.
     *
     * @default true
     */
    readonly checkpointingEnabled?: boolean;
    /**
     * The interval between checkpoints.
     *
     * @default 1 minute
     */
    readonly checkpointInterval?: core.Duration;
    /**
     * The minimum amount of time in to wait after a checkpoint finishes to start
     * a new checkpoint.
     *
     * @default 5 seconds
     */
    readonly minPauseBetweenCheckpoints?: core.Duration;
    /**
     * The level of log verbosity from the Flink application.
     *
     * @default FlinkLogLevel.INFO
     */
    readonly logLevel?: LogLevel;
    /**
     * Describes the granularity of the CloudWatch metrics for an application.
     * Use caution with Parallelism level metrics. Parallelism granularity logs
     * metrics for each parallel thread and can quickly become expensive when
     * parallelism is high (e.g. > 64).
     *
     * @default MetricsLevel.APPLICATION
     */
    readonly metricsLevel?: MetricsLevel;
    /**
     * Whether the Kinesis Data Analytics service can increase the parallelism of
     * the application in response to resource usage.
     *
     * @default true
     */
    readonly autoScalingEnabled?: boolean;
    /**
     * The initial parallelism for the application. Kinesis Data Analytics can
     * stop the app, increase the parallelism, and start the app again if
     * autoScalingEnabled is true (the default value).
     *
     * @default 1
     */
    readonly parallelism?: number;
    /**
     * The Flink parallelism allowed per Kinesis Processing Unit (KPU).
     *
     * @default 1
     */
    readonly parallelismPerKpu?: number;
    /**
     * Determines if Flink snapshots are enabled.
     *
     * @default true
     */
    readonly snapshotsEnabled?: boolean;
    /**
     * Configuration PropertyGroups. You can use these property groups to pass
     * arbitrary runtime configuration values to your Flink app.
     *
     * @default No property group configuration provided to the Flink app
     */
    readonly propertyGroups?: PropertyGroups;
    /**
     * A role to use to grant permissions to your application. Prefer omitting
     * this property and using the default role.
     *
     * @default - a new Role will be created
     */
    readonly role?: iam.IRole;
    /**
     * Provide a RemovalPolicy to override the default.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly removalPolicy?: core.RemovalPolicy;
    /**
     * The log group to send log entries to.
     *
     * @default CDK's default LogGroup
     */
    readonly logGroup?: logs.ILogGroup;
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
export declare class Application extends ApplicationBase {
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope: Construct, id: string, applicationName: string): IApplication;
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope: Construct, id: string, applicationArn: string): IApplication;
    readonly applicationArn: string;
    readonly applicationName: string;
    readonly role?: iam.IRole;
    readonly grantPrincipal: iam.IPrincipal;
    constructor(scope: Construct, id: string, props: ApplicationProps);
}
export {};
