"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenIdConnectProvider = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("../../core");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class OpenIdConnectProvider extends core_1.Resource {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_iam_OpenIdConnectProviderProps(props);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
        this.openIdConnectProviderIssuer = core_1.Arn.extractResourceName(this.openIdConnectProviderArn, 'oidc-provider');
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        const resourceName = core_1.Arn.extractResourceName(openIdConnectProviderArn, 'oidc-provider');
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
                this.openIdConnectProviderIssuer = resourceName;
            }
        }
        return new Import(scope, id);
    }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
_a = JSII_RTTI_SYMBOL_1;
OpenIdConnectProvider[_a] = { fqn: "monocdk.aws_iam.OpenIdConnectProvider", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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