import * as ec2 from '../../aws-ec2';
import * as cdk from '../../core';
import * as constructs from 'constructs';
/**
 * The type of the glue connection
 *
 * If you need to use a connection type that doesn't exist as a static member, you
 * can instantiate a `ConnectionType` object, e.g: `new ConnectionType('NEW_TYPE')`.
 */
export declare class ConnectionType {
    /**
     * Designates a connection to a database through Java Database Connectivity (JDBC).
     */
    static readonly JDBC: ConnectionType;
    /**
     * Designates a connection to an Apache Kafka streaming platform.
     */
    static readonly KAFKA: ConnectionType;
    /**
     * Designates a connection to a MongoDB document database.
     */
    static readonly MONGODB: ConnectionType;
    /**
     * Designates a network connection to a data source within an Amazon Virtual Private Cloud environment (Amazon VPC).
     */
    static readonly NETWORK: ConnectionType;
    /**
     * The name of this ConnectionType, as expected by Connection resource.
     */
    readonly name: string;
    constructor(name: string);
    /**
     * The connection type name as expected by Connection resource.
     */
    toString(): string;
}
/**
 * Interface representing a created or an imported {@link Connection}
 */
export interface IConnection extends cdk.IResource {
    /**
     * The name of the connection
     * @attribute
     */
    readonly connectionName: string;
    /**
     * The ARN of the connection
     * @attribute
     */
    readonly connectionArn: string;
}
/**
 * Base Connection Options
 */
export interface ConnectionOptions {
    /**
     * The name of the connection
     * @default cloudformation generated name
     */
    readonly connectionName?: string;
    /**
     * The description of the connection.
     * @default no description
     */
    readonly description?: string;
    /**
     *  Key-Value pairs that define parameters for the connection.
     *  @default empty properties
     *  @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-connect.html
     */
    readonly properties?: {
        [key: string]: string;
    };
    /**
     * A list of criteria that can be used in selecting this connection.
     * This is useful for filtering the results of https://awscli.amazonaws.com/v2/documentation/api/latest/reference/glue/get-connections.html
     * @default no match criteria
     */
    readonly matchCriteria?: string[];
    /**
     * The list of security groups needed to successfully make this connection e.g. to successfully connect to VPC.
     * @default no security group
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The VPC subnet to connect to resources within a VPC. See more at https://docs.aws.amazon.com/glue/latest/dg/start-connecting.html.
     * @default no subnet
     */
    readonly subnet?: ec2.ISubnet;
}
/**
 * Construction properties for {@link Connection}
 */
export interface ConnectionProps extends ConnectionOptions {
    /**
     * The type of the connection
     */
    readonly type: ConnectionType;
}
/**
 * An AWS Glue connection to a data source.
 */
export declare class Connection extends cdk.Resource implements IConnection {
    /**
     * Creates a Connection construct that represents an external connection.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param connectionArn arn of external connection.
     */
    static fromConnectionArn(scope: constructs.Construct, id: string, connectionArn: string): IConnection;
    /**
     * Creates a Connection construct that represents an external connection.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param connectionName name of external connection.
     */
    static fromConnectionName(scope: constructs.Construct, id: string, connectionName: string): IConnection;
    private static buildConnectionArn;
    /**
     * The ARN of the connection
     */
    readonly connectionArn: string;
    /**
     * The name of the connection
     */
    readonly connectionName: string;
    private readonly properties;
    constructor(scope: constructs.Construct, id: string, props: ConnectionProps);
    /**
     * Add additional connection parameters
     * @param key parameter key
     * @param value parameter value
     */
    addProperty(key: string, value: string): void;
}
