"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetCode = exports.S3Code = exports.Code = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const fs = require("fs");
const s3assets = require("../../aws-s3-assets");
const cdk = require("../../core");
/**
 * Represents a Glue Job's Code assets (an asset can be a scripts, a jar, a python file or any other file).
 */
class Code {
    /**
     * Job code as an S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     */
    static fromBucket(bucket, key) {
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        return new S3Code(bucket, key);
    }
    /**
     * Job code from a local disk path.
     *
     * @param path code file (not a directory).
     */
    static fromAsset(path, options) {
        jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetOptions(options);
        return new AssetCode(path, options);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "monocdk.aws_glue.Code", version: "1.149.0" };
/**
 * Glue job Code from an S3 bucket.
 */
class S3Code extends Code {
    constructor(bucket, key) {
        super();
        this.bucket = bucket;
        this.key = key;
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
    }
    bind(_scope, grantable) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantable);
        this.bucket.grantRead(grantable, this.key);
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
            },
        };
    }
}
exports.S3Code = S3Code;
_b = JSII_RTTI_SYMBOL_1;
S3Code[_b] = { fqn: "monocdk.aws_glue.S3Code", version: "1.149.0" };
/**
 * Job Code from a local file.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the Code file.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetOptions(options);
        if (fs.lstatSync(this.path).isDirectory()) {
            throw new Error(`Code path ${this.path} is a directory. Only files are supported`);
        }
    }
    bind(scope, grantable) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantable);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3assets.Asset(scope, `Code${this.hashcode(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Code instance for every stack.');
        }
        this.asset.grantRead(grantable);
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Hash a string
     */
    hashcode(s) {
        const hash = crypto.createHash('md5');
        hash.update(s);
        return hash.digest('hex');
    }
    ;
}
exports.AssetCode = AssetCode;
_c = JSII_RTTI_SYMBOL_1;
AssetCode[_c] = { fqn: "monocdk.aws_glue.AssetCode", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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