"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * Schedule for scheduled event rules
 */
class Schedule {
    constructor() { }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that EventBridge will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        jsiiDeprecationWarnings.monocdk_Duration(duration);
        if (duration.isUnresolved()) {
            const validDurationUnit = ['minute', 'minutes', 'hour', 'hours', 'day', 'days'];
            if (validDurationUnit.indexOf(duration.unitLabel()) === -1) {
                throw new Error("Allowed units for scheduling are: 'minute', 'minutes', 'hour', 'hours', 'day', 'days'");
            }
            return new LiteralSchedule(`rate(${duration.formatTokenToNumber()})`);
        }
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        jsiiDeprecationWarnings.monocdk_aws_events_CronOptions(options);
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new class extends Schedule {
            constructor() {
                super(...arguments);
                this.expressionString = `cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`;
            }
            _bind(scope) {
                if (!options.minute) {
                    core_1.Annotations.of(scope).addWarning('cron: If you don\'t pass \'minute\', by default the event runs every minute. Pass \'minute: \'*\'\' if that\'s what you intend, or \'minute: 0\' to run once per hour instead.');
                }
                return new LiteralSchedule(this.expressionString);
            }
        };
    }
}
exports.Schedule = Schedule;
_a = JSII_RTTI_SYMBOL_1;
Schedule[_a] = { fqn: "monocdk.aws_events.Schedule", version: "1.149.0" };
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
    _bind() { }
}
function fallback(x, def) {
    return x !== null && x !== void 0 ? x : def;
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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