"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const ecs = require("../../aws-ecs");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const cdk = require("../../core");
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        var _b;
        this.props = props;
        jsiiDeprecationWarnings.monocdk_aws_events_targets_EcsTaskProps(props);
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = (_b = props.taskCount) !== null && _b !== void 0 ? _b : 1;
        this.platformVersion = props.platformVersion;
        if (props.role) {
            const role = props.role;
            this.createEventRolePolicyStatements().forEach(role.addToPrincipalPolicy.bind(role));
            this.role = role;
        }
        else {
            this.role = util_1.singletonEventRole(this.taskDefinition, this.createEventRolePolicyStatements());
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                cdk.Annotations.of(this.taskDefinition).addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        if (!cdk.Construct.isConstruct(this.taskDefinition)) {
            throw new Error('Cannot create a security group for ECS task. ' +
                'The task definition in ECS task is not a Construct. ' +
                'Please pass a taskDefinition as a Construct in EcsTaskProps.');
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        jsiiDeprecationWarnings.monocdk_aws_events_IRule(_rule);
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                platformVersion: this.platformVersion,
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
_a = JSII_RTTI_SYMBOL_1;
EcsTask[_a] = { fqn: "monocdk.aws_events_targets.EcsTask", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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