"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapTagMapToCxschema = exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== null && x !== void 0 ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
/**
 * Helper to map a map of tags to cxschema tag format.
 * @internal
 */
function mapTagMapToCxschema(tagMap) {
    return Object.entries(tagMap)
        .map(([key, value]) => ({ key, value }));
}
exports.mapTagMapToCxschema = mapTagMapToCxschema;
//# sourceMappingURL=data:application/json;base64,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