"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2");
const iam = require("../../../aws-iam");
const aws_iam_1 = require("../../../aws-iam");
const cxschema = require("../../../cloud-assembly-schema");
const core_1 = require("../../../core");
const cxapi = require("../../../cx-api");
const region_info_1 = require("../../../region-info");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_BaseLoadBalancerProps(baseProps);
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _b;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_b = options.userOptions.loadBalancerTags) !== null && _b !== void 0 ? _b : {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to look up a load balancer must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_PROVIDER,
            props: {
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                ipAddressType: cxapi.LoadBalancerIpAddressType.DUAL_STACK,
                loadBalancerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188`,
                loadBalancerCanonicalHostedZoneId: 'Z3DZXE0EXAMPLE',
                loadBalancerDnsName: 'my-load-balancer-1234567890.us-west-2.elb.amazonaws.com',
                securityGroupIds: ['sg-1234'],
                vpcId: 'vpc-12345',
            },
        }).value;
        return props;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        prefix = prefix || '';
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
    validate() {
        const ret = super.validate();
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-loadbalancer.html#cfn-elasticloadbalancingv2-loadbalancer-name
        const loadBalancerName = this.physicalName;
        if (!core_1.Token.isUnresolved(loadBalancerName) && loadBalancerName !== undefined) {
            if (loadBalancerName.length > 32) {
                ret.push(`Load balancer name: "${loadBalancerName}" can have a maximum of 32 characters.`);
            }
            if (loadBalancerName.startsWith('internal-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin with "internal-".`);
            }
            if (loadBalancerName.startsWith('-') || loadBalancerName.endsWith('-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin or end with a hyphen.`);
            }
            if (!/^[0-9a-z-]+$/i.test(loadBalancerName)) {
                ret.push(`Load balancer name: "${loadBalancerName}" must contain only alphanumeric characters or hyphens.`);
            }
        }
        return ret;
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
BaseLoadBalancer[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.BaseLoadBalancer", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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