"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../../aws-cloudwatch");
const cdk = require("../../../core");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_NetworkTargetGroupProps(props);
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        if (props.preserveClientIp !== undefined) {
            this.setAttribute('preserve_client_ip.enabled', props.preserveClientIp ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_TargetGroupAttributes(attrs);
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing listener
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        jsiiDeprecationWarnings.print("monocdk.aws_elasticloadbalancingv2.NetworkTargetGroup#import", "Use `fromTargetGroupAttributes` instead");
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_TargetGroupImportProps(props);
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkLoadBalancerTarget(targets);
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkListener(listener);
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     */
    metricUnHealthyHostCount(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensionsMap: { LoadBalancer: this.firstLoadBalancerFullName, TargetGroup: this.targetGroupFullName },
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
NetworkTargetGroup[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.NetworkTargetGroup", version: "1.149.0" };
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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