"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListener = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../../cloud-assembly-schema");
const core_1 = require("../../../core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const util_1 = require("../shared/util");
const network_listener_action_1 = require("./network-listener-action");
const network_target_group_1 = require("./network-target-group");
/**
 * Define a Network Listener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class NetworkListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_NetworkListenerProps(props);
        const certs = props.certificates || [];
        const proto = props.protocol || (certs.length > 0 ? enums_1.Protocol.TLS : enums_1.Protocol.TCP);
        util_1.validateNetworkProtocol(proto);
        if (proto === enums_1.Protocol.TLS && certs.filter(v => v != null).length === 0) {
            throw new Error('When the protocol is set to TLS, you must specify certificates');
        }
        if (proto !== enums_1.Protocol.TLS && certs.length > 0) {
            throw new Error('Protocol must be TLS when certificates have been specified');
        }
        if (proto !== enums_1.Protocol.TLS && props.alpnPolicy) {
            throw new Error('Protocol must be TLS when alpnPolicy have been specified');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            protocol: proto,
            port: props.port,
            sslPolicy: props.sslPolicy,
            certificates: props.certificates,
            alpnPolicy: props.alpnPolicy ? [props.alpnPolicy] : undefined,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = proto;
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(props.defaultTargetGroups));
        }
    }
    /**
     * Looks up a network listener
     */
    static fromLookup(scope, id, options) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_NetworkListenerLookupOptions(options);
        let listenerProtocol;
        if (options.listenerProtocol) {
            util_1.validateNetworkProtocol(options.listenerProtocol);
            switch (options.listenerProtocol) {
                case enums_1.Protocol.TCP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP;
                    break;
                case enums_1.Protocol.UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.UDP;
                    break;
                case enums_1.Protocol.TCP_UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP_UDP;
                    break;
                case enums_1.Protocol.TLS:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TLS;
                    break;
            }
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            listenerProtocol: listenerProtocol,
            loadBalancerType: cxschema.LoadBalancerType.NETWORK,
        });
        class LookedUp extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = props.listenerArn;
            }
        }
        return new LookedUp(scope, id);
    }
    /**
     * Import an existing listener
     */
    static fromNetworkListenerArn(scope, id, networkListenerArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = networkListenerArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     */
    addTargetGroups(_id, ...targetGroups) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroups);
        this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(targetGroups));
    }
    /**
     * Perform the given Action on incoming requests
     *
     * This allows full control of the default Action of the load balancer,
     * including weighted forwarding. See the `NetworkListenerAction` class for
     * all options.
     */
    addAction(_id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddNetworkActionProps(props);
        this.setDefaultAction(props.action);
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates a NetworkTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        var _b;
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddNetworkTargetsProps(props);
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or imported Load Balancer with specified VPC; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new network_target_group_1.NetworkTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: (_b = props.protocol) !== null && _b !== void 0 ? _b : this.protocol,
            proxyProtocolV2: props.proxyProtocolV2,
            preserveClientIp: props.preserveClientIp,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, group);
        return group;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.NetworkListener = NetworkListener;
_a = JSII_RTTI_SYMBOL_1;
NetworkListener[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.NetworkListener", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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