"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2");
const cxschema = require("../../../cloud-assembly-schema");
const core_1 = require("../../../core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_ApplicationListenerProps(props);
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.any({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * Look up an ApplicationListener.
     */
    static fromLookup(scope, id, options) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_ApplicationListenerLookupOptions(options);
        if (core_1.Token.isUnresolved(options.listenerArn)) {
            throw new Error('All arguments to look up a load balancer listener must be concrete (no Tokens)');
        }
        let listenerProtocol;
        switch (options.listenerProtocol) {
            case enums_1.ApplicationProtocol.HTTP:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTP;
                break;
            case enums_1.ApplicationProtocol.HTTPS:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTPS;
                break;
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            loadBalancerType: cxschema.LoadBalancerType.APPLICATION,
            listenerArn: options.listenerArn,
            listenerProtocol,
        });
        return new LookedUpApplicationListener(scope, id, props);
    }
    /**
     * Import an existing listener
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_ApplicationListenerAttributes(attrs);
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        jsiiDeprecationWarnings.print("monocdk.aws_elasticloadbalancingv2.ApplicationListener#addCertificateArns", "Use `addCertificates` instead.");
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        // Only one certificate can be specified per resource, even though
        // `certificates` is of type Array
        for (let i = 0; i < additionalCerts.length; i++) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, `${id}${i + 1}`, {
                listener: this,
                certificates: [additionalCerts[i]],
            });
        }
    }
    /**
     * Perform the given default action on incoming requests
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     */
    addAction(id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddApplicationActionProps(props);
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     */
    addTargetGroups(id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddApplicationTargetGroupsProps(props);
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddApplicationTargetsProps(props);
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            vpc: this.loadBalancer.vpc,
            ...props,
        });
        this.addTargetGroups(id, {
            targetGroups: [group],
            ...props,
        });
        return group;
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        jsiiDeprecationWarnings.print("monocdk.aws_elasticloadbalancingv2.ApplicationListener#addFixedResponse", "Use `addAction()` instead");
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddFixedResponseProps(props);
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        jsiiDeprecationWarnings.print("monocdk.aws_elasticloadbalancingv2.ApplicationListener#addRedirectResponse", "Use `addAction()` instead");
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddRedirectResponseProps(props);
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_IConnectable(connectable);
        jsiiDeprecationWarnings.monocdk_aws_ec2_Port(portRange);
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
_a = JSII_RTTI_SYMBOL_1;
ApplicationListener[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.ApplicationListener", version: "1.149.0" };
class ExternalApplicationListener extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
    /**
     * Add one or more certificates to this listener.
     * @deprecated use `addCertificates()`
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificates(id, certificates) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificates,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                priority: props.priority,
                ...props,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
}
/**
 * An imported application listener.
 */
class ImportedApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
}
class LookedUpApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(props.listenerPort),
        });
        for (const securityGroupId of props.securityGroupIds) {
            const securityGroup = ec2.SecurityGroup.fromLookupById(this, `SecurityGroup-${securityGroupId}`, securityGroupId);
            this.connections.addSecurityGroup(securityGroup);
        }
    }
}
function checkAddRuleProps(props) {
    var _b;
    const conditionsCount = ((_b = props.conditions) === null || _b === void 0 ? void 0 : _b.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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