"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cognito = require("../../aws-cognito");
const ec2 = require("../../aws-ec2");
const elbv2 = require("../../aws-elasticloadbalancingv2");
const core_1 = require("../../core");
const actions = require("../lib");
class CognitoStack extends core_1.Stack {
    /// !hide
    constructor(scope, id) {
        var _a;
        super(scope, id);
        const vpc = new ec2.Vpc(this, 'Stack', {
            maxAzs: 2,
        });
        const certificate = {
            certificateArn: (_a = process.env.SELF_SIGNED_CERT_ARN) !== null && _a !== void 0 ? _a : '',
        };
        /// !show
        const lb = new elbv2.ApplicationLoadBalancer(this, 'LB', {
            vpc,
            internetFacing: true,
        });
        const userPool = new cognito.UserPool(this, 'UserPool');
        const userPoolClient = new cognito.UserPoolClient(this, 'Client', {
            userPool,
            // Required minimal configuration for use with an ELB
            generateSecret: true,
            authFlows: {
                userPassword: true,
            },
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: [cognito.OAuthScope.EMAIL],
                callbackUrls: [
                    `https://${lb.loadBalancerDnsName}/oauth2/idpresponse`,
                ],
            },
        });
        const cfnClient = userPoolClient.node.defaultChild;
        cfnClient.addPropertyOverride('RefreshTokenValidity', 1);
        cfnClient.addPropertyOverride('SupportedIdentityProviders', ['COGNITO']);
        const userPoolDomain = new cognito.UserPoolDomain(this, 'Domain', {
            userPool,
            cognitoDomain: {
                domainPrefix: 'test-cdk-prefix',
            },
        });
        lb.addListener('Listener', {
            port: 443,
            certificates: [certificate],
            defaultAction: new actions.AuthenticateCognitoAction({
                userPool,
                userPoolClient,
                userPoolDomain,
                next: elbv2.ListenerAction.fixedResponse(200, {
                    contentType: 'text/plain',
                    messageBody: 'Authenticated',
                }),
            }),
        });
        new core_1.CfnOutput(this, 'DNS', {
            value: lb.loadBalancerDnsName,
        });
    }
}
const app = new core_1.App();
new CognitoStack(app, 'integ-cognito');
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuY29nbml0by5saXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5jb2duaXRvLmxpdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDZDQUE2QztBQUM3QyxxQ0FBcUM7QUFDckMsMERBQTBEO0FBQzFELHFDQUFtRDtBQUVuRCxrQ0FBa0M7QUFFbEMsTUFBTSxZQUFhLFNBQVEsWUFBSztJQUU5QixTQUFTO0lBQ1QsWUFBWSxLQUFnQixFQUFFLEVBQVU7O1FBQ3RDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDckMsTUFBTSxFQUFFLENBQUM7U0FDVixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBK0I7WUFDOUMsY0FBYyxRQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsb0JBQW9CLG1DQUFJLEVBQUU7U0FDdkQsQ0FBQztRQUVGLFNBQVM7UUFDVCxNQUFNLEVBQUUsR0FBRyxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFO1lBQ3ZELEdBQUc7WUFDSCxjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLE9BQU8sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3hELE1BQU0sY0FBYyxHQUFHLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ2hFLFFBQVE7WUFFUixxREFBcUQ7WUFDckQsY0FBYyxFQUFFLElBQUk7WUFDcEIsU0FBUyxFQUFFO2dCQUNULFlBQVksRUFBRSxJQUFJO2FBQ25CO1lBQ0QsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRTtvQkFDTCxzQkFBc0IsRUFBRSxJQUFJO2lCQUM3QjtnQkFDRCxNQUFNLEVBQUUsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztnQkFDbEMsWUFBWSxFQUFFO29CQUNaLFdBQVcsRUFBRSxDQUFDLG1CQUFtQixxQkFBcUI7aUJBQ3ZEO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsSUFBSSxDQUFDLFlBQXlDLENBQUM7UUFDaEYsU0FBUyxDQUFDLG1CQUFtQixDQUFDLHNCQUFzQixFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3pELFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7UUFFekUsTUFBTSxjQUFjLEdBQUcsSUFBSSxPQUFPLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7WUFDaEUsUUFBUTtZQUNSLGFBQWEsRUFBRTtnQkFDYixZQUFZLEVBQUUsaUJBQWlCO2FBQ2hDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsRUFBRSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUU7WUFDekIsSUFBSSxFQUFFLEdBQUc7WUFDVCxZQUFZLEVBQUUsQ0FBQyxXQUFXLENBQUM7WUFDM0IsYUFBYSxFQUFFLElBQUksT0FBTyxDQUFDLHlCQUF5QixDQUFDO2dCQUNuRCxRQUFRO2dCQUNSLGNBQWM7Z0JBQ2QsY0FBYztnQkFDZCxJQUFJLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFO29CQUM1QyxXQUFXLEVBQUUsWUFBWTtvQkFDekIsV0FBVyxFQUFFLGVBQWU7aUJBQzdCLENBQUM7YUFDSCxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsSUFBSSxnQkFBUyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7WUFDekIsS0FBSyxFQUFFLEVBQUUsQ0FBQyxtQkFBbUI7U0FDOUIsQ0FBQyxDQUFDO0tBQ0o7Q0FDRjtBQUVELE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7QUFDdEIsSUFBSSxZQUFZLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0FBQ3ZDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZ25pdG8gZnJvbSAnLi4vLi4vYXdzLWNvZ25pdG8nO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJy4uLy4uL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWxidjIgZnJvbSAnLi4vLi4vYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0IHsgQXBwLCBDZm5PdXRwdXQsIFN0YWNrIH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCAqIGFzIGFjdGlvbnMgZnJvbSAnLi4vbGliJztcblxuY2xhc3MgQ29nbml0b1N0YWNrIGV4dGVuZHMgU3RhY2sge1xuXG4gIC8vLyAhaGlkZVxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHZwYyA9IG5ldyBlYzIuVnBjKHRoaXMsICdTdGFjaycsIHtcbiAgICAgIG1heEF6czogMixcbiAgICB9KTtcblxuICAgIGNvbnN0IGNlcnRpZmljYXRlOiBlbGJ2Mi5JTGlzdGVuZXJDZXJ0aWZpY2F0ZSA9IHtcbiAgICAgIGNlcnRpZmljYXRlQXJuOiBwcm9jZXNzLmVudi5TRUxGX1NJR05FRF9DRVJUX0FSTiA/PyAnJyxcbiAgICB9O1xuXG4gICAgLy8vICFzaG93XG4gICAgY29uc3QgbGIgPSBuZXcgZWxidjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgJ0xCJywge1xuICAgICAgdnBjLFxuICAgICAgaW50ZXJuZXRGYWNpbmc6IHRydWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCB1c2VyUG9vbCA9IG5ldyBjb2duaXRvLlVzZXJQb29sKHRoaXMsICdVc2VyUG9vbCcpO1xuICAgIGNvbnN0IHVzZXJQb29sQ2xpZW50ID0gbmV3IGNvZ25pdG8uVXNlclBvb2xDbGllbnQodGhpcywgJ0NsaWVudCcsIHtcbiAgICAgIHVzZXJQb29sLFxuXG4gICAgICAvLyBSZXF1aXJlZCBtaW5pbWFsIGNvbmZpZ3VyYXRpb24gZm9yIHVzZSB3aXRoIGFuIEVMQlxuICAgICAgZ2VuZXJhdGVTZWNyZXQ6IHRydWUsXG4gICAgICBhdXRoRmxvd3M6IHtcbiAgICAgICAgdXNlclBhc3N3b3JkOiB0cnVlLFxuICAgICAgfSxcbiAgICAgIG9BdXRoOiB7XG4gICAgICAgIGZsb3dzOiB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvbkNvZGVHcmFudDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2NvcGVzOiBbY29nbml0by5PQXV0aFNjb3BlLkVNQUlMXSxcbiAgICAgICAgY2FsbGJhY2tVcmxzOiBbXG4gICAgICAgICAgYGh0dHBzOi8vJHtsYi5sb2FkQmFsYW5jZXJEbnNOYW1lfS9vYXV0aDIvaWRwcmVzcG9uc2VgLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBjZm5DbGllbnQgPSB1c2VyUG9vbENsaWVudC5ub2RlLmRlZmF1bHRDaGlsZCBhcyBjb2duaXRvLkNmblVzZXJQb29sQ2xpZW50O1xuICAgIGNmbkNsaWVudC5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdSZWZyZXNoVG9rZW5WYWxpZGl0eScsIDEpO1xuICAgIGNmbkNsaWVudC5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdTdXBwb3J0ZWRJZGVudGl0eVByb3ZpZGVycycsIFsnQ09HTklUTyddKTtcblxuICAgIGNvbnN0IHVzZXJQb29sRG9tYWluID0gbmV3IGNvZ25pdG8uVXNlclBvb2xEb21haW4odGhpcywgJ0RvbWFpbicsIHtcbiAgICAgIHVzZXJQb29sLFxuICAgICAgY29nbml0b0RvbWFpbjoge1xuICAgICAgICBkb21haW5QcmVmaXg6ICd0ZXN0LWNkay1wcmVmaXgnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGxiLmFkZExpc3RlbmVyKCdMaXN0ZW5lcicsIHtcbiAgICAgIHBvcnQ6IDQ0MyxcbiAgICAgIGNlcnRpZmljYXRlczogW2NlcnRpZmljYXRlXSxcbiAgICAgIGRlZmF1bHRBY3Rpb246IG5ldyBhY3Rpb25zLkF1dGhlbnRpY2F0ZUNvZ25pdG9BY3Rpb24oe1xuICAgICAgICB1c2VyUG9vbCxcbiAgICAgICAgdXNlclBvb2xDbGllbnQsXG4gICAgICAgIHVzZXJQb29sRG9tYWluLFxuICAgICAgICBuZXh0OiBlbGJ2Mi5MaXN0ZW5lckFjdGlvbi5maXhlZFJlc3BvbnNlKDIwMCwge1xuICAgICAgICAgIGNvbnRlbnRUeXBlOiAndGV4dC9wbGFpbicsXG4gICAgICAgICAgbWVzc2FnZUJvZHk6ICdBdXRoZW50aWNhdGVkJyxcbiAgICAgICAgfSksXG4gICAgICB9KSxcbiAgICB9KTtcblxuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ0ROUycsIHtcbiAgICAgIHZhbHVlOiBsYi5sb2FkQmFsYW5jZXJEbnNOYW1lLFxuICAgIH0pO1xuICB9XG59XG5cbmNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbm5ldyBDb2duaXRvU3RhY2soYXBwLCAnaW50ZWctY29nbml0bycpO1xuYXBwLnN5bnRoKCk7Il19