"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LifecycleLabel = exports.renderBottlerocketUserData = exports.renderAmazonLinuxUserData = void 0;
const core_1 = require("../../core");
// eslint-disable-next-line max-len
function renderAmazonLinuxUserData(cluster, autoScalingGroup, options = {}) {
    var _a;
    const stack = core_1.Stack.of(autoScalingGroup);
    // determine logical id of ASG so we can signal cloudformation
    const cfn = autoScalingGroup.node.defaultChild;
    const asgLogicalId = cfn.logicalId;
    const extraArgs = new Array();
    try {
        const clusterEndpoint = cluster.clusterEndpoint;
        const clusterCertificateAuthorityData = cluster.clusterCertificateAuthorityData;
        extraArgs.push(`--apiserver-endpoint '${clusterEndpoint}'`);
        extraArgs.push(`--b64-cluster-ca '${clusterCertificateAuthorityData}'`);
    }
    catch (e) {
        /**
         * Errors are ignored here.
         * apiserver-endpoint and b64-cluster-ca arguments are added in #12659 to make nodes join the cluster faster.
         * As these are not necessary arguments, we don't need to pass these arguments when they don't exist.
         *
         * @see https://github.com/aws/aws-cdk/pull/12659
         */
    }
    extraArgs.push(`--use-max-pods ${(_a = options.useMaxPods) !== null && _a !== void 0 ? _a : true}`);
    if (options.awsApiRetryAttempts) {
        extraArgs.push(`--aws-api-retry-attempts ${options.awsApiRetryAttempts}`);
    }
    if (options.enableDockerBridge) {
        extraArgs.push('--enable-docker-bridge true');
    }
    if (options.dockerConfigJson) {
        extraArgs.push(`--docker-config-json '${options.dockerConfigJson}'`);
    }
    if (options.dnsClusterIp) {
        extraArgs.push(`--dns-cluster-ip ${options.dnsClusterIp}`);
    }
    if (options.additionalArgs) {
        extraArgs.push(options.additionalArgs);
    }
    const commandLineSuffix = extraArgs.join(' ');
    const kubeletExtraArgsSuffix = options.kubeletExtraArgs || '';
    // determine lifecycle label based on whether the ASG has a spot price.
    const lifecycleLabel = autoScalingGroup.spotPrice ? LifecycleLabel.SPOT : LifecycleLabel.ON_DEMAND;
    const withTaints = autoScalingGroup.spotPrice ? '--register-with-taints=spotInstance=true:PreferNoSchedule' : '';
    const kubeletExtraArgs = `--node-labels lifecycle=${lifecycleLabel} ${withTaints} ${kubeletExtraArgsSuffix}`.trim();
    return [
        'set -o xtrace',
        `/etc/eks/bootstrap.sh ${cluster.clusterName} --kubelet-extra-args "${kubeletExtraArgs}" ${commandLineSuffix}`.trim(),
        `/opt/aws/bin/cfn-signal --exit-code $? --stack ${stack.stackName} --resource ${asgLogicalId} --region ${stack.region}`,
    ];
}
exports.renderAmazonLinuxUserData = renderAmazonLinuxUserData;
function renderBottlerocketUserData(cluster) {
    return [
        '[settings.kubernetes]',
        `api-server="${cluster.clusterEndpoint}"`,
        `cluster-certificate="${cluster.clusterCertificateAuthorityData}"`,
        `cluster-name="${cluster.clusterName}"`,
    ];
}
exports.renderBottlerocketUserData = renderBottlerocketUserData;
/**
 * The lifecycle label for node selector
 */
var LifecycleLabel;
(function (LifecycleLabel) {
    /**
     * on-demand instances
     */
    LifecycleLabel["ON_DEMAND"] = "OnDemand";
    /**
     * spot instances
     */
    LifecycleLabel["SPOT"] = "Ec2Spot";
})(LifecycleLabel = exports.LifecycleLabel || (exports.LifecycleLabel = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1kYXRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1kYXRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHFDQUFtQztBQUduQyxtQ0FBbUM7QUFDbkMsU0FBZ0IseUJBQXlCLENBQUMsT0FBaUIsRUFBRSxnQkFBOEMsRUFBRSxVQUE0QixFQUFFOztJQUV6SSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFFekMsOERBQThEO0lBQzlELE1BQU0sR0FBRyxHQUFHLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUErQyxDQUFDO0lBQ2xGLE1BQU0sWUFBWSxHQUFHLEdBQUcsQ0FBQyxTQUFTLENBQUM7SUFFbkMsTUFBTSxTQUFTLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztJQUV0QyxJQUFJO1FBQ0YsTUFBTSxlQUFlLEdBQUcsT0FBTyxDQUFDLGVBQWUsQ0FBQztRQUNoRCxNQUFNLCtCQUErQixHQUNuQyxPQUFPLENBQUMsK0JBQStCLENBQUM7UUFDMUMsU0FBUyxDQUFDLElBQUksQ0FBQyx5QkFBeUIsZUFBZSxHQUFHLENBQUMsQ0FBQztRQUM1RCxTQUFTLENBQUMsSUFBSSxDQUFDLHFCQUFxQiwrQkFBK0IsR0FBRyxDQUFDLENBQUM7S0FDekU7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWOzs7Ozs7V0FNRztLQUNKO0lBRUQsU0FBUyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsTUFBQSxPQUFPLENBQUMsVUFBVSxtQ0FBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBRS9ELElBQUksT0FBTyxDQUFDLG1CQUFtQixFQUFFO1FBQy9CLFNBQVMsQ0FBQyxJQUFJLENBQUMsNEJBQTRCLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7S0FDM0U7SUFFRCxJQUFJLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRTtRQUM5QixTQUFTLENBQUMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLENBQUM7S0FDL0M7SUFFRCxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRTtRQUM1QixTQUFTLENBQUMsSUFBSSxDQUFDLHlCQUF5QixPQUFPLENBQUMsZ0JBQWdCLEdBQUcsQ0FBQyxDQUFDO0tBQ3RFO0lBRUQsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO1FBQ3hCLFNBQVMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO0tBQzVEO0lBRUQsSUFBSSxPQUFPLENBQUMsY0FBYyxFQUFFO1FBQzFCLFNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0tBQ3hDO0lBRUQsTUFBTSxpQkFBaUIsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sc0JBQXNCLEdBQUcsT0FBTyxDQUFDLGdCQUFnQixJQUFJLEVBQUUsQ0FBQztJQUU5RCx1RUFBdUU7SUFDdkUsTUFBTSxjQUFjLEdBQUcsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDO0lBQ25HLE1BQU0sVUFBVSxHQUFHLGdCQUFnQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsMkRBQTJELENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNqSCxNQUFNLGdCQUFnQixHQUFHLDJCQUEyQixjQUFjLElBQUksVUFBVSxJQUFJLHNCQUFzQixFQUFFLENBQUMsSUFBSSxFQUFFLENBQUM7SUFFcEgsT0FBTztRQUNMLGVBQWU7UUFDZix5QkFBeUIsT0FBTyxDQUFDLFdBQVcsMEJBQTBCLGdCQUFnQixLQUFLLGlCQUFpQixFQUFFLENBQUMsSUFBSSxFQUFFO1FBQ3JILGtEQUFrRCxLQUFLLENBQUMsU0FBUyxlQUFlLFlBQVksYUFBYSxLQUFLLENBQUMsTUFBTSxFQUFFO0tBQ3hILENBQUM7QUFDSixDQUFDO0FBN0RELDhEQTZEQztBQUVELFNBQWdCLDBCQUEwQixDQUFDLE9BQWlCO0lBQzFELE9BQU87UUFDTCx1QkFBdUI7UUFDdkIsZUFBZSxPQUFPLENBQUMsZUFBZSxHQUFHO1FBQ3pDLHdCQUF3QixPQUFPLENBQUMsK0JBQStCLEdBQUc7UUFDbEUsaUJBQWlCLE9BQU8sQ0FBQyxXQUFXLEdBQUc7S0FDeEMsQ0FBQztBQUNKLENBQUM7QUFQRCxnRUFPQztBQUVEOztHQUVHO0FBQ0gsSUFBWSxjQVNYO0FBVEQsV0FBWSxjQUFjO0lBQ3hCOztPQUVHO0lBQ0gsd0NBQXNCLENBQUE7SUFDdEI7O09BRUc7SUFDSCxrQ0FBZ0IsQ0FBQTtBQUNsQixDQUFDLEVBVFcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFTekIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBhdXRvc2NhbGluZyBmcm9tICcuLi8uLi9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCB7IEJvb3RzdHJhcE9wdGlvbnMsIElDbHVzdGVyIH0gZnJvbSAnLi9jbHVzdGVyJztcblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJBbWF6b25MaW51eFVzZXJEYXRhKGNsdXN0ZXI6IElDbHVzdGVyLCBhdXRvU2NhbGluZ0dyb3VwOiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwLCBvcHRpb25zOiBCb290c3RyYXBPcHRpb25zID0ge30pOiBzdHJpbmdbXSB7XG5cbiAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihhdXRvU2NhbGluZ0dyb3VwKTtcblxuICAvLyBkZXRlcm1pbmUgbG9naWNhbCBpZCBvZiBBU0cgc28gd2UgY2FuIHNpZ25hbCBjbG91ZGZvcm1hdGlvblxuICBjb25zdCBjZm4gPSBhdXRvU2NhbGluZ0dyb3VwLm5vZGUuZGVmYXVsdENoaWxkIGFzIGF1dG9zY2FsaW5nLkNmbkF1dG9TY2FsaW5nR3JvdXA7XG4gIGNvbnN0IGFzZ0xvZ2ljYWxJZCA9IGNmbi5sb2dpY2FsSWQ7XG5cbiAgY29uc3QgZXh0cmFBcmdzID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICB0cnkge1xuICAgIGNvbnN0IGNsdXN0ZXJFbmRwb2ludCA9IGNsdXN0ZXIuY2x1c3RlckVuZHBvaW50O1xuICAgIGNvbnN0IGNsdXN0ZXJDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGEgPVxuICAgICAgY2x1c3Rlci5jbHVzdGVyQ2VydGlmaWNhdGVBdXRob3JpdHlEYXRhO1xuICAgIGV4dHJhQXJncy5wdXNoKGAtLWFwaXNlcnZlci1lbmRwb2ludCAnJHtjbHVzdGVyRW5kcG9pbnR9J2ApO1xuICAgIGV4dHJhQXJncy5wdXNoKGAtLWI2NC1jbHVzdGVyLWNhICcke2NsdXN0ZXJDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGF9J2ApO1xuICB9IGNhdGNoIChlKSB7XG4gICAgLyoqXG4gICAgICogRXJyb3JzIGFyZSBpZ25vcmVkIGhlcmUuXG4gICAgICogYXBpc2VydmVyLWVuZHBvaW50IGFuZCBiNjQtY2x1c3Rlci1jYSBhcmd1bWVudHMgYXJlIGFkZGVkIGluICMxMjY1OSB0byBtYWtlIG5vZGVzIGpvaW4gdGhlIGNsdXN0ZXIgZmFzdGVyLlxuICAgICAqIEFzIHRoZXNlIGFyZSBub3QgbmVjZXNzYXJ5IGFyZ3VtZW50cywgd2UgZG9uJ3QgbmVlZCB0byBwYXNzIHRoZXNlIGFyZ3VtZW50cyB3aGVuIHRoZXkgZG9uJ3QgZXhpc3QuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzEyNjU5XG4gICAgICovXG4gIH1cblxuICBleHRyYUFyZ3MucHVzaChgLS11c2UtbWF4LXBvZHMgJHtvcHRpb25zLnVzZU1heFBvZHMgPz8gdHJ1ZX1gKTtcblxuICBpZiAob3B0aW9ucy5hd3NBcGlSZXRyeUF0dGVtcHRzKSB7XG4gICAgZXh0cmFBcmdzLnB1c2goYC0tYXdzLWFwaS1yZXRyeS1hdHRlbXB0cyAke29wdGlvbnMuYXdzQXBpUmV0cnlBdHRlbXB0c31gKTtcbiAgfVxuXG4gIGlmIChvcHRpb25zLmVuYWJsZURvY2tlckJyaWRnZSkge1xuICAgIGV4dHJhQXJncy5wdXNoKCctLWVuYWJsZS1kb2NrZXItYnJpZGdlIHRydWUnKTtcbiAgfVxuXG4gIGlmIChvcHRpb25zLmRvY2tlckNvbmZpZ0pzb24pIHtcbiAgICBleHRyYUFyZ3MucHVzaChgLS1kb2NrZXItY29uZmlnLWpzb24gJyR7b3B0aW9ucy5kb2NrZXJDb25maWdKc29ufSdgKTtcbiAgfVxuXG4gIGlmIChvcHRpb25zLmRuc0NsdXN0ZXJJcCkge1xuICAgIGV4dHJhQXJncy5wdXNoKGAtLWRucy1jbHVzdGVyLWlwICR7b3B0aW9ucy5kbnNDbHVzdGVySXB9YCk7XG4gIH1cblxuICBpZiAob3B0aW9ucy5hZGRpdGlvbmFsQXJncykge1xuICAgIGV4dHJhQXJncy5wdXNoKG9wdGlvbnMuYWRkaXRpb25hbEFyZ3MpO1xuICB9XG5cbiAgY29uc3QgY29tbWFuZExpbmVTdWZmaXggPSBleHRyYUFyZ3Muam9pbignICcpO1xuICBjb25zdCBrdWJlbGV0RXh0cmFBcmdzU3VmZml4ID0gb3B0aW9ucy5rdWJlbGV0RXh0cmFBcmdzIHx8ICcnO1xuXG4gIC8vIGRldGVybWluZSBsaWZlY3ljbGUgbGFiZWwgYmFzZWQgb24gd2hldGhlciB0aGUgQVNHIGhhcyBhIHNwb3QgcHJpY2UuXG4gIGNvbnN0IGxpZmVjeWNsZUxhYmVsID0gYXV0b1NjYWxpbmdHcm91cC5zcG90UHJpY2UgPyBMaWZlY3ljbGVMYWJlbC5TUE9UIDogTGlmZWN5Y2xlTGFiZWwuT05fREVNQU5EO1xuICBjb25zdCB3aXRoVGFpbnRzID0gYXV0b1NjYWxpbmdHcm91cC5zcG90UHJpY2UgPyAnLS1yZWdpc3Rlci13aXRoLXRhaW50cz1zcG90SW5zdGFuY2U9dHJ1ZTpQcmVmZXJOb1NjaGVkdWxlJyA6ICcnO1xuICBjb25zdCBrdWJlbGV0RXh0cmFBcmdzID0gYC0tbm9kZS1sYWJlbHMgbGlmZWN5Y2xlPSR7bGlmZWN5Y2xlTGFiZWx9ICR7d2l0aFRhaW50c30gJHtrdWJlbGV0RXh0cmFBcmdzU3VmZml4fWAudHJpbSgpO1xuXG4gIHJldHVybiBbXG4gICAgJ3NldCAtbyB4dHJhY2UnLFxuICAgIGAvZXRjL2Vrcy9ib290c3RyYXAuc2ggJHtjbHVzdGVyLmNsdXN0ZXJOYW1lfSAtLWt1YmVsZXQtZXh0cmEtYXJncyBcIiR7a3ViZWxldEV4dHJhQXJnc31cIiAke2NvbW1hbmRMaW5lU3VmZml4fWAudHJpbSgpLFxuICAgIGAvb3B0L2F3cy9iaW4vY2ZuLXNpZ25hbCAtLWV4aXQtY29kZSAkPyAtLXN0YWNrICR7c3RhY2suc3RhY2tOYW1lfSAtLXJlc291cmNlICR7YXNnTG9naWNhbElkfSAtLXJlZ2lvbiAke3N0YWNrLnJlZ2lvbn1gLFxuICBdO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyQm90dGxlcm9ja2V0VXNlckRhdGEoY2x1c3RlcjogSUNsdXN0ZXIpOiBzdHJpbmdbXSB7XG4gIHJldHVybiBbXG4gICAgJ1tzZXR0aW5ncy5rdWJlcm5ldGVzXScsXG4gICAgYGFwaS1zZXJ2ZXI9XCIke2NsdXN0ZXIuY2x1c3RlckVuZHBvaW50fVwiYCxcbiAgICBgY2x1c3Rlci1jZXJ0aWZpY2F0ZT1cIiR7Y2x1c3Rlci5jbHVzdGVyQ2VydGlmaWNhdGVBdXRob3JpdHlEYXRhfVwiYCxcbiAgICBgY2x1c3Rlci1uYW1lPVwiJHtjbHVzdGVyLmNsdXN0ZXJOYW1lfVwiYCxcbiAgXTtcbn1cblxuLyoqXG4gKiBUaGUgbGlmZWN5Y2xlIGxhYmVsIGZvciBub2RlIHNlbGVjdG9yXG4gKi9cbmV4cG9ydCBlbnVtIExpZmVjeWNsZUxhYmVsIHtcbiAgLyoqXG4gICAqIG9uLWRlbWFuZCBpbnN0YW5jZXNcbiAgICovXG4gIE9OX0RFTUFORCA9ICdPbkRlbWFuZCcsXG4gIC8qKlxuICAgKiBzcG90IGluc3RhbmNlc1xuICAgKi9cbiAgU1BPVCA9ICdFYzJTcG90J1xufVxuIl19