"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_2.Construct {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_eks_FargateProfileProps(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
            onEventLayer: props.cluster.onEventLayer,
        });
        this.podExecutionRole = (_b = props.podExecutionRole) !== null && _b !== void 0 ? _b : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        if (props.subnetSelection && !props.vpc) {
            core_1.Annotations.of(this).addWarning('Vpc must be defined to use a custom subnet selection. All private subnets belonging to the EKS cluster will be used by default');
        }
        let subnets;
        if (props.vpc) {
            const selection = (_c = props.subnetSelection) !== null && _c !== void 0 ? _c : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.any({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
_a = JSII_RTTI_SYMBOL_1;
FargateProfile[_a] = { fqn: "monocdk.aws_eks.FargateProfile", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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