"use strict";
/// !cdk-integ *
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const lib_1 = require("../lib");
const env = {
    region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
};
class VpcStack extends core_1.Stack {
    constructor(scope, id) {
        super(scope, id, { env });
        this.vpc = new ec2.Vpc(this, 'vpc', { maxAzs: 2 });
    }
}
class ClusterStack extends core_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, { env });
        /// !show
        // define the cluster. kubectl is enabled by default.
        this.cluster = new lib_1.Cluster(this, 'cluster22', {
            vpc: props.vpc,
            defaultCapacity: 0,
        });
        // define an IAM role assumable by anyone in the account and map it to the k8s
        // `system:masters` group this is required if you want to be able to issue
        // manual `kubectl` commands against the cluster.
        const mastersRole = new iam.Role(this, 'AdminRole', { assumedBy: new iam.AccountRootPrincipal() });
        this.cluster.awsAuth.addMastersRole(mastersRole);
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            desiredCapacity: 3,
        });
        // add two Helm charts to the cluster. This will be the Kubernetes dashboard and the Nginx Ingress Controller
        this.cluster.addChart('dashboard', { chart: 'kubernetes-dashboard', repository: 'https://kubernetes-charts.storage.googleapis.com' });
        this.cluster.addChart('nginx-ingress', { chart: 'nginx-ingress', repository: 'https://helm.nginx.com/stable', namespace: 'kube-system' });
    }
}
const app = new core_1.App();
const vpcStack = new VpcStack(app, 'k8s-vpc');
new ClusterStack(app, 'k8s-cluster', { vpc: vpcStack.vpc });
app.synth();
//# sourceMappingURL=data:application/json;base64,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