"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.OutOfInfrequentAccessPolicy = exports.LifecyclePolicy = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports
const core_2 = require("../../core");
const cxapi = require("../../cx-api");
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Out Of Infrequent Access Policy, if a file is accessed given times, it will move back to primary
 * storage class.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-efs-filesystem-lifecyclepolicy.html#cfn-efs-filesystem-lifecyclepolicy-transitiontoprimarystorageclass
 */
var OutOfInfrequentAccessPolicy;
(function (OutOfInfrequentAccessPolicy) {
    /**
     * After 1 access
     */
    OutOfInfrequentAccessPolicy["AFTER_1_ACCESS"] = "AFTER_1_ACCESS";
})(OutOfInfrequentAccessPolicy = exports.OutOfInfrequentAccessPolicy || (exports.OutOfInfrequentAccessPolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * General Purpose is ideal for latency-sensitive use cases, like web serving
     * environments, content management systems, home directories, and general file serving.
     * Recommended for the majority of Amazon EFS file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * File systems in the Max I/O mode can scale to higher levels of aggregate
     * throughput and operations per second. This scaling is done with a tradeoff
     * of slightly higher latencies for file metadata operations.
     * Highly parallelized applications and workloads, such as big data analysis,
     * media processing, and genomics analysis, can benefit from this mode.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     * This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
class FileSystemBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this File System resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.fileSystemArn],
        });
    }
}
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends FileSystemBase {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        jsiiDeprecationWarnings.monocdk_aws_efs_FileSystemProps(props);
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        // we explictly use 'undefined' to represent 'false' to maintain backwards compatibility since
        // its considered an actual change in CloudFormations eyes, even though they have the same meaning.
        const encrypted = (_b = props.encrypted) !== null && _b !== void 0 ? _b : (core_2.FeatureFlags.of(this).isEnabled(cxapi.EFS_DEFAULT_ENCRYPTION_AT_REST) ? true : undefined);
        // LifecyclePolicies is an array of lists containing a single policy
        let lifecyclePolicies = [];
        if (props.lifecyclePolicy) {
            lifecyclePolicies.push({ transitionToIa: props.lifecyclePolicy });
        }
        if (props.outOfInfrequentAccessPolicy) {
            lifecyclePolicies.push({ transitionToPrimaryStorageClass: props.outOfInfrequentAccessPolicy });
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: encrypted,
            kmsKeyId: (_c = props.kmsKey) === null || _c === void 0 ? void 0 : _c.keyArn,
            lifecyclePolicies: lifecyclePolicies.length > 0 ? lifecyclePolicies : undefined,
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_d = props.provisionedThroughputPerSecond) === null || _d === void 0 ? void 0 : _d.toMebibytes(),
            backupPolicy: props.enableAutomaticBackups ? { status: 'ENABLED' } : undefined,
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        this.fileSystemArn = filesystem.attrArn;
        core_1.Tags.of(this).add('Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets((_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { onePerAz: true });
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_efs_FileSystemAttributes(attrs);
        return new ImportedFileSystem(scope, id, attrs);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        jsiiDeprecationWarnings.monocdk_aws_efs_AccessPointOptions(accessPointOptions);
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
_a = JSII_RTTI_SYMBOL_1;
FileSystem[_a] = { fqn: "monocdk.aws_efs.FileSystem", version: "1.149.0" };
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
class ImportedFileSystem extends FileSystemBase {
    constructor(scope, id, attrs) {
        var _b, _c;
        super(scope, id);
        if (!!attrs.fileSystemId === !!attrs.fileSystemArn) {
            throw new Error('One of fileSystemId or fileSystemArn, but not both, must be provided.');
        }
        this.fileSystemArn = (_b = attrs.fileSystemArn) !== null && _b !== void 0 ? _b : core_1.Stack.of(scope).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: attrs.fileSystemId,
        });
        const parsedArn = core_1.Stack.of(scope).splitArn(this.fileSystemArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Invalid FileSystem Arn ${this.fileSystemArn}`);
        }
        this.fileSystemId = (_c = attrs.fileSystemId) !== null && _c !== void 0 ? _c : parsedArn.resourceName;
        this.connections = new ec2.Connections({
            securityGroups: [attrs.securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        this.mountTargetsAvailable = new core_1.ConcreteDependable();
    }
}
//# sourceMappingURL=data:application/json;base64,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