"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const ssm = require("../../aws-ssm");
const cdk = require("../../core");
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    /**
     * fluentbit
     */
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    /**
     * fluentd
     */
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    /**
     * s3
     */
    FirelensConfigFileType["S3"] = "s3";
    /**
     * fluentd
     */
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings.monocdk_aws_ecs_FirelensLogRouterProps(props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object
     */
    renderContainerDefinition(_taskDefinition) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinition(_taskDefinition);
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
_a = JSII_RTTI_SYMBOL_1;
FirelensLogRouter[_a] = { fqn: "monocdk.aws_ecs.FirelensLogRouter", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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