"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AsgCapacityProvider = exports.ExecuteCommandLogging = exports.Cluster = exports.MachineImageType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const autoscaling = require("../../aws-autoscaling");
const cloudwatch = require("../../aws-cloudwatch");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const cloudmap = require("../../aws-servicediscovery");
const core_1 = require("../../core");
const amis_1 = require("./amis");
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_canned_metrics_generated_1 = require("./ecs-canned-metrics.generated");
const ecs_generated_1 = require("./ecs.generated");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core");
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon ECS-optimized Amazon Linux 2 AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * The names of both ASG and Fargate capacity providers associated with the cluster.
         */
        this._capacityProviderNames = [];
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        jsiiDeprecationWarnings.monocdk_aws_ecs_ClusterProps(props);
        /**
         * clusterSettings needs to be undefined if containerInsights is not explicitly set in order to allow any
         * containerInsights settings on the account to apply.  See:
         * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-value
        */
        let clusterSettings = undefined;
        if (props.containerInsights !== undefined) {
            clusterSettings = [{ name: 'containerInsights', value: props.containerInsights ? ContainerInsights.ENABLED : ContainerInsights.DISABLED }];
        }
        this._capacityProviderNames = (_c = props.capacityProviders) !== null && _c !== void 0 ? _c : [];
        if (props.enableFargateCapacityProviders) {
            this.enableFargateCapacityProviders();
        }
        if (props.executeCommandConfiguration) {
            if ((props.executeCommandConfiguration.logging === ExecuteCommandLogging.OVERRIDE) !==
                (props.executeCommandConfiguration.logConfiguration !== undefined)) {
                throw new Error('Execute command log configuration must only be specified when logging is OVERRIDE.');
            }
            this._executeCommandConfiguration = props.executeCommandConfiguration;
        }
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
            configuration: this._executeCommandConfiguration && this.renderExecuteCommandConfiguration(),
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
        // Only create cluster capacity provider associations if there are any EC2
        // capacity providers. Ordinarily we'd just add the construct to the tree
        // since it's harmless, but we'd prefer not to add unexpected new
        // resources to the stack which could surprise users working with
        // brown-field CDK apps and stacks.
        core_1.Aspects.of(this).add(new MaybeCreateCapacityProviderAssociations(this, id, this._capacityProviderNames));
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_ClusterAttributes(attrs);
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Import an existing cluster to the stack from the cluster ARN.
     * This does not provide access to the vpc, hasEc2Capacity, or connections -
     * use the `fromClusterAttributes` method to access those properties.
     */
    static fromClusterArn(scope, id, clusterArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.splitArn(clusterArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const clusterName = arn.resourceName;
        if (!clusterName) {
            throw new Error(`Missing required Cluster Name from Cluster ARN: ${clusterArn}`);
        }
        const errorSuffix = 'is not available for a Cluster imported using fromClusterArn(), please use fromClusterAttributes() instead.';
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = clusterName;
            }
            get hasEc2Capacity() {
                throw new Error(`hasEc2Capacity ${errorSuffix}`);
            }
            get connections() {
                throw new Error(`connections ${errorSuffix}`);
            }
            get vpc() {
                throw new Error(`vpc ${errorSuffix}`);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: clusterArn,
        });
    }
    /**
     * Enable the Fargate capacity providers for this cluster.
     */
    enableFargateCapacityProviders() {
        for (const provider of ['FARGATE', 'FARGATE_SPOT']) {
            if (!this._capacityProviderNames.includes(provider)) {
                this._capacityProviderNames.push(provider);
            }
        }
    }
    renderExecuteCommandConfiguration() {
        var _c, _d, _e, _f;
        return {
            executeCommandConfiguration: {
                kmsKeyId: (_d = (_c = this._executeCommandConfiguration) === null || _c === void 0 ? void 0 : _c.kmsKey) === null || _d === void 0 ? void 0 : _d.keyArn,
                logConfiguration: ((_e = this._executeCommandConfiguration) === null || _e === void 0 ? void 0 : _e.logConfiguration) && this.renderExecuteCommandLogConfiguration(),
                logging: (_f = this._executeCommandConfiguration) === null || _f === void 0 ? void 0 : _f.logging,
            },
        };
    }
    renderExecuteCommandLogConfiguration() {
        var _c, _d, _e;
        const logConfiguration = (_c = this._executeCommandConfiguration) === null || _c === void 0 ? void 0 : _c.logConfiguration;
        if ((logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3EncryptionEnabled) && !(logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3Bucket)) {
            throw new Error('You must specify an S3 bucket name in the execute command log configuration to enable S3 encryption.');
        }
        if ((logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.cloudWatchEncryptionEnabled) && !(logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.cloudWatchLogGroup)) {
            throw new Error('You must specify a CloudWatch log group in the execute command log configuration to enable CloudWatch encryption.');
        }
        return {
            cloudWatchEncryptionEnabled: logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.cloudWatchEncryptionEnabled,
            cloudWatchLogGroupName: (_d = logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.cloudWatchLogGroup) === null || _d === void 0 ? void 0 : _d.logGroupName,
            s3BucketName: (_e = logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3Bucket) === null || _e === void 0 ? void 0 : _e.bucketName,
            s3EncryptionEnabled: logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3EncryptionEnabled,
            s3KeyPrefix: logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3KeyPrefix,
        };
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_CloudMapNamespaceOptions(options);
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * It is highly recommended to use {@link Cluster.addAsgCapacityProvider} instead of this method.
     *
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        var _c, _d;
        jsiiDeprecationWarnings.monocdk_aws_ecs_AddCapacityOptions(options);
        // Do 2-way defaulting here: if the machineImageType is BOTTLEROCKET, pick the right AMI.
        // Otherwise, determine the machineImageType from the given AMI.
        const machineImage = (_c = options.machineImage) !== null && _c !== void 0 ? _c : (options.machineImageType === MachineImageType.BOTTLEROCKET ? new amis_1.BottleRocketImage({
            architecture: options.instanceType.architecture,
        }) : new amis_1.EcsOptimizedAmi());
        const machineImageType = (_d = options.machineImageType) !== null && _d !== void 0 ? _d : (isBottleRocketImage(machineImage) ? MachineImageType.BOTTLEROCKET : MachineImageType.AMAZON_LINUX_2);
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            vpc: this.vpc,
            machineImage,
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            ...options,
        });
        this.addAutoScalingGroup(autoScalingGroup, {
            machineImageType: machineImageType,
            ...options,
        });
        return autoScalingGroup;
    }
    /**
     * This method adds an Auto Scaling Group Capacity Provider to a cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     */
    addAsgCapacityProvider(provider, options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_AsgCapacityProvider(provider);
        jsiiDeprecationWarnings.monocdk_aws_ecs_AddAutoScalingGroupCapacityOptions(options);
        // Don't add the same capacity provider more than once.
        if (this._capacityProviderNames.includes(provider.capacityProviderName)) {
            return;
        }
        this._hasEc2Capacity = true;
        this.configureAutoScalingGroup(provider.autoScalingGroup, {
            ...options,
            machineImageType: provider.machineImageType,
            // Don't enable the instance-draining lifecycle hook if managed termination protection is enabled
            taskDrainTime: provider.enableManagedTerminationProtection ? core_1.Duration.seconds(0) : options.taskDrainTime,
        });
        this._capacityProviderNames.push(provider.capacityProviderName);
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @deprecated Use {@link Cluster.addAsgCapacityProvider} instead.
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        jsiiDeprecationWarnings.print("monocdk.aws_ecs.Cluster#addAutoScalingGroup", "Use {@link Cluster.addAsgCapacityProvider} instead.");
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_AutoScalingGroup(autoScalingGroup);
        jsiiDeprecationWarnings.monocdk_aws_ecs_AddAutoScalingGroupCapacityOptions(options);
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        this.configureAutoScalingGroup(autoScalingGroup, options);
    }
    configureAutoScalingGroup(autoScalingGroup, options = {}) {
        if (autoScalingGroup.osType === ec2.OperatingSystemType.WINDOWS) {
            this.configureWindowsAutoScalingGroup(autoScalingGroup, options);
        }
        else {
            // Tie instances to cluster
            switch (options.machineImageType) {
                // Bottlerocket AMI
                case MachineImageType.BOTTLEROCKET: {
                    autoScalingGroup.addUserData(
                    // Connect to the cluster
                    // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#connecting-to-your-cluster
                    '[settings.ecs]', `cluster = "${this.clusterName}"`);
                    // Enabling SSM
                    // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#enabling-ssm
                    autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
                    // required managed policy
                    autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'));
                    break;
                }
                default:
                    // Amazon ECS-optimized AMI for Amazon Linux 2
                    autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
                    if (!options.canContainersAccessInstanceRole) {
                        // Deny containers access to instance metadata service
                        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
                        autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
                        autoScalingGroup.addUserData('sudo service iptables save');
                        // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
                        autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
                    }
                    if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
                        autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
                    }
            }
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        // But, scoped down to minimal permissions required.
        //  Notes:
        //   - 'ecs:CreateCluster' removed. The cluster already exists.
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DeregisterContainerInstance',
                'ecs:RegisterContainerInstance',
                'ecs:Submit*',
            ],
            resources: [
                this.clusterArn,
            ],
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These act on a cluster instance, and the instance doesn't exist until the service starts.
                // Thus, scope to the cluster using a condition.
                // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
                'ecs:Poll',
                'ecs:StartTelemetrySession',
            ],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': this.clusterArn },
            },
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These do not support resource constraints, and must be resource '*'
                'ecs:DiscoverPollEndpoint',
                'ecr:GetAuthorizationToken',
                // Preserved for backwards compatibility.
                // Users are able to enable cloudwatch agent using CDK. Existing
                // customers might be installing CW agent as part of user-data so if we
                // remove these permissions we will break that customer use cases.
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
                topicEncryptionKey: options.topicEncryptionKey,
            });
        }
    }
    /**
     * This method enables the Fargate or Fargate Spot capacity providers on the cluster.
     *
     * @param provider the capacity provider to add to this cluster.
     * @deprecated Use {@link enableFargateCapacityProviders} instead.
     * @see {@link addAsgCapacityProvider} to add an Auto Scaling Group capacity provider to the cluster.
     */
    addCapacityProvider(provider) {
        jsiiDeprecationWarnings.print("monocdk.aws_ecs.Cluster#addCapacityProvider", "Use {@link enableFargateCapacityProviders} instead.");
        if (!(provider === 'FARGATE' || provider === 'FARGATE_SPOT')) {
            throw new Error('CapacityProvider not supported');
        }
        if (!this._capacityProviderNames.includes(provider)) {
            this._capacityProviderNames.push(provider);
        }
    }
    configureWindowsAutoScalingGroup(autoScalingGroup, options = {}) {
        // clear the cache of the agent
        autoScalingGroup.addUserData('Remove-Item -Recurse C:\\ProgramData\\Amazon\\ECS\\Cache');
        // pull the latest ECS Tools
        autoScalingGroup.addUserData('Import-Module ECSTools');
        // set the cluster name environment variable
        autoScalingGroup.addUserData(`[Environment]::SetEnvironmentVariable("ECS_CLUSTER", "${this.clusterName}", "Machine")`);
        autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_AWSLOGS_EXECUTIONROLE_OVERRIDE", "true", "Machine")');
        // tslint:disable-next-line: max-line-length
        autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_AVAILABLE_LOGGING_DRIVERS", \'["json-file","awslogs"]\', "Machine")');
        // enable instance draining
        if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
            autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_SPOT_INSTANCE_DRAINING", "true", "Machine")');
        }
        // enable task iam role
        if (!options.canContainersAccessInstanceRole) {
            autoScalingGroup.addUserData('[Environment]::SetEnvironmentVariable("ECS_ENABLE_TASK_IAM_ROLE", "true", "Machine")');
            autoScalingGroup.addUserData(`Initialize-ECSAgent -Cluster '${this.clusterName}' -EnableTaskIAMRole`);
        }
        else {
            autoScalingGroup.addUserData(`Initialize-ECSAgent -Cluster '${this.clusterName}'`);
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * Getter for execute command configuration associated with the cluster.
     */
    get executeCommandConfiguration() {
        return this._executeCommandConfiguration;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.cpuReservationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.cpuUtilizationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.memoryReservationAverage, props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.cannedMetric(ecs_canned_metrics_generated_1.ECSMetrics.memoryUtilizationAverage, props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensionsMap: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ClusterName: this.clusterName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "monocdk.aws_ecs.Cluster", version: "1.149.0" };
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this._executeCommandConfiguration = props.executeCommandConfiguration;
        this.clusterArn = (_c = props.clusterArn) !== null && _c !== void 0 ? _c : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    get executeCommandConfiguration() {
        return this._executeCommandConfiguration;
    }
}
var ContainerInsights;
(function (ContainerInsights) {
    /**
     * Enable CloudWatch Container Insights for the cluster
     */
    ContainerInsights["ENABLED"] = "enabled";
    /**
     * Disable CloudWatch Container Insights for the cluster
     */
    ContainerInsights["DISABLED"] = "disabled";
})(ContainerInsights || (ContainerInsights = {}));
/**
 * The log settings to use to for logging the execute command session. For more information, see
 * [Logging] https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logging
 */
var ExecuteCommandLogging;
(function (ExecuteCommandLogging) {
    /**
     * The execute command session is not logged.
     */
    ExecuteCommandLogging["NONE"] = "NONE";
    /**
     * The awslogs configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no awslogs log driver is configured in the task definition, the output won't be logged.
     */
    ExecuteCommandLogging["DEFAULT"] = "DEFAULT";
    /**
     * Specify the logging details as a part of logConfiguration.
     */
    ExecuteCommandLogging["OVERRIDE"] = "OVERRIDE";
})(ExecuteCommandLogging = exports.ExecuteCommandLogging || (exports.ExecuteCommandLogging = {}));
/**
 * An Auto Scaling Group Capacity Provider. This allows an ECS cluster to target
 * a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
 * recommended), ECS can manage the number of instances in the ASG to fit the
 * tasks, and can ensure that instances are not prematurely terminated while
 * there are still tasks running on them.
 */
class AsgCapacityProvider extends core_2.Construct {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_ecs_AsgCapacityProviderProps(props);
        this.autoScalingGroup = props.autoScalingGroup;
        this.machineImageType = (_c = props.machineImageType) !== null && _c !== void 0 ? _c : MachineImageType.AMAZON_LINUX_2;
        this.enableManagedTerminationProtection =
            props.enableManagedTerminationProtection === undefined ? true : props.enableManagedTerminationProtection;
        if (this.enableManagedTerminationProtection) {
            this.autoScalingGroup.protectNewInstancesFromScaleIn();
        }
        if (props.capacityProviderName) {
            if (!(/^(?!aws|ecs|fargate).+/gm.test(props.capacityProviderName))) {
                throw new Error(`Invalid Capacity Provider Name: ${props.capacityProviderName}, If a name is specified, it cannot start with aws, ecs, or fargate.`);
            }
        }
        const capacityProvider = new ecs_generated_1.CfnCapacityProvider(this, id, {
            name: props.capacityProviderName,
            autoScalingGroupProvider: {
                autoScalingGroupArn: this.autoScalingGroup.autoScalingGroupName,
                managedScaling: props.enableManagedScaling === false ? undefined : {
                    status: 'ENABLED',
                    targetCapacity: props.targetCapacityPercent || 100,
                    maximumScalingStepSize: props.maximumScalingStepSize,
                    minimumScalingStepSize: props.minimumScalingStepSize,
                },
                managedTerminationProtection: this.enableManagedTerminationProtection ? 'ENABLED' : 'DISABLED',
            },
        });
        this.capacityProviderName = capacityProvider.ref;
    }
}
exports.AsgCapacityProvider = AsgCapacityProvider;
_b = JSII_RTTI_SYMBOL_1;
AsgCapacityProvider[_b] = { fqn: "monocdk.aws_ecs.AsgCapacityProvider", version: "1.149.0" };
/**
 * A visitor that adds a capacity provider association to a Cluster only if
 * the caller created any EC2 Capacity Providers.
 */
class MaybeCreateCapacityProviderAssociations {
    constructor(scope, id, capacityProviders) {
        this.scope = scope;
        this.id = id;
        this.capacityProviders = capacityProviders;
    }
    visit(node) {
        if (node instanceof Cluster) {
            if (this.capacityProviders.length > 0 && !this.resource) {
                const resource = new ecs_generated_1.CfnClusterCapacityProviderAssociations(this.scope, this.id, {
                    cluster: node.clusterName,
                    defaultCapacityProviderStrategy: [],
                    capacityProviders: core_1.Lazy.list({ produce: () => this.capacityProviders }),
                });
                this.resource = resource;
            }
        }
    }
}
function isBottleRocketImage(image) {
    return image instanceof amis_1.BottleRocketImage;
}
//# sourceMappingURL=data:application/json;base64,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