"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isExternalCompatible = exports.isFargateCompatible = exports.isEc2Compatible = exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
const _imported_task_definition_1 = require("./_imported-task-definition");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a ECS anywhere cluster
     */
    get isExternalCompatible() {
        return isExternalCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        /**
         * Inference accelerators for task instances
         */
        this._inferenceAccelerators = [];
        jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinitionProps(props);
        this.family = props.family || core_1.Names.uniqueId(this);
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = (_b = props.networkMode) !== null && _b !== void 0 ? _b : (this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE);
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        if (props.inferenceAccelerators && props.inferenceAccelerators.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        if (this.isExternalCompatible && this.networkMode !== NetworkMode.BRIDGE) {
            throw new Error(`External tasks can only have Bridge network mode, got: ${this.networkMode}`);
        }
        if (!this.isFargateCompatible && props.runtimePlatform) {
            throw new Error('Cannot specify runtimePlatform in non-Fargate compatible tasks');
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        if (props.inferenceAccelerators) {
            props.inferenceAccelerators.forEach(ia => this.addInferenceAccelerator(ia));
        }
        this.ephemeralStorageGiB = props.ephemeralStorageGiB;
        // validate the cpu and memory size for the Windows operation system family.
        if ((_d = (_c = props.runtimePlatform) === null || _c === void 0 ? void 0 : _c.operatingSystemFamily) === null || _d === void 0 ? void 0 : _d._operatingSystemFamily.includes('WINDOWS')) {
            // We know that props.cpu and props.memoryMiB are defined because an error would have been thrown previously if they were not.
            // But, typescript is not able to figure this out, so using the `!` operator here to let the type-checker know they are defined.
            this.checkFargateWindowsBasedTasksSize(props.cpu, props.memoryMiB, props.runtimePlatform);
        }
        this.runtimePlatform = props.runtimePlatform;
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.any({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.any({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.string({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
                ...(isExternalCompatible(props.compatibility) ? ['EXTERNAL'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
            inferenceAccelerators: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.renderInferenceAccelerators() : undefined,
            }, { omitEmptyArray: true }),
            ephemeralStorage: this.ephemeralStorageGiB ? {
                sizeInGiB: this.ephemeralStorageGiB,
            } : undefined,
            runtimePlatform: this.isFargateCompatible && this.runtimePlatform ? {
                cpuArchitecture: (_f = (_e = this.runtimePlatform) === null || _e === void 0 ? void 0 : _e.cpuArchitecture) === null || _f === void 0 ? void 0 : _f._cpuArchitecture,
                operatingSystemFamily: (_h = (_g = this.runtimePlatform) === null || _g === void 0 ? void 0 : _g.operatingSystemFamily) === null || _h === void 0 ? void 0 : _h._operatingSystemFamily,
            } : undefined,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, { taskDefinitionArn: taskDefinitionArn });
    }
    /**
     * Create a task definition from a task definition reference
     */
    static fromTaskDefinitionAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinitionAttributes(attrs);
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, {
            taskDefinitionArn: attrs.taskDefinitionArn,
            compatibility: attrs.compatibility,
            networkMode: attrs.networkMode,
            taskRole: attrs.taskRole,
        });
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Public getter method to access list of inference accelerators attached to the instance.
     */
    get inferenceAccelerators() {
        return this._inferenceAccelerators;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    fileSystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    renderInferenceAccelerators() {
        return this._inferenceAccelerators.map(renderInferenceAccelerator);
        function renderInferenceAccelerator(inferenceAccelerator) {
            return {
                deviceName: inferenceAccelerator.deviceName,
                deviceType: inferenceAccelerator.deviceType,
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.taskRole.addToPrincipalPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.obtainExecutionRole().addToPrincipalPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinitionOptions(props);
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_FirelensLogRouterDefinitionOptions(props);
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
        if (container.referencesSecretJsonField) {
            this._referencesSecretJsonField = true;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_Volume(volume);
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_PlacementConstraint(constraint);
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extension to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_ITaskDefinitionExtension(extension);
        extension.extend(this);
    }
    /**
     * Adds an inference accelerator to the task definition.
     */
    addInferenceAccelerator(inferenceAccelerator) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_InferenceAccelerator(inferenceAccelerator);
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        this._inferenceAccelerators.push(inferenceAccelerator);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                // needed for cross-account access with TagParameterContainerImage
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            });
        }
        return this._executionRole;
    }
    /**
     * Whether this task definition has at least a container that references a
     * specific JSON field of a secret stored in Secrets Manager.
     */
    get referencesSecretJsonField() {
        return this._referencesSecretJsonField;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
    checkFargateWindowsBasedTasksSize(cpu, memory, runtimePlatform) {
        if (Number(cpu) === 1024) {
            if (Number(memory) < 1024 || Number(memory) > 8192 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 1024 and a max of 8192, in 1024 increments. Provided memoryMiB was ${Number(memory)}.`);
            }
        }
        else if (Number(cpu) === 2048) {
            if (Number(memory) < 4096 || Number(memory) > 16384 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 4096 and max of 16384, in 1024 increments. Provided memoryMiB ${Number(memory)}.`);
            }
        }
        else if (Number(cpu) === 4096) {
            if (Number(memory) < 8192 || Number(memory) > 30720 || (Number(memory) % 1024 !== 0)) {
                throw new Error(`If provided cpu is ${cpu}, then memoryMiB must have a min of 8192 and a max of 30720, in 1024 increments.Provided memoryMiB was ${Number(memory)}.`);
            }
        }
        else {
            throw new Error(`If operatingSystemFamily is ${runtimePlatform.operatingSystemFamily._operatingSystemFamily}, then cpu must be in 1024 (1 vCPU), 2048 (2 vCPU), or 4096 (4 vCPU). Provided value was: ${cpu}`);
        }
    }
    ;
}
exports.TaskDefinition = TaskDefinition;
_a = JSII_RTTI_SYMBOL_1;
TaskDefinition[_a] = { fqn: "monocdk.aws_ecs.TaskDefinition", version: "1.149.0" };
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
    /**
     * The task should specify the External launch type.
     */
    Compatibility[Compatibility["EXTERNAL"] = 3] = "EXTERNAL";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isEc2Compatible = isEc2Compatible;
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isFargateCompatible = isFargateCompatible;
/**
 * Return true if the given task definition can be run on a ECS Anywhere cluster
 */
function isExternalCompatible(compatibility) {
    return [Compatibility.EXTERNAL].includes(compatibility);
}
exports.isExternalCompatible = isExternalCompatible;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFzay1kZWZpbml0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGFzay1kZWZpbml0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsd0NBQStFO0FBRS9FLGtFQUFpSDtBQUNqSCxvREFBcUQ7QUFDckQsZ0VBQXNKO0FBQ3RKLGtFQUE2RDtBQUk3RCwyRUFBcUU7QUF1UHJFLE1BQWUsa0JBQW1CLFNBQVEsZUFBUTtJQVFoRDs7T0FFRztJQUNILElBQVcsZUFBZTtRQUN4QixPQUFPLGVBQWUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7S0FDNUM7SUFFRDs7T0FFRztJQUNILElBQVcsbUJBQW1CO1FBQzVCLE9BQU8sbUJBQW1CLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQ2hEO0lBRUQ7O09BRUc7SUFDSCxJQUFXLG9CQUFvQjtRQUM3QixPQUFPLG9CQUFvQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztLQUNqRDtDQUNGO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGNBQWUsU0FBUSxrQkFBa0I7SUE0RnBEOztPQUVHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjs7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQTlCbkI7O1dBRUc7UUFDZ0IsZUFBVSxHQUFHLElBQUksS0FBSyxFQUF1QixDQUFDO1FBRWpFOztXQUVHO1FBQ2MsWUFBTyxHQUFhLEVBQUUsQ0FBQztRQUV4Qzs7V0FFRztRQUNjLHlCQUFvQixHQUFHLElBQUksS0FBSyxFQUErRCxDQUFDO1FBRWpIOztXQUVHO1FBQ2MsMkJBQXNCLEdBQTJCLEVBQUUsQ0FBQzs7UUFjbkUsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRXpDLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNqQixLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMvQztRQUVELElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5RyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxPQUFPLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FBQyx5REFBeUQsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDOUY7UUFDRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxPQUFPLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDM0c7UUFDRCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDbkcsTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDO1NBQ2xGO1FBRUQsSUFBSSxJQUFJLENBQUMsbUJBQW1CLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDaEUsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsS0FBSyxDQUFDLEdBQUcsaUJBQWlCLEtBQUssQ0FBQyxTQUFTLGtCQUFrQixDQUFDLENBQUM7U0FDNUg7UUFFRCxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSxLQUFLLENBQUMscUJBQXFCLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDckcsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ25GO1FBRUQsSUFBSSxJQUFJLENBQUMsb0JBQW9CLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsTUFBTSxFQUFFO1lBQ3hFLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQy9GO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUVELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUUxQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0QsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1NBQy9ELENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO1lBQy9CLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUM3RTtRQUVELElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUM7UUFFckQsNEVBQTRFO1FBQzVFLGdCQUFJLEtBQUssQ0FBQyxlQUFlLDBDQUFFLHFCQUFxQiwwQ0FBRSxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsU0FBUyxHQUFHO1lBQzVGLDhIQUE4SDtZQUM5SCxnSUFBZ0k7WUFDaEksSUFBSSxDQUFDLGlDQUFpQyxDQUFDLEtBQUssQ0FBQyxHQUFJLEVBQUUsS0FBSyxDQUFDLFNBQVUsRUFBRSxLQUFLLENBQUMsZUFBZ0IsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO1FBRTdDLE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN0RCxvQkFBb0IsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDcEcsT0FBTyxFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDcEYsZ0JBQWdCLEVBQUUsV0FBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDbEcsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO1lBQ25CLFdBQVcsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU87WUFDbEMsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEdBQUcsQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ3hELEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDaEUsR0FBRyxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ25FO1lBQ0QsV0FBVyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1lBQ3JELG9CQUFvQixFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUM7Z0JBQzdCLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FDWixDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ25GLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDNUIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDMUcsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsTUFBTSxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3ZCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIscUJBQXFCLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQztnQkFDOUIsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUNaLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUM1RixFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDO1lBQzVCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7Z0JBQzNDLFNBQVMsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2FBQ3BDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixlQUFlLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO2dCQUNsRSxlQUFlLGNBQUUsSUFBSSxDQUFDLGVBQWUsMENBQUUsZUFBZSwwQ0FBRSxnQkFBZ0I7Z0JBQ3hFLHFCQUFxQixjQUFFLElBQUksQ0FBQyxlQUFlLDBDQUFFLHFCQUFxQiwwQ0FBRSxzQkFBc0I7YUFDM0YsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNkLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLG9CQUFvQixFQUFFO1lBQzlCLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUMzRTtRQUVELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDO0tBQ3RDO0lBOUxEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQ3pGLE9BQU8sSUFBSSxrREFBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO0tBQ3hGO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsNEJBQTRCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBK0I7O1FBQ3RHLE9BQU8sSUFBSSxrREFBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQzNDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7U0FDekIsQ0FBQyxDQUFDO0tBQ0o7SUE2S0QsSUFBVyxhQUFhO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQztLQUM1QjtJQUVEOztPQUVHO0lBQ0gsSUFBVyxxQkFBcUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsc0JBQXNCLENBQUM7S0FDcEM7SUFFTyxhQUFhO1FBQ25CLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFdEMsU0FBUyxZQUFZLENBQUMsSUFBWTtZQUNoQyxPQUFPO2dCQUNMLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YseUJBQXlCLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixJQUFJO29CQUMzRCxhQUFhLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixDQUFDLGFBQWE7b0JBQzNELE1BQU0sRUFBRSxJQUFJLENBQUMseUJBQXlCLENBQUMsTUFBTTtvQkFDN0MsVUFBVSxFQUFFLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVO29CQUNyRCxNQUFNLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixDQUFDLE1BQU07b0JBQzdDLEtBQUssRUFBRSxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSztpQkFDNUM7Z0JBQ0Qsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixJQUFJO29CQUNyRCxZQUFZLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLFlBQVk7b0JBQ3RELG1CQUFtQixFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxtQkFBbUI7b0JBQ3BFLGFBQWEsRUFBRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsYUFBYTtvQkFDeEQsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLGlCQUFpQjtvQkFDaEUscUJBQXFCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLHFCQUFxQjtpQkFFekU7YUFDRixDQUFDO1FBQ0osQ0FBQztLQUNGO0lBRU8sMkJBQTJCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBRW5FLFNBQVMsMEJBQTBCLENBQUMsb0JBQTBDO1lBQzVFLE9BQU87Z0JBQ0wsVUFBVSxFQUFFLG9CQUFvQixDQUFDLFVBQVU7Z0JBQzNDLFVBQVUsRUFBRSxvQkFBb0IsQ0FBQyxVQUFVO2FBQzVDLENBQUM7UUFDSixDQUFDO0tBQ0Y7SUFFRDs7OztPQUlHO0lBQ0ksZUFBZSxDQUFDLE9BQWtDO1FBQ3ZELE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ2xFLElBQUksZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixPQUFPLENBQUMsYUFBYSxtQ0FBbUMsQ0FBQyxDQUFDO1NBQ2xHO1FBQ0QsTUFBTSxjQUFjLEdBQUcsT0FBTyxDQUFDLFFBQVEsSUFBSSwrQkFBUSxDQUFDLEdBQUcsQ0FBQztRQUN4RCxNQUFNLG1CQUFtQixHQUFHLE9BQU8sQ0FBQyxhQUFhLElBQUksZUFBZSxDQUFDLGFBQWEsQ0FBQztRQUNuRixNQUFNLFdBQVcsR0FBRyxlQUFlLENBQUMsZUFBZSxDQUFDLG1CQUFtQixFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3pGLElBQUksV0FBVyxLQUFLLFNBQVMsRUFBRTtZQUM3QixtQ0FBbUM7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLGVBQWUsNkJBQTZCLE9BQU8sQ0FBQyxhQUFhLGlCQUFpQixjQUFjLCtDQUErQyxDQUFDLENBQUM7U0FDaEw7UUFDRCxPQUFPO1lBQ0wsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQ3BDLFdBQVc7U0FDWixDQUFDO0tBQ0g7SUFFRDs7OztPQUlHO0lBQ0kseUJBQXlCLENBQUMsV0FBd0I7UUFDdkQsSUFBSSxXQUFXLENBQUMsUUFBUSxLQUFLLFNBQVMsSUFBSSxXQUFXLENBQUMsUUFBUSxLQUFLLENBQUMsRUFBRTtZQUNwRSxPQUFPLFdBQVcsQ0FBQyxRQUFRLEtBQUssK0JBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3hIO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsR0FBRyxFQUFFO1lBQ25GLE9BQU8sb0JBQW9CLENBQUM7U0FDN0I7UUFDRCxPQUFPLFdBQVcsQ0FBQyxRQUFRLEtBQUssK0JBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQ2xJO0lBRUQ7O09BRUc7SUFDSSxtQkFBbUIsQ0FBQyxTQUE4Qjs7UUFDdkQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztLQUMvQztJQUVEOztPQUVHO0lBQ0ksd0JBQXdCLENBQUMsU0FBOEI7O1FBQzVELElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQzVEO0lBRUQ7O09BRUc7SUFDSSxZQUFZLENBQUMsRUFBVSxFQUFFLEtBQWlDOztRQUMvRCxPQUFPLElBQUksMENBQW1CLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0tBQzlFO0lBRUQ7O09BRUc7SUFDSSxvQkFBb0IsQ0FBQyxFQUFVLEVBQUUsS0FBeUM7O1FBQy9FLHdEQUF3RDtRQUN4RCxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxZQUFZLHVDQUFpQixDQUFDLEVBQUU7WUFDN0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1NBQ3ZFO1FBRUQsT0FBTyxJQUFJLHVDQUFpQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztLQUM1RTtJQUVEOzs7T0FHRztJQUNJLGNBQWMsQ0FBQyxTQUE4QjtRQUNsRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoQyxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLElBQUksU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUM5RCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsU0FBUyxDQUFDO1NBQ25DO1FBQ0QsSUFBSSxTQUFTLENBQUMseUJBQXlCLEVBQUU7WUFDdkMsSUFBSSxDQUFDLDBCQUEwQixHQUFHLElBQUksQ0FBQztTQUN4QztLQUNGO0lBRUQ7O09BRUc7SUFDSSxTQUFTLENBQUMsTUFBYzs7UUFDN0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7S0FDM0I7SUFFRDs7T0FFRztJQUNJLHNCQUFzQixDQUFDLFVBQStCOztRQUMzRCxJQUFJLG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLCtEQUErRCxDQUFDLENBQUM7U0FDbEY7UUFDRCxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLEdBQUcsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7S0FDeEQ7SUFFRDs7Ozs7T0FLRztJQUNJLFlBQVksQ0FBQyxTQUFtQzs7UUFDckQsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUN4QjtJQUVEOztPQUVHO0lBQ0ksdUJBQXVCLENBQUMsb0JBQTBDOztRQUN2RSxJQUFJLG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDbkY7UUFDRCxJQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7S0FDeEQ7SUFFRDs7T0FFRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4QixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUN4RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7Z0JBQzlELGtFQUFrRTtnQkFDbEUsUUFBUSxFQUFFLG1CQUFZLENBQUMsa0JBQWtCO2FBQzFDLENBQUMsQ0FBQztTQUNKO1FBQ0QsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDO0tBQzVCO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyx5QkFBeUI7UUFDbEMsT0FBTyxJQUFJLENBQUMsMEJBQTBCLENBQUM7S0FDeEM7SUFFRDs7T0FFRztJQUNPLFFBQVE7UUFDaEIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRTdCLElBQUksZUFBZSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUN2Qyx1QkFBdUI7WUFFdkIsa0JBQWtCO1lBQ2xCLEtBQUssTUFBTSxTQUFTLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtnQkFDdkMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsRUFBRTtvQkFDbkMsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsU0FBUyxDQUFDLGFBQWEsaUZBQWlGLENBQUMsQ0FBQztpQkFDckk7YUFDRjtTQUNGO1FBQ0QsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVEOztPQUVHO0lBQ0ksYUFBYSxDQUFDLGFBQXFCO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsYUFBYSxLQUFLLGFBQWEsQ0FBQyxDQUFDO0tBQ3JFO0lBRU8saUJBQWlCLENBQUMsV0FBd0I7UUFDaEQsT0FBTyxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDO0tBQ3BFO0lBRU8sZ0JBQWdCO1FBQ3RCLCtGQUErRjtRQUMvRixxREFBcUQ7UUFDckQsS0FBSyxNQUFNLFNBQVMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ3ZDLElBQUksU0FBUyxDQUFDLGVBQWUsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLFNBQVMsS0FBSyxhQUFhO21CQUNqRixDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxZQUFZLHVDQUFpQixDQUFDLEVBQUU7Z0JBQy9ELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZLEVBQUU7b0JBQ3RDLEtBQUssRUFBRSxvREFBOEIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQztvQkFDdEUsY0FBYyxFQUFFO3dCQUNkLElBQUksRUFBRSwyQ0FBcUIsQ0FBQyxTQUFTO3FCQUN0QztvQkFDRCxPQUFPLEVBQUUsSUFBSSw2QkFBWSxDQUFDLEVBQUUsWUFBWSxFQUFFLFVBQVUsRUFBRSxDQUFDO29CQUN2RCxvQkFBb0IsRUFBRSxFQUFFO2lCQUN6QixDQUFDLENBQUM7Z0JBRUgsTUFBTTthQUNQO1NBQ0Y7UUFFRCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLHlCQUF5QixFQUFFLENBQUMsQ0FBQztLQUNoRTtJQUVPLGlDQUFpQyxDQUFDLEdBQVcsRUFBRSxNQUFjLEVBQUUsZUFBZ0M7UUFDckcsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ3hCLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQUksSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFFLElBQUksS0FBSyxDQUFDLENBQUMsRUFBRTtnQkFDbEYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsR0FBRywwR0FBMEcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUN2SztTQUNGO2FBQU0sSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQy9CLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQUksSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsS0FBSyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLENBQUMsRUFBRTtnQkFDcEYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsR0FBRyxxR0FBcUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNsSztTQUNGO2FBQU0sSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQy9CLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQUksSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsS0FBSyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLENBQUMsRUFBRTtnQkFDcEYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQkFBdUIsR0FBSSwwR0FBMkcsTUFBTSxDQUFDLE1BQU0sQ0FBRSxHQUFHLENBQUMsQ0FBQzthQUMzSztTQUNGO2FBQU07WUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixlQUFlLENBQUMscUJBQXNCLENBQUMsc0JBQXNCLDZGQUE2RixHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ2pOO0tBQ0Y7SUFBQSxDQUFDOztBQXRjSix3Q0F1Y0M7OztBQUVEOztHQUVHO0FBQ0gsTUFBTSxvQkFBb0IsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFFN0Q7O0dBRUc7QUFDSCxJQUFZLFdBK0JYO0FBL0JELFdBQVksV0FBVztJQUNyQjs7T0FFRztJQUNILDRCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILGdDQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsaUNBQWtCLENBQUE7SUFFbEI7Ozs7O09BS0c7SUFDSCw0QkFBYSxDQUFBO0lBRWI7Ozs7O09BS0c7SUFDSCwwQkFBVyxDQUFBO0FBQ2IsQ0FBQyxFQS9CVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQStCdEI7QUFFRDs7R0FFRztBQUNILElBQVksT0FpQlg7QUFqQkQsV0FBWSxPQUFPO0lBQ2pCOzs7T0FHRztJQUNILHdCQUFhLENBQUE7SUFFYjs7O09BR0c7SUFDSCx3QkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCx3QkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQWpCVyxPQUFPLEdBQVAsZUFBTyxLQUFQLGVBQU8sUUFpQmxCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLE9BV1g7QUFYRCxXQUFZLE9BQU87SUFDakI7OztPQUdHO0lBQ0gsd0JBQWEsQ0FBQTtJQUViOztPQUVHO0lBQ0gsd0JBQWEsQ0FBQTtBQUNmLENBQUMsRUFYVyxPQUFPLEdBQVAsZUFBTyxLQUFQLGVBQU8sUUFXbEI7QUFnT0Q7Ozs7R0FJRztBQUNILElBQVksS0FVWDtBQVZELFdBQVksS0FBSztJQUNmOztPQUVHO0lBQ0gsc0JBQWEsQ0FBQTtJQUViOztPQUVHO0lBQ0gsMEJBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQVZXLEtBQUssR0FBTCxhQUFLLEtBQUwsYUFBSyxRQVVoQjtBQUVEOztHQUVHO0FBQ0gsSUFBWSxhQW9CWDtBQXBCRCxXQUFZLGFBQWE7SUFDdkI7O09BRUc7SUFDSCwrQ0FBRyxDQUFBO0lBRUg7O09BRUc7SUFDSCx1REFBTyxDQUFBO0lBRVA7O09BRUc7SUFDSCx1RUFBZSxDQUFBO0lBRWY7O09BRUc7SUFDSCx5REFBUSxDQUFBO0FBQ1YsQ0FBQyxFQXBCVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQW9CeEI7QUFvQkQ7O0dBRUc7QUFDSCxTQUFnQixlQUFlLENBQUMsYUFBNEI7SUFDMUQsT0FBTyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsYUFBYSxDQUFDLGVBQWUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztBQUNwRixDQUFDO0FBRkQsMENBRUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLG1CQUFtQixDQUFDLGFBQTRCO0lBQzlELE9BQU8sQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLGFBQWEsQ0FBQyxlQUFlLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7QUFDeEYsQ0FBQztBQUZELGtEQUVDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixvQkFBb0IsQ0FBQyxhQUE0QjtJQUMvRCxPQUFPLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztBQUMxRCxDQUFDO0FBRkQsb0RBRUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnLi4vLi4vLi4vYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnLi4vLi4vLi4vYXdzLWlhbSc7XG5pbXBvcnQgeyBJUmVzb3VyY2UsIExhenksIE5hbWVzLCBQaHlzaWNhbE5hbWUsIFJlc291cmNlIH0gZnJvbSAnLi4vLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvbnRhaW5lckRlZmluaXRpb24sIENvbnRhaW5lckRlZmluaXRpb25PcHRpb25zLCBQb3J0TWFwcGluZywgUHJvdG9jb2wgfSBmcm9tICcuLi9jb250YWluZXItZGVmaW5pdGlvbic7XG5pbXBvcnQgeyBDZm5UYXNrRGVmaW5pdGlvbiB9IGZyb20gJy4uL2Vjcy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgRmlyZWxlbnNMb2dSb3V0ZXIsIEZpcmVsZW5zTG9nUm91dGVyRGVmaW5pdGlvbk9wdGlvbnMsIEZpcmVsZW5zTG9nUm91dGVyVHlwZSwgb2J0YWluRGVmYXVsdEZsdWVudEJpdEVDUkltYWdlIH0gZnJvbSAnLi4vZmlyZWxlbnMtbG9nLXJvdXRlcic7XG5pbXBvcnQgeyBBd3NMb2dEcml2ZXIgfSBmcm9tICcuLi9sb2ctZHJpdmVycy9hd3MtbG9nLWRyaXZlcic7XG5pbXBvcnQgeyBQbGFjZW1lbnRDb25zdHJhaW50IH0gZnJvbSAnLi4vcGxhY2VtZW50JztcbmltcG9ydCB7IFByb3h5Q29uZmlndXJhdGlvbiB9IGZyb20gJy4uL3Byb3h5LWNvbmZpZ3VyYXRpb24vcHJveHktY29uZmlndXJhdGlvbic7XG5pbXBvcnQgeyBSdW50aW1lUGxhdGZvcm0gfSBmcm9tICcuLi9ydW50aW1lLXBsYXRmb3JtJztcbmltcG9ydCB7IEltcG9ydGVkVGFza0RlZmluaXRpb24gfSBmcm9tICcuL19pbXBvcnRlZC10YXNrLWRlZmluaXRpb24nO1xuXG4vKipcbiAqIFRoZSBpbnRlcmZhY2UgZm9yIGFsbCB0YXNrIGRlZmluaXRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElUYXNrRGVmaW5pdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBBUk4gb2YgdGhpcyB0YXNrIGRlZmluaXRpb25cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdGFza0RlZmluaXRpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogRXhlY3V0aW9uIHJvbGUgZm9yIHRoaXMgdGFzayBkZWZpbml0aW9uXG4gICAqL1xuICByZWFkb25seSBleGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBXaGF0IGxhdW5jaCB0eXBlcyB0aGlzIHRhc2sgZGVmaW5pdGlvbiBzaG91bGQgYmUgY29tcGF0aWJsZSB3aXRoLlxuICAgKi9cbiAgcmVhZG9ubHkgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eTtcblxuICAvKipcbiAgICogUmV0dXJuIHRydWUgaWYgdGhlIHRhc2sgZGVmaW5pdGlvbiBjYW4gYmUgcnVuIG9uIGFuIEVDMiBjbHVzdGVyXG4gICAqL1xuICByZWFkb25seSBpc0VjMkNvbXBhdGlibGU6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFJldHVybiB0cnVlIGlmIHRoZSB0YXNrIGRlZmluaXRpb24gY2FuIGJlIHJ1biBvbiBhIEZhcmdhdGUgY2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgaXNGYXJnYXRlQ29tcGF0aWJsZTogYm9vbGVhbjtcblxuICAvKipcbiAgICogUmV0dXJuIHRydWUgaWYgdGhlIHRhc2sgZGVmaW5pdGlvbiBjYW4gYmUgcnVuIG9uIGEgRUNTIEFueXdoZXJlIGNsdXN0ZXJcbiAgICovXG4gIHJlYWRvbmx5IGlzRXh0ZXJuYWxDb21wYXRpYmxlOiBib29sZWFuO1xuXG5cbiAgLyoqXG4gICAqIFRoZSBuZXR3b3JraW5nIG1vZGUgdG8gdXNlIGZvciB0aGUgY29udGFpbmVycyBpbiB0aGUgdGFzay5cbiAgICovXG4gIHJlYWRvbmx5IG5ldHdvcmtNb2RlOiBOZXR3b3JrTW9kZTtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIElBTSByb2xlIHRoYXQgZ3JhbnRzIGNvbnRhaW5lcnMgaW4gdGhlIHRhc2sgcGVybWlzc2lvbiB0byBjYWxsIEFXUyBBUElzIG9uIHlvdXIgYmVoYWxmLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFza1JvbGU6IGlhbS5JUm9sZTtcbn1cblxuLyoqXG4gKiBUaGUgY29tbW9uIHByb3BlcnRpZXMgZm9yIGFsbCB0YXNrIGRlZmluaXRpb25zLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gKiBbVGFzayBEZWZpbml0aW9uIFBhcmFtZXRlcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3Rhc2tfZGVmaW5pdGlvbl9wYXJhbWV0ZXJzLmh0bWwpLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbW1vblRhc2tEZWZpbml0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgYSBmYW1pbHkgdGhhdCB0aGlzIHRhc2sgZGVmaW5pdGlvbiBpcyByZWdpc3RlcmVkIHRvLiBBIGZhbWlseSBncm91cHMgbXVsdGlwbGUgdmVyc2lvbnMgb2YgYSB0YXNrIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IGZhbWlseT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIElBTSB0YXNrIGV4ZWN1dGlvbiByb2xlIHRoYXQgZ3JhbnRzIHRoZSBFQ1MgYWdlbnQgcGVybWlzc2lvbiB0byBjYWxsIEFXUyBBUElzIG9uIHlvdXIgYmVoYWxmLlxuICAgKlxuICAgKiBUaGUgcm9sZSB3aWxsIGJlIHVzZWQgdG8gcmV0cmlldmUgY29udGFpbmVyIGltYWdlcyBmcm9tIEVDUiBhbmQgY3JlYXRlIENsb3VkV2F0Y2ggbG9nIGdyb3Vwcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbiBleGVjdXRpb24gcm9sZSB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgY3JlYXRlZCBpZiB5b3UgdXNlIEVDUiBpbWFnZXMgaW4geW91ciB0YXNrIGRlZmluaXRpb24uXG4gICAqL1xuICByZWFkb25seSBleGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgSUFNIHJvbGUgdGhhdCBncmFudHMgY29udGFpbmVycyBpbiB0aGUgdGFzayBwZXJtaXNzaW9uIHRvIGNhbGwgQVdTIEFQSXMgb24geW91ciBiZWhhbGYuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSB0YXNrIHJvbGUgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkIGZvciB5b3UuXG4gICAqL1xuICByZWFkb25seSB0YXNrUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIGNvbmZpZ3VyYXRpb24gZGV0YWlscyBmb3IgdGhlIEFwcCBNZXNoIHByb3h5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHByb3h5IGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICByZWFkb25seSBwcm94eUNvbmZpZ3VyYXRpb24/OiBQcm94eUNvbmZpZ3VyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIHZvbHVtZSBkZWZpbml0aW9ucyBmb3IgdGhlIHRhc2suIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAgICogW1Rhc2sgRGVmaW5pdGlvbiBQYXJhbWV0ZXIgVm9sdW1lc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvL3Rhc2tfZGVmaW5pdGlvbl9wYXJhbWV0ZXJzLmh0bWwjdm9sdW1lcykuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gdm9sdW1lcyBhcmUgcGFzc2VkIHRvIHRoZSBEb2NrZXIgZGFlbW9uIG9uIGEgY29udGFpbmVyIGluc3RhbmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lcz86IFZvbHVtZVtdO1xufVxuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciB0YXNrIGRlZmluaXRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhc2tEZWZpbml0aW9uUHJvcHMgZXh0ZW5kcyBDb21tb25UYXNrRGVmaW5pdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuZXR3b3JraW5nIG1vZGUgdG8gdXNlIGZvciB0aGUgY29udGFpbmVycyBpbiB0aGUgdGFzay5cbiAgICpcbiAgICogT24gRmFyZ2F0ZSwgdGhlIG9ubHkgc3VwcG9ydGVkIG5ldHdvcmtpbmcgbW9kZSBpcyBBd3NWcGMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTmV0d29ya01vZGUuQnJpZGdlIGZvciBFQzIgJiBFeHRlcm5hbCB0YXNrcywgQXdzVnBjIGZvciBGYXJnYXRlIHRhc2tzLlxuICAgKi9cbiAgcmVhZG9ubHkgbmV0d29ya01vZGU/OiBOZXR3b3JrTW9kZTtcblxuICAvKipcbiAgICogVGhlIHBsYWNlbWVudCBjb25zdHJhaW50cyB0byB1c2UgZm9yIHRhc2tzIGluIHRoZSBzZXJ2aWNlLlxuICAgKlxuICAgKiBZb3UgY2FuIHNwZWNpZnkgYSBtYXhpbXVtIG9mIDEwIGNvbnN0cmFpbnRzIHBlciB0YXNrICh0aGlzIGxpbWl0IGluY2x1ZGVzXG4gICAqIGNvbnN0cmFpbnRzIGluIHRoZSB0YXNrIGRlZmluaXRpb24gYW5kIHRob3NlIHNwZWNpZmllZCBhdCBydW4gdGltZSkuXG4gICAqXG4gICAqIE5vdCBzdXBwb3J0ZWQgaW4gRmFyZ2F0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwbGFjZW1lbnQgY29uc3RyYWludHMuXG4gICAqL1xuICByZWFkb25seSBwbGFjZW1lbnRDb25zdHJhaW50cz86IFBsYWNlbWVudENvbnN0cmFpbnRbXTtcblxuICAvKipcbiAgICogVGhlIHRhc2sgbGF1bmNoIHR5cGUgY29tcGF0aWJsaXR5IHJlcXVpcmVtZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eTtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBjcHUgdW5pdHMgdXNlZCBieSB0aGUgdGFzay5cbiAgICpcbiAgICogSWYgeW91IGFyZSB1c2luZyB0aGUgRUMyIGxhdW5jaCB0eXBlLCB0aGlzIGZpZWxkIGlzIG9wdGlvbmFsIGFuZCBhbnkgdmFsdWUgY2FuIGJlIHVzZWQuXG4gICAqIElmIHlvdSBhcmUgdXNpbmcgdGhlIEZhcmdhdGUgbGF1bmNoIHR5cGUsIHRoaXMgZmllbGQgaXMgcmVxdWlyZWQgYW5kIHlvdSBtdXN0IHVzZSBvbmUgb2YgdGhlIGZvbGxvd2luZyB2YWx1ZXMsXG4gICAqIHdoaWNoIGRldGVybWluZXMgeW91ciByYW5nZSBvZiB2YWxpZCB2YWx1ZXMgZm9yIHRoZSBtZW1vcnkgcGFyYW1ldGVyOlxuICAgKlxuICAgKiAyNTYgKC4yNSB2Q1BVKSAtIEF2YWlsYWJsZSBtZW1vcnkgdmFsdWVzOiA1MTIgKDAuNSBHQiksIDEwMjQgKDEgR0IpLCAyMDQ4ICgyIEdCKVxuICAgKlxuICAgKiA1MTIgKC41IHZDUFUpIC0gQXZhaWxhYmxlIG1lbW9yeSB2YWx1ZXM6IDEwMjQgKDEgR0IpLCAyMDQ4ICgyIEdCKSwgMzA3MiAoMyBHQiksIDQwOTYgKDQgR0IpXG4gICAqXG4gICAqIDEwMjQgKDEgdkNQVSkgLSBBdmFpbGFibGUgbWVtb3J5IHZhbHVlczogMjA0OCAoMiBHQiksIDMwNzIgKDMgR0IpLCA0MDk2ICg0IEdCKSwgNTEyMCAoNSBHQiksIDYxNDQgKDYgR0IpLCA3MTY4ICg3IEdCKSwgODE5MiAoOCBHQilcbiAgICpcbiAgICogMjA0OCAoMiB2Q1BVKSAtIEF2YWlsYWJsZSBtZW1vcnkgdmFsdWVzOiBCZXR3ZWVuIDQwOTYgKDQgR0IpIGFuZCAxNjM4NCAoMTYgR0IpIGluIGluY3JlbWVudHMgb2YgMTAyNCAoMSBHQilcbiAgICpcbiAgICogNDA5NiAoNCB2Q1BVKSAtIEF2YWlsYWJsZSBtZW1vcnkgdmFsdWVzOiBCZXR3ZWVuIDgxOTIgKDggR0IpIGFuZCAzMDcyMCAoMzAgR0IpIGluIGluY3JlbWVudHMgb2YgMTAyNCAoMSBHQilcbiAgICpcbiAgICogQGRlZmF1bHQgLSBDUFUgdW5pdHMgYXJlIG5vdCBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBjcHU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBhbW91bnQgKGluIE1pQikgb2YgbWVtb3J5IHVzZWQgYnkgdGhlIHRhc2suXG4gICAqXG4gICAqIElmIHVzaW5nIHRoZSBFQzIgbGF1bmNoIHR5cGUsIHRoaXMgZmllbGQgaXMgb3B0aW9uYWwgYW5kIGFueSB2YWx1ZSBjYW4gYmUgdXNlZC5cbiAgICogSWYgdXNpbmcgdGhlIEZhcmdhdGUgbGF1bmNoIHR5cGUsIHRoaXMgZmllbGQgaXMgcmVxdWlyZWQgYW5kIHlvdSBtdXN0IHVzZSBvbmUgb2YgdGhlIGZvbGxvd2luZyB2YWx1ZXMsXG4gICAqIHdoaWNoIGRldGVybWluZXMgeW91ciByYW5nZSBvZiB2YWxpZCB2YWx1ZXMgZm9yIHRoZSBjcHUgcGFyYW1ldGVyOlxuICAgKlxuICAgKiA1MTIgKDAuNSBHQiksIDEwMjQgKDEgR0IpLCAyMDQ4ICgyIEdCKSAtIEF2YWlsYWJsZSBjcHUgdmFsdWVzOiAyNTYgKC4yNSB2Q1BVKVxuICAgKlxuICAgKiAxMDI0ICgxIEdCKSwgMjA0OCAoMiBHQiksIDMwNzIgKDMgR0IpLCA0MDk2ICg0IEdCKSAtIEF2YWlsYWJsZSBjcHUgdmFsdWVzOiA1MTIgKC41IHZDUFUpXG4gICAqXG4gICAqIDIwNDggKDIgR0IpLCAzMDcyICgzIEdCKSwgNDA5NiAoNCBHQiksIDUxMjAgKDUgR0IpLCA2MTQ0ICg2IEdCKSwgNzE2OCAoNyBHQiksIDgxOTIgKDggR0IpIC0gQXZhaWxhYmxlIGNwdSB2YWx1ZXM6IDEwMjQgKDEgdkNQVSlcbiAgICpcbiAgICogQmV0d2VlbiA0MDk2ICg0IEdCKSBhbmQgMTYzODQgKDE2IEdCKSBpbiBpbmNyZW1lbnRzIG9mIDEwMjQgKDEgR0IpIC0gQXZhaWxhYmxlIGNwdSB2YWx1ZXM6IDIwNDggKDIgdkNQVSlcbiAgICpcbiAgICogQmV0d2VlbiA4MTkyICg4IEdCKSBhbmQgMzA3MjAgKDMwIEdCKSBpbiBpbmNyZW1lbnRzIG9mIDEwMjQgKDEgR0IpIC0gQXZhaWxhYmxlIGNwdSB2YWx1ZXM6IDQwOTYgKDQgdkNQVSlcbiAgICpcbiAgICogQGRlZmF1bHQgLSBNZW1vcnkgdXNlZCBieSB0YXNrIGlzIG5vdCBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBtZW1vcnlNaUI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJUEMgcmVzb3VyY2UgbmFtZXNwYWNlIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lcnMgaW4gdGhlIHRhc2suXG4gICAqXG4gICAqIE5vdCBzdXBwb3J0ZWQgaW4gRmFyZ2F0ZSBhbmQgV2luZG93cyBjb250YWluZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElwY01vZGUgdXNlZCBieSB0aGUgdGFzayBpcyBub3Qgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSBpcGNNb2RlPzogSXBjTW9kZTtcblxuICAvKipcbiAgICogVGhlIHByb2Nlc3MgbmFtZXNwYWNlIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lcnMgaW4gdGhlIHRhc2suXG4gICAqXG4gICAqIE5vdCBzdXBwb3J0ZWQgaW4gRmFyZ2F0ZSBhbmQgV2luZG93cyBjb250YWluZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFBpZE1vZGUgdXNlZCBieSB0aGUgdGFzayBpcyBub3Qgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSBwaWRNb2RlPzogUGlkTW9kZTtcblxuICAvKipcbiAgICogVGhlIGluZmVyZW5jZSBhY2NlbGVyYXRvcnMgdG8gdXNlIGZvciB0aGUgY29udGFpbmVycyBpbiB0aGUgdGFzay5cbiAgICpcbiAgICogTm90IHN1cHBvcnRlZCBpbiBGYXJnYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGluZmVyZW5jZSBhY2NlbGVyYXRvcnMuXG4gICAqL1xuICByZWFkb25seSBpbmZlcmVuY2VBY2NlbGVyYXRvcnM/OiBJbmZlcmVuY2VBY2NlbGVyYXRvcltdO1xuXG4gIC8qKlxuICAgKiBUaGUgYW1vdW50IChpbiBHaUIpIG9mIGVwaGVtZXJhbCBzdG9yYWdlIHRvIGJlIGFsbG9jYXRlZCB0byB0aGUgdGFzay5cbiAgICpcbiAgICogT25seSBzdXBwb3J0ZWQgaW4gRmFyZ2F0ZSBwbGF0Zm9ybSB2ZXJzaW9uIDEuNC4wIG9yIGxhdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFVuZGVmaW5lZCwgaW4gd2hpY2ggY2FzZSwgdGhlIHRhc2sgd2lsbCByZWNlaXZlIDIwR2lCIGVwaGVtZXJhbCBzdG9yYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgZXBoZW1lcmFsU3RvcmFnZUdpQj86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG9wZXJhdGluZyBzeXN0ZW0gdGhhdCB5b3VyIHRhc2sgZGVmaW5pdGlvbnMgYXJlIHJ1bm5pbmcgb24uXG4gICAqIEEgcnVudGltZVBsYXRmb3JtIGlzIHN1cHBvcnRlZCBvbmx5IGZvciB0YXNrcyB1c2luZyB0aGUgRmFyZ2F0ZSBsYXVuY2ggdHlwZS5cbiAgICpcbiAgICpcbiAgICogQGRlZmF1bHQgLSBVbmRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBydW50aW1lUGxhdGZvcm0/OiBSdW50aW1lUGxhdGZvcm07XG59XG5cbi8qKlxuICogVGhlIGNvbW1vbiB0YXNrIGRlZmluaXRpb24gYXR0cmlidXRlcyB1c2VkIGFjcm9zcyBhbGwgdHlwZXMgb2YgdGFzayBkZWZpbml0aW9ucy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21tb25UYXNrRGVmaW5pdGlvbkF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIGFybiBvZiB0aGUgdGFzayBkZWZpbml0aW9uXG4gICAqL1xuICByZWFkb25seSB0YXNrRGVmaW5pdGlvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmV0d29ya2luZyBtb2RlIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lcnMgaW4gdGhlIHRhc2suXG4gICAqXG4gICAqIEBkZWZhdWx0IE5ldHdvcmsgbW9kZSBjYW5ub3QgYmUgcHJvdmlkZWQgdG8gdGhlIGltcG9ydGVkIHRhc2suXG4gICAqL1xuICByZWFkb25seSBuZXR3b3JrTW9kZT86IE5ldHdvcmtNb2RlO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgSUFNIHJvbGUgdGhhdCBncmFudHMgY29udGFpbmVycyBpbiB0aGUgdGFzayBwZXJtaXNzaW9uIHRvIGNhbGwgQVdTIEFQSXMgb24geW91ciBiZWhhbGYuXG4gICAqXG4gICAqIEBkZWZhdWx0IFBlcm1pc3Npb25zIGNhbm5vdCBiZSBncmFudGVkIHRvIHRoZSBpbXBvcnRlZCB0YXNrLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFza1JvbGU/OiBpYW0uSVJvbGU7XG59XG5cbi8qKlxuICogIEEgcmVmZXJlbmNlIHRvIGFuIGV4aXN0aW5nIHRhc2sgZGVmaW5pdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhc2tEZWZpbml0aW9uQXR0cmlidXRlcyBleHRlbmRzIENvbW1vblRhc2tEZWZpbml0aW9uQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBXaGF0IGxhdW5jaCB0eXBlcyB0aGlzIHRhc2sgZGVmaW5pdGlvbiBzaG91bGQgYmUgY29tcGF0aWJsZSB3aXRoLlxuICAgKlxuICAgKiBAZGVmYXVsdCBDb21wYXRpYmlsaXR5LkVDMl9BTkRfRkFSR0FURVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcGF0aWJpbGl0eT86IENvbXBhdGliaWxpdHk7XG59XG5cbmFic3RyYWN0IGNsYXNzIFRhc2tEZWZpbml0aW9uQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVRhc2tEZWZpbml0aW9uIHtcblxuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eTtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IG5ldHdvcmtNb2RlOiBOZXR3b3JrTW9kZTtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0YXNrUm9sZTogaWFtLklSb2xlO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZXhlY3V0aW9uUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogUmV0dXJuIHRydWUgaWYgdGhlIHRhc2sgZGVmaW5pdGlvbiBjYW4gYmUgcnVuIG9uIGFuIEVDMiBjbHVzdGVyXG4gICAqL1xuICBwdWJsaWMgZ2V0IGlzRWMyQ29tcGF0aWJsZSgpOiBib29sZWFuIHtcbiAgICByZXR1cm4gaXNFYzJDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIHRydWUgaWYgdGhlIHRhc2sgZGVmaW5pdGlvbiBjYW4gYmUgcnVuIG9uIGEgRmFyZ2F0ZSBjbHVzdGVyXG4gICAqL1xuICBwdWJsaWMgZ2V0IGlzRmFyZ2F0ZUNvbXBhdGlibGUoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIGlzRmFyZ2F0ZUNvbXBhdGlibGUodGhpcy5jb21wYXRpYmlsaXR5KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdHJ1ZSBpZiB0aGUgdGFzayBkZWZpbml0aW9uIGNhbiBiZSBydW4gb24gYSBFQ1MgYW55d2hlcmUgY2x1c3RlclxuICAgKi9cbiAgcHVibGljIGdldCBpc0V4dGVybmFsQ29tcGF0aWJsZSgpOiBib29sZWFuIHtcbiAgICByZXR1cm4gaXNFeHRlcm5hbENvbXBhdGlibGUodGhpcy5jb21wYXRpYmlsaXR5KTtcbiAgfVxufVxuXG4vKipcbiAqIFRoZSBiYXNlIGNsYXNzIGZvciBhbGwgdGFzayBkZWZpbml0aW9ucy5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhc2tEZWZpbml0aW9uIGV4dGVuZHMgVGFza0RlZmluaXRpb25CYXNlIHtcblxuICAvKipcbiAgICogSW1wb3J0cyBhIHRhc2sgZGVmaW5pdGlvbiBmcm9tIHRoZSBzcGVjaWZpZWQgdGFzayBkZWZpbml0aW9uIEFSTi5cbiAgICpcbiAgICogVGhlIHRhc2sgd2lsbCBoYXZlIGEgY29tcGF0aWJpbGl0eSBvZiBFQzIrRmFyZ2F0ZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRhc2tEZWZpbml0aW9uQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHRhc2tEZWZpbml0aW9uQXJuOiBzdHJpbmcpOiBJVGFza0RlZmluaXRpb24ge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRUYXNrRGVmaW5pdGlvbihzY29wZSwgaWQsIHsgdGFza0RlZmluaXRpb25Bcm46IHRhc2tEZWZpbml0aW9uQXJuIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHRhc2sgZGVmaW5pdGlvbiBmcm9tIGEgdGFzayBkZWZpbml0aW9uIHJlZmVyZW5jZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVGFza0RlZmluaXRpb25BdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBUYXNrRGVmaW5pdGlvbkF0dHJpYnV0ZXMpOiBJVGFza0RlZmluaXRpb24ge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRUYXNrRGVmaW5pdGlvbihzY29wZSwgaWQsIHtcbiAgICAgIHRhc2tEZWZpbml0aW9uQXJuOiBhdHRycy50YXNrRGVmaW5pdGlvbkFybixcbiAgICAgIGNvbXBhdGliaWxpdHk6IGF0dHJzLmNvbXBhdGliaWxpdHksXG4gICAgICBuZXR3b3JrTW9kZTogYXR0cnMubmV0d29ya01vZGUsXG4gICAgICB0YXNrUm9sZTogYXR0cnMudGFza1JvbGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgYSBmYW1pbHkgdGhhdCB0aGlzIHRhc2sgZGVmaW5pdGlvbiBpcyByZWdpc3RlcmVkIHRvLlxuICAgKiBBIGZhbWlseSBncm91cHMgbXVsdGlwbGUgdmVyc2lvbnMgb2YgYSB0YXNrIGRlZmluaXRpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZmFtaWx5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBmdWxsIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIG9mIHRoZSB0YXNrIGRlZmluaXRpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YXNrRGVmaW5pdGlvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgSUFNIHJvbGUgdGhhdCBncmFudHMgY29udGFpbmVycyBpbiB0aGUgdGFzayBwZXJtaXNzaW9uIHRvIGNhbGwgQVdTIEFQSXMgb24geW91ciBiZWhhbGYuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGFza1JvbGU6IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIG5ldHdvcmtpbmcgbW9kZSB0byB1c2UgZm9yIHRoZSBjb250YWluZXJzIGluIHRoZSB0YXNrLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5ldHdvcmtNb2RlOiBOZXR3b3JrTW9kZTtcblxuICAvKipcbiAgICogRGVmYXVsdCBjb250YWluZXIgZm9yIHRoaXMgdGFza1xuICAgKlxuICAgKiBMb2FkIGJhbGFuY2VycyB3aWxsIHNlbmQgdHJhZmZpYyB0byB0aGlzIGNvbnRhaW5lci4gVGhlIGZpcnN0XG4gICAqIGVzc2VudGlhbCBjb250YWluZXIgdGhhdCBpcyBhZGRlZCB0byB0aGlzIHRhc2sgd2lsbCBiZWNvbWUgdGhlIGRlZmF1bHRcbiAgICogY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIGRlZmF1bHRDb250YWluZXI/OiBDb250YWluZXJEZWZpbml0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdGFzayBsYXVuY2ggdHlwZSBjb21wYXRpYmlsaXR5IHJlcXVpcmVtZW50LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbXBhdGliaWxpdHk6IENvbXBhdGliaWxpdHk7XG5cbiAgLyoqXG4gICAqIFRoZSBhbW91bnQgKGluIEdpQikgb2YgZXBoZW1lcmFsIHN0b3JhZ2UgdG8gYmUgYWxsb2NhdGVkIHRvIHRoZSB0YXNrLlxuICAgKlxuICAgKiBPbmx5IHN1cHBvcnRlZCBpbiBGYXJnYXRlIHBsYXRmb3JtIHZlcnNpb24gMS40LjAgb3IgbGF0ZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXBoZW1lcmFsU3RvcmFnZUdpQj86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGNvbnRhaW5lciBkZWZpbml0aW9ucy5cbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBjb250YWluZXJzID0gbmV3IEFycmF5PENvbnRhaW5lckRlZmluaXRpb24+KCk7XG5cbiAgLyoqXG4gICAqIEFsbCB2b2x1bWVzXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHZvbHVtZXM6IFZvbHVtZVtdID0gW107XG5cbiAgLyoqXG4gICAqIFBsYWNlbWVudCBjb25zdHJhaW50cyBmb3IgdGFzayBpbnN0YW5jZXNcbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcGxhY2VtZW50Q29uc3RyYWludHMgPSBuZXcgQXJyYXk8Q2ZuVGFza0RlZmluaXRpb24uVGFza0RlZmluaXRpb25QbGFjZW1lbnRDb25zdHJhaW50UHJvcGVydHk+KCk7XG5cbiAgLyoqXG4gICAqIEluZmVyZW5jZSBhY2NlbGVyYXRvcnMgZm9yIHRhc2sgaW5zdGFuY2VzXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IF9pbmZlcmVuY2VBY2NlbGVyYXRvcnM6IEluZmVyZW5jZUFjY2VsZXJhdG9yW10gPSBbXTtcblxuICBwcml2YXRlIF9leGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuXG4gIHByaXZhdGUgX3JlZmVyZW5jZXNTZWNyZXRKc29uRmllbGQ/OiBib29sZWFuO1xuXG4gIHByaXZhdGUgcnVudGltZVBsYXRmb3JtPzogUnVudGltZVBsYXRmb3JtO1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBUYXNrRGVmaW5pdGlvbiBjbGFzcy5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBUYXNrRGVmaW5pdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuZmFtaWx5ID0gcHJvcHMuZmFtaWx5IHx8IE5hbWVzLnVuaXF1ZUlkKHRoaXMpO1xuICAgIHRoaXMuY29tcGF0aWJpbGl0eSA9IHByb3BzLmNvbXBhdGliaWxpdHk7XG5cbiAgICBpZiAocHJvcHMudm9sdW1lcykge1xuICAgICAgcHJvcHMudm9sdW1lcy5mb3JFYWNoKHYgPT4gdGhpcy5hZGRWb2x1bWUodikpO1xuICAgIH1cblxuICAgIHRoaXMubmV0d29ya01vZGUgPSBwcm9wcy5uZXR3b3JrTW9kZSA/PyAodGhpcy5pc0ZhcmdhdGVDb21wYXRpYmxlID8gTmV0d29ya01vZGUuQVdTX1ZQQyA6IE5ldHdvcmtNb2RlLkJSSURHRSk7XG4gICAgaWYgKHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSAmJiB0aGlzLm5ldHdvcmtNb2RlICE9PSBOZXR3b3JrTW9kZS5BV1NfVlBDKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZhcmdhdGUgdGFza3MgY2FuIG9ubHkgaGF2ZSBBd3NWcGMgbmV0d29yayBtb2RlLCBnb3Q6ICR7dGhpcy5uZXR3b3JrTW9kZX1gKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLnByb3h5Q29uZmlndXJhdGlvbiAmJiB0aGlzLm5ldHdvcmtNb2RlICE9PSBOZXR3b3JrTW9kZS5BV1NfVlBDKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFByb3h5Q29uZmlndXJhdGlvbiBjYW4gb25seSBiZSB1c2VkIHdpdGggQXdzVnBjIG5ldHdvcmsgbW9kZSwgZ290OiAke3RoaXMubmV0d29ya01vZGV9YCk7XG4gICAgfVxuICAgIGlmIChwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cyAmJiBwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cy5sZW5ndGggPiAwICYmIHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc2V0IHBsYWNlbWVudCBjb25zdHJhaW50cyBvbiB0YXNrcyB0aGF0IHJ1biBvbiBGYXJnYXRlJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSAmJiAoIXByb3BzLmNwdSB8fCAhcHJvcHMubWVtb3J5TWlCKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBGYXJnYXRlLWNvbXBhdGlibGUgdGFza3MgcmVxdWlyZSBib3RoIENQVSAoJHtwcm9wcy5jcHV9KSBhbmQgbWVtb3J5ICgke3Byb3BzLm1lbW9yeU1pQn0pIHNwZWNpZmljYXRpb25zYCk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmluZmVyZW5jZUFjY2VsZXJhdG9ycyAmJiBwcm9wcy5pbmZlcmVuY2VBY2NlbGVyYXRvcnMubGVuZ3RoID4gMCAmJiB0aGlzLmlzRmFyZ2F0ZUNvbXBhdGlibGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHVzZSBpbmZlcmVuY2UgYWNjZWxlcmF0b3JzIG9uIHRhc2tzIHRoYXQgcnVuIG9uIEZhcmdhdGUnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc0V4dGVybmFsQ29tcGF0aWJsZSAmJiB0aGlzLm5ldHdvcmtNb2RlICE9PSBOZXR3b3JrTW9kZS5CUklER0UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRXh0ZXJuYWwgdGFza3MgY2FuIG9ubHkgaGF2ZSBCcmlkZ2UgbmV0d29yayBtb2RlLCBnb3Q6ICR7dGhpcy5uZXR3b3JrTW9kZX1gKTtcbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSAmJiBwcm9wcy5ydW50aW1lUGxhdGZvcm0pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNwZWNpZnkgcnVudGltZVBsYXRmb3JtIGluIG5vbi1GYXJnYXRlIGNvbXBhdGlibGUgdGFza3MnKTtcbiAgICB9XG5cbiAgICB0aGlzLl9leGVjdXRpb25Sb2xlID0gcHJvcHMuZXhlY3V0aW9uUm9sZTtcblxuICAgIHRoaXMudGFza1JvbGUgPSBwcm9wcy50YXNrUm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ1Rhc2tSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Vjcy10YXNrcy5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuaW5mZXJlbmNlQWNjZWxlcmF0b3JzKSB7XG4gICAgICBwcm9wcy5pbmZlcmVuY2VBY2NlbGVyYXRvcnMuZm9yRWFjaChpYSA9PiB0aGlzLmFkZEluZmVyZW5jZUFjY2VsZXJhdG9yKGlhKSk7XG4gICAgfVxuXG4gICAgdGhpcy5lcGhlbWVyYWxTdG9yYWdlR2lCID0gcHJvcHMuZXBoZW1lcmFsU3RvcmFnZUdpQjtcblxuICAgIC8vIHZhbGlkYXRlIHRoZSBjcHUgYW5kIG1lbW9yeSBzaXplIGZvciB0aGUgV2luZG93cyBvcGVyYXRpb24gc3lzdGVtIGZhbWlseS5cbiAgICBpZiAocHJvcHMucnVudGltZVBsYXRmb3JtPy5vcGVyYXRpbmdTeXN0ZW1GYW1pbHk/Ll9vcGVyYXRpbmdTeXN0ZW1GYW1pbHkuaW5jbHVkZXMoJ1dJTkRPV1MnKSkge1xuICAgICAgLy8gV2Uga25vdyB0aGF0IHByb3BzLmNwdSBhbmQgcHJvcHMubWVtb3J5TWlCIGFyZSBkZWZpbmVkIGJlY2F1c2UgYW4gZXJyb3Igd291bGQgaGF2ZSBiZWVuIHRocm93biBwcmV2aW91c2x5IGlmIHRoZXkgd2VyZSBub3QuXG4gICAgICAvLyBCdXQsIHR5cGVzY3JpcHQgaXMgbm90IGFibGUgdG8gZmlndXJlIHRoaXMgb3V0LCBzbyB1c2luZyB0aGUgYCFgIG9wZXJhdG9yIGhlcmUgdG8gbGV0IHRoZSB0eXBlLWNoZWNrZXIga25vdyB0aGV5IGFyZSBkZWZpbmVkLlxuICAgICAgdGhpcy5jaGVja0ZhcmdhdGVXaW5kb3dzQmFzZWRUYXNrc1NpemUocHJvcHMuY3B1ISwgcHJvcHMubWVtb3J5TWlCISwgcHJvcHMucnVudGltZVBsYXRmb3JtISk7XG4gICAgfVxuXG4gICAgdGhpcy5ydW50aW1lUGxhdGZvcm0gPSBwcm9wcy5ydW50aW1lUGxhdGZvcm07XG5cbiAgICBjb25zdCB0YXNrRGVmID0gbmV3IENmblRhc2tEZWZpbml0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNvbnRhaW5lckRlZmluaXRpb25zOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ29udGFpbmVycygpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICB2b2x1bWVzOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyVm9sdW1lcygpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICBleGVjdXRpb25Sb2xlQXJuOiBMYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZXhlY3V0aW9uUm9sZSAmJiB0aGlzLmV4ZWN1dGlvblJvbGUucm9sZUFybiB9KSxcbiAgICAgIGZhbWlseTogdGhpcy5mYW1pbHksXG4gICAgICB0YXNrUm9sZUFybjogdGhpcy50YXNrUm9sZS5yb2xlQXJuLFxuICAgICAgcmVxdWlyZXNDb21wYXRpYmlsaXRpZXM6IFtcbiAgICAgICAgLi4uKGlzRWMyQ29tcGF0aWJsZShwcm9wcy5jb21wYXRpYmlsaXR5KSA/IFsnRUMyJ10gOiBbXSksXG4gICAgICAgIC4uLihpc0ZhcmdhdGVDb21wYXRpYmxlKHByb3BzLmNvbXBhdGliaWxpdHkpID8gWydGQVJHQVRFJ10gOiBbXSksXG4gICAgICAgIC4uLihpc0V4dGVybmFsQ29tcGF0aWJsZShwcm9wcy5jb21wYXRpYmlsaXR5KSA/IFsnRVhURVJOQUwnXSA6IFtdKSxcbiAgICAgIF0sXG4gICAgICBuZXR3b3JrTW9kZTogdGhpcy5yZW5kZXJOZXR3b3JrTW9kZSh0aGlzLm5ldHdvcmtNb2RlKSxcbiAgICAgIHBsYWNlbWVudENvbnN0cmFpbnRzOiBMYXp5LmFueSh7XG4gICAgICAgIHByb2R1Y2U6ICgpID0+XG4gICAgICAgICAgIWlzRmFyZ2F0ZUNvbXBhdGlibGUodGhpcy5jb21wYXRpYmlsaXR5KSA/IHRoaXMucGxhY2VtZW50Q29uc3RyYWludHMgOiB1bmRlZmluZWQsXG4gICAgICB9LCB7IG9taXRFbXB0eUFycmF5OiB0cnVlIH0pLFxuICAgICAgcHJveHlDb25maWd1cmF0aW9uOiBwcm9wcy5wcm94eUNvbmZpZ3VyYXRpb24gPyBwcm9wcy5wcm94eUNvbmZpZ3VyYXRpb24uYmluZCh0aGlzLnN0YWNrLCB0aGlzKSA6IHVuZGVmaW5lZCxcbiAgICAgIGNwdTogcHJvcHMuY3B1LFxuICAgICAgbWVtb3J5OiBwcm9wcy5tZW1vcnlNaUIsXG4gICAgICBpcGNNb2RlOiBwcm9wcy5pcGNNb2RlLFxuICAgICAgcGlkTW9kZTogcHJvcHMucGlkTW9kZSxcbiAgICAgIGluZmVyZW5jZUFjY2VsZXJhdG9yczogTGF6eS5hbnkoe1xuICAgICAgICBwcm9kdWNlOiAoKSA9PlxuICAgICAgICAgICFpc0ZhcmdhdGVDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSkgPyB0aGlzLnJlbmRlckluZmVyZW5jZUFjY2VsZXJhdG9ycygpIDogdW5kZWZpbmVkLFxuICAgICAgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9KSxcbiAgICAgIGVwaGVtZXJhbFN0b3JhZ2U6IHRoaXMuZXBoZW1lcmFsU3RvcmFnZUdpQiA/IHtcbiAgICAgICAgc2l6ZUluR2lCOiB0aGlzLmVwaGVtZXJhbFN0b3JhZ2VHaUIsXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgcnVudGltZVBsYXRmb3JtOiB0aGlzLmlzRmFyZ2F0ZUNvbXBhdGlibGUgJiYgdGhpcy5ydW50aW1lUGxhdGZvcm0gPyB7XG4gICAgICAgIGNwdUFyY2hpdGVjdHVyZTogdGhpcy5ydW50aW1lUGxhdGZvcm0/LmNwdUFyY2hpdGVjdHVyZT8uX2NwdUFyY2hpdGVjdHVyZSxcbiAgICAgICAgb3BlcmF0aW5nU3lzdGVtRmFtaWx5OiB0aGlzLnJ1bnRpbWVQbGF0Zm9ybT8ub3BlcmF0aW5nU3lzdGVtRmFtaWx5Py5fb3BlcmF0aW5nU3lzdGVtRmFtaWx5LFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cykge1xuICAgICAgcHJvcHMucGxhY2VtZW50Q29uc3RyYWludHMuZm9yRWFjaChwYyA9PiB0aGlzLmFkZFBsYWNlbWVudENvbnN0cmFpbnQocGMpKTtcbiAgICB9XG5cbiAgICB0aGlzLnRhc2tEZWZpbml0aW9uQXJuID0gdGFza0RlZi5yZWY7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGV4ZWN1dGlvblJvbGUoKTogaWFtLklSb2xlIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fZXhlY3V0aW9uUm9sZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQdWJsaWMgZ2V0dGVyIG1ldGhvZCB0byBhY2Nlc3MgbGlzdCBvZiBpbmZlcmVuY2UgYWNjZWxlcmF0b3JzIGF0dGFjaGVkIHRvIHRoZSBpbnN0YW5jZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgaW5mZXJlbmNlQWNjZWxlcmF0b3JzKCk6IEluZmVyZW5jZUFjY2VsZXJhdG9yW10ge1xuICAgIHJldHVybiB0aGlzLl9pbmZlcmVuY2VBY2NlbGVyYXRvcnM7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclZvbHVtZXMoKTogQ2ZuVGFza0RlZmluaXRpb24uVm9sdW1lUHJvcGVydHlbXSB7XG4gICAgcmV0dXJuIHRoaXMudm9sdW1lcy5tYXAocmVuZGVyVm9sdW1lKTtcblxuICAgIGZ1bmN0aW9uIHJlbmRlclZvbHVtZShzcGVjOiBWb2x1bWUpOiBDZm5UYXNrRGVmaW5pdGlvbi5Wb2x1bWVQcm9wZXJ0eSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBob3N0OiBzcGVjLmhvc3QsXG4gICAgICAgIG5hbWU6IHNwZWMubmFtZSxcbiAgICAgICAgZG9ja2VyVm9sdW1lQ29uZmlndXJhdGlvbjogc3BlYy5kb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uICYmIHtcbiAgICAgICAgICBhdXRvcHJvdmlzaW9uOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24uYXV0b3Byb3Zpc2lvbixcbiAgICAgICAgICBkcml2ZXI6IHNwZWMuZG9ja2VyVm9sdW1lQ29uZmlndXJhdGlvbi5kcml2ZXIsXG4gICAgICAgICAgZHJpdmVyT3B0czogc3BlYy5kb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uLmRyaXZlck9wdHMsXG4gICAgICAgICAgbGFiZWxzOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24ubGFiZWxzLFxuICAgICAgICAgIHNjb3BlOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24uc2NvcGUsXG4gICAgICAgIH0sXG4gICAgICAgIGVmc1ZvbHVtZUNvbmZpZ3VyYXRpb246IHNwZWMuZWZzVm9sdW1lQ29uZmlndXJhdGlvbiAmJiB7XG4gICAgICAgICAgZmlsZVN5c3RlbUlkOiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24uZmlsZVN5c3RlbUlkLFxuICAgICAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHNwZWMuZWZzVm9sdW1lQ29uZmlndXJhdGlvbi5hdXRob3JpemF0aW9uQ29uZmlnLFxuICAgICAgICAgIHJvb3REaXJlY3Rvcnk6IHNwZWMuZWZzVm9sdW1lQ29uZmlndXJhdGlvbi5yb290RGlyZWN0b3J5LFxuICAgICAgICAgIHRyYW5zaXRFbmNyeXB0aW9uOiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24udHJhbnNpdEVuY3J5cHRpb24sXG4gICAgICAgICAgdHJhbnNpdEVuY3J5cHRpb25Qb3J0OiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24udHJhbnNpdEVuY3J5cHRpb25Qb3J0LFxuXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVySW5mZXJlbmNlQWNjZWxlcmF0b3JzKCk6IENmblRhc2tEZWZpbml0aW9uLkluZmVyZW5jZUFjY2VsZXJhdG9yUHJvcGVydHlbXSB7XG4gICAgcmV0dXJuIHRoaXMuX2luZmVyZW5jZUFjY2VsZXJhdG9ycy5tYXAocmVuZGVySW5mZXJlbmNlQWNjZWxlcmF0b3IpO1xuXG4gICAgZnVuY3Rpb24gcmVuZGVySW5mZXJlbmNlQWNjZWxlcmF0b3IoaW5mZXJlbmNlQWNjZWxlcmF0b3I6IEluZmVyZW5jZUFjY2VsZXJhdG9yKSA6IENmblRhc2tEZWZpbml0aW9uLkluZmVyZW5jZUFjY2VsZXJhdG9yUHJvcGVydHkge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgZGV2aWNlTmFtZTogaW5mZXJlbmNlQWNjZWxlcmF0b3IuZGV2aWNlTmFtZSxcbiAgICAgICAgZGV2aWNlVHlwZTogaW5mZXJlbmNlQWNjZWxlcmF0b3IuZGV2aWNlVHlwZSxcbiAgICAgIH07XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoZSBleGlzdGVuY2Ugb2YgdGhlIGlucHV0IHRhcmdldCBhbmQgc2V0IGRlZmF1bHQgdmFsdWVzLlxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdmFsaWRhdGVUYXJnZXQob3B0aW9uczogTG9hZEJhbGFuY2VyVGFyZ2V0T3B0aW9ucyk6IExvYWRCYWxhbmNlclRhcmdldCB7XG4gICAgY29uc3QgdGFyZ2V0Q29udGFpbmVyID0gdGhpcy5maW5kQ29udGFpbmVyKG9wdGlvbnMuY29udGFpbmVyTmFtZSk7XG4gICAgaWYgKHRhcmdldENvbnRhaW5lciA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE5vIGNvbnRhaW5lciBuYW1lZCAnJHtvcHRpb25zLmNvbnRhaW5lck5hbWV9Jy4gRGlkIHlvdSBjYWxsIFwiYWRkQ29udGFpbmVyKClcIj9gKTtcbiAgICB9XG4gICAgY29uc3QgdGFyZ2V0UHJvdG9jb2wgPSBvcHRpb25zLnByb3RvY29sIHx8IFByb3RvY29sLlRDUDtcbiAgICBjb25zdCB0YXJnZXRDb250YWluZXJQb3J0ID0gb3B0aW9ucy5jb250YWluZXJQb3J0IHx8IHRhcmdldENvbnRhaW5lci5jb250YWluZXJQb3J0O1xuICAgIGNvbnN0IHBvcnRNYXBwaW5nID0gdGFyZ2V0Q29udGFpbmVyLmZpbmRQb3J0TWFwcGluZyh0YXJnZXRDb250YWluZXJQb3J0LCB0YXJnZXRQcm90b2NvbCk7XG4gICAgaWYgKHBvcnRNYXBwaW5nID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENvbnRhaW5lciAnJHt0YXJnZXRDb250YWluZXJ9JyBoYXMgbm8gbWFwcGluZyBmb3IgcG9ydCAke29wdGlvbnMuY29udGFpbmVyUG9ydH0gYW5kIHByb3RvY29sICR7dGFyZ2V0UHJvdG9jb2x9LiBEaWQgeW91IGNhbGwgXCJjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKClcIj9gKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbnRhaW5lck5hbWU6IG9wdGlvbnMuY29udGFpbmVyTmFtZSxcbiAgICAgIHBvcnRNYXBwaW5nLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgcG9ydCByYW5nZSB0byBiZSBvcGVuZWQgdGhhdCBtYXRjaCB0aGUgcHJvdmlkZWQgY29udGFpbmVyIG5hbWUgYW5kIGNvbnRhaW5lciBwb3J0LlxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfcG9ydFJhbmdlRnJvbVBvcnRNYXBwaW5nKHBvcnRNYXBwaW5nOiBQb3J0TWFwcGluZyk6IGVjMi5Qb3J0IHtcbiAgICBpZiAocG9ydE1hcHBpbmcuaG9zdFBvcnQgIT09IHVuZGVmaW5lZCAmJiBwb3J0TWFwcGluZy5ob3N0UG9ydCAhPT0gMCkge1xuICAgICAgcmV0dXJuIHBvcnRNYXBwaW5nLnByb3RvY29sID09PSBQcm90b2NvbC5VRFAgPyBlYzIuUG9ydC51ZHAocG9ydE1hcHBpbmcuaG9zdFBvcnQpIDogZWMyLlBvcnQudGNwKHBvcnRNYXBwaW5nLmhvc3RQb3J0KTtcbiAgICB9XG4gICAgaWYgKHRoaXMubmV0d29ya01vZGUgPT09IE5ldHdvcmtNb2RlLkJSSURHRSB8fCB0aGlzLm5ldHdvcmtNb2RlID09PSBOZXR3b3JrTW9kZS5OQVQpIHtcbiAgICAgIHJldHVybiBFUEhFTUVSQUxfUE9SVF9SQU5HRTtcbiAgICB9XG4gICAgcmV0dXJuIHBvcnRNYXBwaW5nLnByb3RvY29sID09PSBQcm90b2NvbC5VRFAgPyBlYzIuUG9ydC51ZHAocG9ydE1hcHBpbmcuY29udGFpbmVyUG9ydCkgOiBlYzIuUG9ydC50Y3AocG9ydE1hcHBpbmcuY29udGFpbmVyUG9ydCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHBvbGljeSBzdGF0ZW1lbnQgdG8gdGhlIHRhc2sgSUFNIHJvbGUuXG4gICAqL1xuICBwdWJsaWMgYWRkVG9UYXNrUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICB0aGlzLnRhc2tSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KHN0YXRlbWVudCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHBvbGljeSBzdGF0ZW1lbnQgdG8gdGhlIHRhc2sgZXhlY3V0aW9uIElBTSByb2xlLlxuICAgKi9cbiAgcHVibGljIGFkZFRvRXhlY3V0aW9uUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICB0aGlzLm9idGFpbkV4ZWN1dGlvblJvbGUoKS5hZGRUb1ByaW5jaXBhbFBvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgY29udGFpbmVyIHRvIHRoZSB0YXNrIGRlZmluaXRpb24uXG4gICAqL1xuICBwdWJsaWMgYWRkQ29udGFpbmVyKGlkOiBzdHJpbmcsIHByb3BzOiBDb250YWluZXJEZWZpbml0aW9uT3B0aW9ucykge1xuICAgIHJldHVybiBuZXcgQ29udGFpbmVyRGVmaW5pdGlvbih0aGlzLCBpZCwgeyB0YXNrRGVmaW5pdGlvbjogdGhpcywgLi4ucHJvcHMgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGZpcmVsZW5zIGxvZyByb3V0ZXIgdG8gdGhlIHRhc2sgZGVmaW5pdGlvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRGaXJlbGVuc0xvZ1JvdXRlcihpZDogc3RyaW5nLCBwcm9wczogRmlyZWxlbnNMb2dSb3V0ZXJEZWZpbml0aW9uT3B0aW9ucykge1xuICAgIC8vIG9ubHkgb25lIGZpcmVsZW5zIGxvZyByb3V0ZXIgaXMgYWxsb3dlZCBpbiBlYWNoIHRhc2suXG4gICAgaWYgKHRoaXMuY29udGFpbmVycy5maW5kKHggPT4geCBpbnN0YW5jZW9mIEZpcmVsZW5zTG9nUm91dGVyKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGaXJlbGVucyBsb2cgcm91dGVyIGlzIGFscmVhZHkgYWRkZWQgaW4gdGhpcyB0YXNrLicpO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgRmlyZWxlbnNMb2dSb3V0ZXIodGhpcywgaWQsIHsgdGFza0RlZmluaXRpb246IHRoaXMsIC4uLnByb3BzIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIExpbmtzIGEgY29udGFpbmVyIHRvIHRoaXMgdGFzayBkZWZpbml0aW9uLlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfbGlua0NvbnRhaW5lcihjb250YWluZXI6IENvbnRhaW5lckRlZmluaXRpb24pIHtcbiAgICB0aGlzLmNvbnRhaW5lcnMucHVzaChjb250YWluZXIpO1xuICAgIGlmICh0aGlzLmRlZmF1bHRDb250YWluZXIgPT09IHVuZGVmaW5lZCAmJiBjb250YWluZXIuZXNzZW50aWFsKSB7XG4gICAgICB0aGlzLmRlZmF1bHRDb250YWluZXIgPSBjb250YWluZXI7XG4gICAgfVxuICAgIGlmIChjb250YWluZXIucmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZCkge1xuICAgICAgdGhpcy5fcmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZCA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSB2b2x1bWUgdG8gdGhlIHRhc2sgZGVmaW5pdGlvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRWb2x1bWUodm9sdW1lOiBWb2x1bWUpIHtcbiAgICB0aGlzLnZvbHVtZXMucHVzaCh2b2x1bWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgdGhlIHNwZWNpZmllZCBwbGFjZW1lbnQgY29uc3RyYWludCB0byB0aGUgdGFzayBkZWZpbml0aW9uLlxuICAgKi9cbiAgcHVibGljIGFkZFBsYWNlbWVudENvbnN0cmFpbnQoY29uc3RyYWludDogUGxhY2VtZW50Q29uc3RyYWludCkge1xuICAgIGlmIChpc0ZhcmdhdGVDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNldCBwbGFjZW1lbnQgY29uc3RyYWludHMgb24gdGFza3MgdGhhdCBydW4gb24gRmFyZ2F0ZScpO1xuICAgIH1cbiAgICB0aGlzLnBsYWNlbWVudENvbnN0cmFpbnRzLnB1c2goLi4uY29uc3RyYWludC50b0pzb24oKSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyB0aGUgc3BlY2lmaWVkIGV4dGVuc2lvbiB0byB0aGUgdGFzayBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBFeHRlbnNpb24gY2FuIGJlIHVzZWQgdG8gYXBwbHkgYSBwYWNrYWdlZCBtb2RpZmljYXRpb24gdG9cbiAgICogYSB0YXNrIGRlZmluaXRpb24uXG4gICAqL1xuICBwdWJsaWMgYWRkRXh0ZW5zaW9uKGV4dGVuc2lvbjogSVRhc2tEZWZpbml0aW9uRXh0ZW5zaW9uKSB7XG4gICAgZXh0ZW5zaW9uLmV4dGVuZCh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFuIGluZmVyZW5jZSBhY2NlbGVyYXRvciB0byB0aGUgdGFzayBkZWZpbml0aW9uLlxuICAgKi9cbiAgcHVibGljIGFkZEluZmVyZW5jZUFjY2VsZXJhdG9yKGluZmVyZW5jZUFjY2VsZXJhdG9yOiBJbmZlcmVuY2VBY2NlbGVyYXRvcikge1xuICAgIGlmIChpc0ZhcmdhdGVDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHVzZSBpbmZlcmVuY2UgYWNjZWxlcmF0b3JzIG9uIHRhc2tzIHRoYXQgcnVuIG9uIEZhcmdhdGUnKTtcbiAgICB9XG4gICAgdGhpcy5faW5mZXJlbmNlQWNjZWxlcmF0b3JzLnB1c2goaW5mZXJlbmNlQWNjZWxlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgdGhlIHRhc2sgZXhlY3V0aW9uIElBTSByb2xlIGlmIGl0IGRvZXNuJ3QgYWxyZWFkeSBleGlzdC5cbiAgICovXG4gIHB1YmxpYyBvYnRhaW5FeGVjdXRpb25Sb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgaWYgKCF0aGlzLl9leGVjdXRpb25Sb2xlKSB7XG4gICAgICB0aGlzLl9leGVjdXRpb25Sb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdFeGVjdXRpb25Sb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWNzLXRhc2tzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgLy8gbmVlZGVkIGZvciBjcm9zcy1hY2NvdW50IGFjY2VzcyB3aXRoIFRhZ1BhcmFtZXRlckNvbnRhaW5lckltYWdlXG4gICAgICAgIHJvbGVOYW1lOiBQaHlzaWNhbE5hbWUuR0VORVJBVEVfSUZfTkVFREVELFxuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLl9leGVjdXRpb25Sb2xlO1xuICB9XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyB0YXNrIGRlZmluaXRpb24gaGFzIGF0IGxlYXN0IGEgY29udGFpbmVyIHRoYXQgcmVmZXJlbmNlcyBhXG4gICAqIHNwZWNpZmljIEpTT04gZmllbGQgb2YgYSBzZWNyZXQgc3RvcmVkIGluIFNlY3JldHMgTWFuYWdlci5cbiAgICovXG4gIHB1YmxpYyBnZXQgcmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZCgpOiBib29sZWFuIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fcmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIHRhc2sgZGVmaW5pdGlvbi5cbiAgICovXG4gIHByb3RlY3RlZCB2YWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgcmV0ID0gc3VwZXIudmFsaWRhdGUoKTtcblxuICAgIGlmIChpc0VjMkNvbXBhdGlibGUodGhpcy5jb21wYXRpYmlsaXR5KSkge1xuICAgICAgLy8gRUMyIG1vZGUgdmFsaWRhdGlvbnNcblxuICAgICAgLy8gQ29udGFpbmVyIHNpemVzXG4gICAgICBmb3IgKGNvbnN0IGNvbnRhaW5lciBvZiB0aGlzLmNvbnRhaW5lcnMpIHtcbiAgICAgICAgaWYgKCFjb250YWluZXIubWVtb3J5TGltaXRTcGVjaWZpZWQpIHtcbiAgICAgICAgICByZXQucHVzaChgRUNTIENvbnRhaW5lciAke2NvbnRhaW5lci5jb250YWluZXJOYW1lfSBtdXN0IGhhdmUgYXQgbGVhc3Qgb25lIG9mICdtZW1vcnlMaW1pdE1pQicgb3IgJ21lbW9yeVJlc2VydmF0aW9uTWlCJyBzcGVjaWZpZWRgKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGNvbnRhaW5lciB0aGF0IG1hdGNoIHRoZSBwcm92aWRlZCBjb250YWluZXJOYW1lLlxuICAgKi9cbiAgcHVibGljIGZpbmRDb250YWluZXIoY29udGFpbmVyTmFtZTogc3RyaW5nKTogQ29udGFpbmVyRGVmaW5pdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuY29udGFpbmVycy5maW5kKGMgPT4gYy5jb250YWluZXJOYW1lID09PSBjb250YWluZXJOYW1lKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyTmV0d29ya01vZGUobmV0d29ya01vZGU6IE5ldHdvcmtNb2RlKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gKG5ldHdvcmtNb2RlID09PSBOZXR3b3JrTW9kZS5OQVQpID8gdW5kZWZpbmVkIDogbmV0d29ya01vZGU7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckNvbnRhaW5lcnMoKSB7XG4gICAgLy8gYWRkIGZpcmVsZW5zIGxvZyByb3V0ZXIgY29udGFpbmVyIGlmIGFueSBhcHBsaWNhdGlvbiBjb250YWluZXIgaXMgdXNpbmcgZmlyZWxlbnMgbG9nIGRyaXZlcixcbiAgICAvLyBhbHNvIGNoZWNrIGlmIGFscmVhZHkgY3JlYXRlZCBsb2cgcm91dGVyIGNvbnRhaW5lclxuICAgIGZvciAoY29uc3QgY29udGFpbmVyIG9mIHRoaXMuY29udGFpbmVycykge1xuICAgICAgaWYgKGNvbnRhaW5lci5sb2dEcml2ZXJDb25maWcgJiYgY29udGFpbmVyLmxvZ0RyaXZlckNvbmZpZy5sb2dEcml2ZXIgPT09ICdhd3NmaXJlbGVucydcbiAgICAgICAgJiYgIXRoaXMuY29udGFpbmVycy5maW5kKHggPT4geCBpbnN0YW5jZW9mIEZpcmVsZW5zTG9nUm91dGVyKSkge1xuICAgICAgICB0aGlzLmFkZEZpcmVsZW5zTG9nUm91dGVyKCdsb2ctcm91dGVyJywge1xuICAgICAgICAgIGltYWdlOiBvYnRhaW5EZWZhdWx0Rmx1ZW50Qml0RUNSSW1hZ2UodGhpcywgY29udGFpbmVyLmxvZ0RyaXZlckNvbmZpZyksXG4gICAgICAgICAgZmlyZWxlbnNDb25maWc6IHtcbiAgICAgICAgICAgIHR5cGU6IEZpcmVsZW5zTG9nUm91dGVyVHlwZS5GTFVFTlRCSVQsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBsb2dnaW5nOiBuZXcgQXdzTG9nRHJpdmVyKHsgc3RyZWFtUHJlZml4OiAnZmlyZWxlbnMnIH0pLFxuICAgICAgICAgIG1lbW9yeVJlc2VydmF0aW9uTWlCOiA1MCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgYnJlYWs7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuY29udGFpbmVycy5tYXAoeCA9PiB4LnJlbmRlckNvbnRhaW5lckRlZmluaXRpb24oKSk7XG4gIH1cblxuICBwcml2YXRlIGNoZWNrRmFyZ2F0ZVdpbmRvd3NCYXNlZFRhc2tzU2l6ZShjcHU6IHN0cmluZywgbWVtb3J5OiBzdHJpbmcsIHJ1bnRpbWVQbGF0Zm9ybTogUnVudGltZVBsYXRmb3JtKSB7XG4gICAgaWYgKE51bWJlcihjcHUpID09PSAxMDI0KSB7XG4gICAgICBpZiAoTnVtYmVyKG1lbW9yeSkgPCAxMDI0IHx8IE51bWJlcihtZW1vcnkpID4gODE5MiB8fCAoTnVtYmVyKG1lbW9yeSklIDEwMjQgIT09IDApKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSWYgcHJvdmlkZWQgY3B1IGlzICR7Y3B1fSwgdGhlbiBtZW1vcnlNaUIgbXVzdCBoYXZlIGEgbWluIG9mIDEwMjQgYW5kIGEgbWF4IG9mIDgxOTIsIGluIDEwMjQgaW5jcmVtZW50cy4gUHJvdmlkZWQgbWVtb3J5TWlCIHdhcyAke051bWJlcihtZW1vcnkpfS5gKTtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKE51bWJlcihjcHUpID09PSAyMDQ4KSB7XG4gICAgICBpZiAoTnVtYmVyKG1lbW9yeSkgPCA0MDk2IHx8IE51bWJlcihtZW1vcnkpID4gMTYzODQgfHwgKE51bWJlcihtZW1vcnkpICUgMTAyNCAhPT0gMCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJZiBwcm92aWRlZCBjcHUgaXMgJHtjcHV9LCB0aGVuIG1lbW9yeU1pQiBtdXN0IGhhdmUgYSBtaW4gb2YgNDA5NiBhbmQgbWF4IG9mIDE2Mzg0LCBpbiAxMDI0IGluY3JlbWVudHMuIFByb3ZpZGVkIG1lbW9yeU1pQiAke051bWJlcihtZW1vcnkpfS5gKTtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKE51bWJlcihjcHUpID09PSA0MDk2KSB7XG4gICAgICBpZiAoTnVtYmVyKG1lbW9yeSkgPCA4MTkyIHx8IE51bWJlcihtZW1vcnkpID4gMzA3MjAgfHwgKE51bWJlcihtZW1vcnkpICUgMTAyNCAhPT0gMCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJZiBwcm92aWRlZCBjcHUgaXMgJHsgY3B1IH0sIHRoZW4gbWVtb3J5TWlCIG11c3QgaGF2ZSBhIG1pbiBvZiA4MTkyIGFuZCBhIG1heCBvZiAzMDcyMCwgaW4gMTAyNCBpbmNyZW1lbnRzLlByb3ZpZGVkIG1lbW9yeU1pQiB3YXMgJHsgTnVtYmVyKG1lbW9yeSkgfS5gKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJZiBvcGVyYXRpbmdTeXN0ZW1GYW1pbHkgaXMgJHtydW50aW1lUGxhdGZvcm0ub3BlcmF0aW5nU3lzdGVtRmFtaWx5IS5fb3BlcmF0aW5nU3lzdGVtRmFtaWx5fSwgdGhlbiBjcHUgbXVzdCBiZSBpbiAxMDI0ICgxIHZDUFUpLCAyMDQ4ICgyIHZDUFUpLCBvciA0MDk2ICg0IHZDUFUpLiBQcm92aWRlZCB2YWx1ZSB3YXM6ICR7Y3B1fWApO1xuICAgIH1cbiAgfTtcbn1cblxuLyoqXG4gKiBUaGUgcG9ydCByYW5nZSB0byBvcGVuIHVwIGZvciBkeW5hbWljIHBvcnQgbWFwcGluZ1xuICovXG5jb25zdCBFUEhFTUVSQUxfUE9SVF9SQU5HRSA9IGVjMi5Qb3J0LnRjcFJhbmdlKDMyNzY4LCA2NTUzNSk7XG5cbi8qKlxuICogVGhlIG5ldHdvcmtpbmcgbW9kZSB0byB1c2UgZm9yIHRoZSBjb250YWluZXJzIGluIHRoZSB0YXNrLlxuICovXG5leHBvcnQgZW51bSBOZXR3b3JrTW9kZSB7XG4gIC8qKlxuICAgKiBUaGUgdGFzaydzIGNvbnRhaW5lcnMgZG8gbm90IGhhdmUgZXh0ZXJuYWwgY29ubmVjdGl2aXR5IGFuZCBwb3J0IG1hcHBpbmdzIGNhbid0IGJlIHNwZWNpZmllZCBpbiB0aGUgY29udGFpbmVyIGRlZmluaXRpb24uXG4gICAqL1xuICBOT05FID0gJ25vbmUnLFxuXG4gIC8qKlxuICAgKiBUaGUgdGFzayB1dGlsaXplcyBEb2NrZXIncyBidWlsdC1pbiB2aXJ0dWFsIG5ldHdvcmsgd2hpY2ggcnVucyBpbnNpZGUgZWFjaCBjb250YWluZXIgaW5zdGFuY2UuXG4gICAqL1xuICBCUklER0UgPSAnYnJpZGdlJyxcblxuICAvKipcbiAgICogVGhlIHRhc2sgaXMgYWxsb2NhdGVkIGFuIGVsYXN0aWMgbmV0d29yayBpbnRlcmZhY2UuXG4gICAqL1xuICBBV1NfVlBDID0gJ2F3c3ZwYycsXG5cbiAgLyoqXG4gICAqIFRoZSB0YXNrIGJ5cGFzc2VzIERvY2tlcidzIGJ1aWx0LWluIHZpcnR1YWwgbmV0d29yayBhbmQgbWFwcyBjb250YWluZXIgcG9ydHMgZGlyZWN0bHkgdG8gdGhlIEVDMiBpbnN0YW5jZSdzIG5ldHdvcmsgaW50ZXJmYWNlIGRpcmVjdGx5LlxuICAgKlxuICAgKiBJbiB0aGlzIG1vZGUsIHlvdSBjYW4ndCBydW4gbXVsdGlwbGUgaW5zdGFudGlhdGlvbnMgb2YgdGhlIHNhbWUgdGFzayBvbiBhXG4gICAqIHNpbmdsZSBjb250YWluZXIgaW5zdGFuY2Ugd2hlbiBwb3J0IG1hcHBpbmdzIGFyZSB1c2VkLlxuICAgKi9cbiAgSE9TVCA9ICdob3N0JyxcblxuICAvKipcbiAgICogVGhlIHRhc2sgdXRpbGl6ZXMgTkFUIG5ldHdvcmsgbW9kZSByZXF1aXJlZCBieSBXaW5kb3dzIGNvbnRhaW5lcnMuXG4gICAqXG4gICAqIFRoaXMgaXMgdGhlIG9ubHkgc3VwcG9ydGVkIG5ldHdvcmsgbW9kZSBmb3IgV2luZG93cyBjb250YWluZXJzLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIFtUYXNrIERlZmluaXRpb24gUGFyYW1ldGVyc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdGFza19kZWZpbml0aW9uX3BhcmFtZXRlcnMuaHRtbCNuZXR3b3JrX21vZGUpLlxuICAgKi9cbiAgTkFUID0gJ25hdCdcbn1cblxuLyoqXG4gKiBUaGUgSVBDIHJlc291cmNlIG5hbWVzcGFjZSB0byB1c2UgZm9yIHRoZSBjb250YWluZXJzIGluIHRoZSB0YXNrLlxuICovXG5leHBvcnQgZW51bSBJcGNNb2RlIHtcbiAgLyoqXG4gICAqIElmIG5vbmUgaXMgc3BlY2lmaWVkLCB0aGVuIElQQyByZXNvdXJjZXMgd2l0aGluIHRoZSBjb250YWluZXJzIG9mIGEgdGFzayBhcmUgcHJpdmF0ZSBhbmQgbm90XG4gICAqIHNoYXJlZCB3aXRoIG90aGVyIGNvbnRhaW5lcnMgaW4gYSB0YXNrIG9yIG9uIHRoZSBjb250YWluZXIgaW5zdGFuY2VcbiAgICovXG4gIE5PTkUgPSAnbm9uZScsXG5cbiAgLyoqXG4gICAqIElmIGhvc3QgaXMgc3BlY2lmaWVkLCB0aGVuIGFsbCBjb250YWluZXJzIHdpdGhpbiB0aGUgdGFza3MgdGhhdCBzcGVjaWZpZWQgdGhlIGhvc3QgSVBDIG1vZGUgb25cbiAgICogdGhlIHNhbWUgY29udGFpbmVyIGluc3RhbmNlIHNoYXJlIHRoZSBzYW1lIElQQyByZXNvdXJjZXMgd2l0aCB0aGUgaG9zdCBBbWF6b24gRUMyIGluc3RhbmNlLlxuICAgKi9cbiAgSE9TVCA9ICdob3N0JyxcblxuICAvKipcbiAgICogSWYgdGFzayBpcyBzcGVjaWZpZWQsIGFsbCBjb250YWluZXJzIHdpdGhpbiB0aGUgc3BlY2lmaWVkIHRhc2sgc2hhcmUgdGhlIHNhbWUgSVBDIHJlc291cmNlcy5cbiAgICovXG4gIFRBU0sgPSAndGFzaycsXG59XG5cbi8qKlxuICogVGhlIHByb2Nlc3MgbmFtZXNwYWNlIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lcnMgaW4gdGhlIHRhc2suXG4gKi9cbmV4cG9ydCBlbnVtIFBpZE1vZGUge1xuICAvKipcbiAgICogSWYgaG9zdCBpcyBzcGVjaWZpZWQsIHRoZW4gYWxsIGNvbnRhaW5lcnMgd2l0aGluIHRoZSB0YXNrcyB0aGF0IHNwZWNpZmllZCB0aGUgaG9zdCBQSUQgbW9kZVxuICAgKiBvbiB0aGUgc2FtZSBjb250YWluZXIgaW5zdGFuY2Ugc2hhcmUgdGhlIHNhbWUgcHJvY2VzcyBuYW1lc3BhY2Ugd2l0aCB0aGUgaG9zdCBBbWF6b24gRUMyIGluc3RhbmNlLlxuICAgKi9cbiAgSE9TVCA9ICdob3N0JyxcblxuICAvKipcbiAgICogSWYgdGFzayBpcyBzcGVjaWZpZWQsIGFsbCBjb250YWluZXJzIHdpdGhpbiB0aGUgc3BlY2lmaWVkIHRhc2sgc2hhcmUgdGhlIHNhbWUgcHJvY2VzcyBuYW1lc3BhY2UuXG4gICAqL1xuICBUQVNLID0gJ3Rhc2snLFxufVxuXG4vKipcbiAqIEVsYXN0aWMgSW5mZXJlbmNlIEFjY2VsZXJhdG9yLlxuICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbRWxhc3RpYyBJbmZlcmVuY2UgQmFzaWNzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpYy1pbmZlcmVuY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2Jhc2ljcy5odG1sKVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluZmVyZW5jZUFjY2VsZXJhdG9yIHtcbiAgLyoqXG4gICAqIFRoZSBFbGFzdGljIEluZmVyZW5jZSBhY2NlbGVyYXRvciBkZXZpY2UgbmFtZS5cbiAgICogQGRlZmF1bHQgLSBlbXB0eVxuICAgKi9cbiAgcmVhZG9ubHkgZGV2aWNlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEVsYXN0aWMgSW5mZXJlbmNlIGFjY2VsZXJhdG9yIHR5cGUgdG8gdXNlLiBUaGUgYWxsb3dlZCB2YWx1ZXMgYXJlOiBlaWEyLm1lZGl1bSwgZWlhMi5sYXJnZSBhbmQgZWlhMi54bGFyZ2UuXG4gICAqIEBkZWZhdWx0IC0gZW1wdHlcbiAgICovXG4gIHJlYWRvbmx5IGRldmljZVR5cGU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSBkYXRhIHZvbHVtZSB1c2VkIGluIGEgdGFzayBkZWZpbml0aW9uLlxuICpcbiAqIEZvciB0YXNrcyB0aGF0IHVzZSBhIERvY2tlciB2b2x1bWUsIHNwZWNpZnkgYSBEb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uLlxuICogRm9yIHRhc2tzIHRoYXQgdXNlIGEgYmluZCBtb3VudCBob3N0IHZvbHVtZSwgc3BlY2lmeSBhIGhvc3QgYW5kIG9wdGlvbmFsIHNvdXJjZVBhdGguXG4gKlxuICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbVXNpbmcgRGF0YSBWb2x1bWVzIGluIFRhc2tzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2luZ19kYXRhX3ZvbHVtZXMuaHRtbCkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVm9sdW1lIHtcbiAgLyoqXG4gICAqIFRoaXMgcHJvcGVydHkgaXMgc3BlY2lmaWVkIHdoZW4geW91IGFyZSB1c2luZyBiaW5kIG1vdW50IGhvc3Qgdm9sdW1lcy5cbiAgICpcbiAgICogQmluZCBtb3VudCBob3N0IHZvbHVtZXMgYXJlIHN1cHBvcnRlZCB3aGVuIHlvdSBhcmUgdXNpbmcgZWl0aGVyIHRoZSBFQzIgb3IgRmFyZ2F0ZSBsYXVuY2ggdHlwZXMuXG4gICAqIFRoZSBjb250ZW50cyBvZiB0aGUgaG9zdCBwYXJhbWV0ZXIgZGV0ZXJtaW5lIHdoZXRoZXIgeW91ciBiaW5kIG1vdW50IGhvc3Qgdm9sdW1lIHBlcnNpc3RzIG9uIHRoZVxuICAgKiBob3N0IGNvbnRhaW5lciBpbnN0YW5jZSBhbmQgd2hlcmUgaXQgaXMgc3RvcmVkLiBJZiB0aGUgaG9zdCBwYXJhbWV0ZXIgaXMgZW1wdHksIHRoZW4gdGhlIERvY2tlclxuICAgKiBkYWVtb24gYXNzaWducyBhIGhvc3QgcGF0aCBmb3IgeW91ciBkYXRhIHZvbHVtZS4gSG93ZXZlciwgdGhlIGRhdGEgaXMgbm90IGd1YXJhbnRlZWQgdG8gcGVyc2lzdFxuICAgKiBhZnRlciB0aGUgY29udGFpbmVycyBhc3NvY2lhdGVkIHdpdGggaXQgc3RvcCBydW5uaW5nLlxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdD86IEhvc3Q7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSB2b2x1bWUuXG4gICAqXG4gICAqIFVwIHRvIDI1NSBsZXR0ZXJzICh1cHBlcmNhc2UgYW5kIGxvd2VyY2FzZSksIG51bWJlcnMsIGFuZCBoeXBoZW5zIGFyZSBhbGxvd2VkLlxuICAgKiBUaGlzIG5hbWUgaXMgcmVmZXJlbmNlZCBpbiB0aGUgc291cmNlVm9sdW1lIHBhcmFtZXRlciBvZiBjb250YWluZXIgZGVmaW5pdGlvbiBtb3VudFBvaW50cy5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhpcyBwcm9wZXJ0eSBpcyBzcGVjaWZpZWQgd2hlbiB5b3UgYXJlIHVzaW5nIERvY2tlciB2b2x1bWVzLlxuICAgKlxuICAgKiBEb2NrZXIgdm9sdW1lcyBhcmUgb25seSBzdXBwb3J0ZWQgd2hlbiB5b3UgYXJlIHVzaW5nIHRoZSBFQzIgbGF1bmNoIHR5cGUuXG4gICAqIFdpbmRvd3MgY29udGFpbmVycyBvbmx5IHN1cHBvcnQgdGhlIHVzZSBvZiB0aGUgbG9jYWwgZHJpdmVyLlxuICAgKiBUbyB1c2UgYmluZCBtb3VudHMsIHNwZWNpZnkgYSBob3N0IGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uPzogRG9ja2VyVm9sdW1lQ29uZmlndXJhdGlvbjtcblxuICAvKipcbiAgICogVGhpcyBwcm9wZXJ0eSBpcyBzcGVjaWZpZWQgd2hlbiB5b3UgYXJlIHVzaW5nIEFtYXpvbiBFRlMuXG4gICAqXG4gICAqIFdoZW4gc3BlY2lmeWluZyBBbWF6b24gRUZTIHZvbHVtZXMgaW4gdGFza3MgdXNpbmcgdGhlIEZhcmdhdGUgbGF1bmNoIHR5cGUsXG4gICAqIEZhcmdhdGUgY3JlYXRlcyBhIHN1cGVydmlzb3IgY29udGFpbmVyIHRoYXQgaXMgcmVzcG9uc2libGUgZm9yIG1hbmFnaW5nIHRoZSBBbWF6b24gRUZTIHZvbHVtZS5cbiAgICogVGhlIHN1cGVydmlzb3IgY29udGFpbmVyIHVzZXMgYSBzbWFsbCBhbW91bnQgb2YgdGhlIHRhc2sncyBtZW1vcnkuXG4gICAqIFRoZSBzdXBlcnZpc29yIGNvbnRhaW5lciBpcyB2aXNpYmxlIHdoZW4gcXVlcnlpbmcgdGhlIHRhc2sgbWV0YWRhdGEgdmVyc2lvbiA0IGVuZHBvaW50LFxuICAgKiBidXQgaXMgbm90IHZpc2libGUgaW4gQ2xvdWRXYXRjaCBDb250YWluZXIgSW5zaWdodHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIEVsYXN0aWMgRmlsZVN5c3RlbSBpcyBzZXR1cFxuICAgKi9cbiAgcmVhZG9ubHkgZWZzVm9sdW1lQ29uZmlndXJhdGlvbj86IEVmc1ZvbHVtZUNvbmZpZ3VyYXRpb247XG59XG5cbi8qKlxuICogVGhlIGRldGFpbHMgb24gYSBjb250YWluZXIgaW5zdGFuY2UgYmluZCBtb3VudCBob3N0IHZvbHVtZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBIb3N0IHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgcGF0aCBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2UgdGhhdCBpcyBwcmVzZW50ZWQgdG8gdGhlIGNvbnRhaW5lci5cbiAgICogSWYgdGhlIHNvdXJjZVBhdGggdmFsdWUgZG9lcyBub3QgZXhpc3Qgb24gdGhlIGhvc3QgY29udGFpbmVyIGluc3RhbmNlLCB0aGUgRG9ja2VyIGRhZW1vbiBjcmVhdGVzIGl0LlxuICAgKiBJZiB0aGUgbG9jYXRpb24gZG9lcyBleGlzdCwgdGhlIGNvbnRlbnRzIG9mIHRoZSBzb3VyY2UgcGF0aCBmb2xkZXIgYXJlIGV4cG9ydGVkLlxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IGlzIG5vdCBzdXBwb3J0ZWQgZm9yIHRhc2tzIHRoYXQgdXNlIHRoZSBGYXJnYXRlIGxhdW5jaCB0eXBlLlxuICAgKi9cbiAgcmVhZG9ubHkgc291cmNlUGF0aD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhbiBFQ1MgdGFyZ2V0LlxuICpcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIExvYWRCYWxhbmNlclRhcmdldCB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgY29udGFpbmVyTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCBtYXBwaW5nIG9mIHRoZSB0YXJnZXQuXG4gICAqL1xuICByZWFkb25seSBwb3J0TWFwcGluZzogUG9ydE1hcHBpbmdcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBkZWZpbmluZyBhbiBFQ1MgdGFyZ2V0LiBUaGUgcG9ydCBtYXBwaW5nIGZvciBpdCBtdXN0IGFscmVhZHkgaGF2ZSBiZWVuIGNyZWF0ZWQgdGhyb3VnaCBhZGRQb3J0TWFwcGluZygpLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExvYWRCYWxhbmNlclRhcmdldE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lck5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHBvcnQgbnVtYmVyIG9mIHRoZSBjb250YWluZXIuIE9ubHkgYXBwbGljYWJsZSB3aGVuIHVzaW5nIGFwcGxpY2F0aW9uL25ldHdvcmsgbG9hZCBiYWxhbmNlcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ29udGFpbmVyIHBvcnQgb2YgdGhlIGZpcnN0IGFkZGVkIHBvcnQgbWFwcGluZy5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lclBvcnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBwcm90b2NvbCB1c2VkIGZvciB0aGUgcG9ydCBtYXBwaW5nLiBPbmx5IGFwcGxpY2FibGUgd2hlbiB1c2luZyBhcHBsaWNhdGlvbiBsb2FkIGJhbGFuY2Vycy5cbiAgICpcbiAgICogQGRlZmF1bHQgUHJvdG9jb2wuVENQXG4gICAqL1xuICByZWFkb25seSBwcm90b2NvbD86IFByb3RvY29sO1xufVxuXG4vKipcbiAqIFRoZSBjb25maWd1cmF0aW9uIGZvciBhIERvY2tlciB2b2x1bWUuIERvY2tlciB2b2x1bWVzIGFyZSBvbmx5IHN1cHBvcnRlZCB3aGVuIHlvdSBhcmUgdXNpbmcgdGhlIEVDMiBsYXVuY2ggdHlwZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRoZSBEb2NrZXIgdm9sdW1lIHNob3VsZCBiZSBjcmVhdGVkIGlmIGl0IGRvZXMgbm90IGFscmVhZHkgZXhpc3QuXG4gICAqIElmIHRydWUgaXMgc3BlY2lmaWVkLCB0aGUgRG9ja2VyIHZvbHVtZSB3aWxsIGJlIGNyZWF0ZWQgZm9yIHlvdS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9wcm92aXNpb24/OiBib29sZWFuO1xuICAvKipcbiAgICogVGhlIERvY2tlciB2b2x1bWUgZHJpdmVyIHRvIHVzZS5cbiAgICovXG4gIHJlYWRvbmx5IGRyaXZlcjogc3RyaW5nO1xuICAvKipcbiAgICogQSBtYXAgb2YgRG9ja2VyIGRyaXZlci1zcGVjaWZpYyBvcHRpb25zIHBhc3NlZCB0aHJvdWdoLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBkcml2ZXJPcHRzPzoge1trZXk6IHN0cmluZ106IHN0cmluZ307XG4gIC8qKlxuICAgKiBDdXN0b20gbWV0YWRhdGEgdG8gYWRkIHRvIHlvdXIgRG9ja2VyIHZvbHVtZS5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gbGFiZWxzXG4gICAqL1xuICByZWFkb25seSBsYWJlbHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZzsgfVxuICAvKipcbiAgICogVGhlIHNjb3BlIGZvciB0aGUgRG9ja2VyIHZvbHVtZSB0aGF0IGRldGVybWluZXMgaXRzIGxpZmVjeWNsZS5cbiAgICovXG4gIHJlYWRvbmx5IHNjb3BlOiBTY29wZTtcbn1cblxuLyoqXG4gKiBUaGUgYXV0aG9yaXphdGlvbiBjb25maWd1cmF0aW9uIGRldGFpbHMgZm9yIHRoZSBBbWF6b24gRUZTIGZpbGUgc3lzdGVtLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Db25maWcge1xuICAvKipcbiAgICogVGhlIGFjY2VzcyBwb2ludCBJRCB0byB1c2UuXG4gICAqIElmIGFuIGFjY2VzcyBwb2ludCBpcyBzcGVjaWZpZWQsIHRoZSByb290IGRpcmVjdG9yeSB2YWx1ZSB3aWxsIGJlXG4gICAqIHJlbGF0aXZlIHRvIHRoZSBkaXJlY3Rvcnkgc2V0IGZvciB0aGUgYWNjZXNzIHBvaW50LlxuICAgKiBJZiBzcGVjaWZpZWQsIHRyYW5zaXQgZW5jcnlwdGlvbiBtdXN0IGJlIGVuYWJsZWQgaW4gdGhlIEVGU1ZvbHVtZUNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIGlkXG4gICAqL1xuICByZWFkb25seSBhY2Nlc3NQb2ludElkPzogc3RyaW5nO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gdXNlIHRoZSBBbWF6b24gRUNTIHRhc2sgSUFNIHJvbGUgZGVmaW5lZFxuICAgKiBpbiBhIHRhc2sgZGVmaW5pdGlvbiB3aGVuIG1vdW50aW5nIHRoZSBBbWF6b24gRUZTIGZpbGUgc3lzdGVtLlxuICAgKiBJZiBlbmFibGVkLCB0cmFuc2l0IGVuY3J5cHRpb24gbXVzdCBiZSBlbmFibGVkIGluIHRoZSBFRlNWb2x1bWVDb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBWYWxpZCB2YWx1ZXM6IEVOQUJMRUQgfCBESVNBQkxFRFxuICAgKlxuICAgKiBAZGVmYXVsdCBJZiB0aGlzIHBhcmFtZXRlciBpcyBvbWl0dGVkLCB0aGUgZGVmYXVsdCB2YWx1ZSBvZiBESVNBQkxFRCBpcyB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgaWFtPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFRoZSBjb25maWd1cmF0aW9uIGZvciBhbiBFbGFzdGljIEZpbGVTeXN0ZW0gdm9sdW1lLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogVGhlIEFtYXpvbiBFRlMgZmlsZSBzeXN0ZW0gSUQgdG8gdXNlLlxuICAgKi9cbiAgcmVhZG9ubHkgZmlsZVN5c3RlbUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGlyZWN0b3J5IHdpdGhpbiB0aGUgQW1hem9uIEVGUyBmaWxlIHN5c3RlbSB0byBtb3VudCBhcyB0aGUgcm9vdCBkaXJlY3RvcnkgaW5zaWRlIHRoZSBob3N0LlxuICAgKiBTcGVjaWZ5aW5nIC8gd2lsbCBoYXZlIHRoZSBzYW1lIGVmZmVjdCBhcyBvbWl0dGluZyB0aGlzIHBhcmFtZXRlci5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIHJvb3Qgb2YgdGhlIEFtYXpvbiBFRlMgdm9sdW1lXG4gICAqL1xuICByZWFkb25seSByb290RGlyZWN0b3J5Pzogc3RyaW5nO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZW5hYmxlIGVuY3J5cHRpb24gZm9yIEFtYXpvbiBFRlMgZGF0YSBpbiB0cmFuc2l0IGJldHdlZW5cbiAgICogdGhlIEFtYXpvbiBFQ1MgaG9zdCBhbmQgdGhlIEFtYXpvbiBFRlMgc2VydmVyLlxuICAgKiBUcmFuc2l0IGVuY3J5cHRpb24gbXVzdCBiZSBlbmFibGVkIGlmIEFtYXpvbiBFRlMgSUFNIGF1dGhvcml6YXRpb24gaXMgdXNlZC5cbiAgICpcbiAgICogVmFsaWQgdmFsdWVzOiBFTkFCTEVEIHwgRElTQUJMRURcbiAgICpcbiAgICogQGRlZmF1bHQgRElTQUJMRURcbiAgICovXG4gIHJlYWRvbmx5IHRyYW5zaXRFbmNyeXB0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHBvcnQgdG8gdXNlIHdoZW4gc2VuZGluZyBlbmNyeXB0ZWQgZGF0YSBiZXR3ZWVuXG4gICAqIHRoZSBBbWF6b24gRUNTIGhvc3QgYW5kIHRoZSBBbWF6b24gRUZTIHNlcnZlci4gRUZTIG1vdW50IGhlbHBlciB1c2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBQb3J0IHNlbGVjdGlvbiBzdHJhdGVneSB0aGF0IHRoZSBBbWF6b24gRUZTIG1vdW50IGhlbHBlciB1c2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgdHJhbnNpdEVuY3J5cHRpb25Qb3J0PzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIGF1dGhvcml6YXRpb24gY29uZmlndXJhdGlvbiBkZXRhaWxzIGZvciB0aGUgQW1hem9uIEVGUyBmaWxlIHN5c3RlbS5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvcml6YXRpb25Db25maWc/OiBBdXRob3JpemF0aW9uQ29uZmlnO1xufVxuXG4vKipcbiAqIFRoZSBzY29wZSBmb3IgdGhlIERvY2tlciB2b2x1bWUgdGhhdCBkZXRlcm1pbmVzIGl0cyBsaWZlY3ljbGUuXG4gKiBEb2NrZXIgdm9sdW1lcyB0aGF0IGFyZSBzY29wZWQgdG8gYSB0YXNrIGFyZSBhdXRvbWF0aWNhbGx5IHByb3Zpc2lvbmVkIHdoZW4gdGhlIHRhc2sgc3RhcnRzIGFuZCBkZXN0cm95ZWQgd2hlbiB0aGUgdGFzayBzdG9wcy5cbiAqIERvY2tlciB2b2x1bWVzIHRoYXQgYXJlIHNjb3BlZCBhcyBzaGFyZWQgcGVyc2lzdCBhZnRlciB0aGUgdGFzayBzdG9wcy5cbiAqL1xuZXhwb3J0IGVudW0gU2NvcGUge1xuICAvKipcbiAgICogRG9ja2VyIHZvbHVtZXMgdGhhdCBhcmUgc2NvcGVkIHRvIGEgdGFzayBhcmUgYXV0b21hdGljYWxseSBwcm92aXNpb25lZCB3aGVuIHRoZSB0YXNrIHN0YXJ0cyBhbmQgZGVzdHJveWVkIHdoZW4gdGhlIHRhc2sgc3RvcHMuXG4gICAqL1xuICBUQVNLID0gJ3Rhc2snLFxuXG4gIC8qKlxuICAgKiBEb2NrZXIgdm9sdW1lcyB0aGF0IGFyZSBzY29wZWQgYXMgc2hhcmVkIHBlcnNpc3QgYWZ0ZXIgdGhlIHRhc2sgc3RvcHMuXG4gICAqL1xuICBTSEFSRUQgPSAnc2hhcmVkJ1xufVxuXG4vKipcbiAqIFRoZSB0YXNrIGxhdW5jaCB0eXBlIGNvbXBhdGliaWxpdHkgcmVxdWlyZW1lbnQuXG4gKi9cbmV4cG9ydCBlbnVtIENvbXBhdGliaWxpdHkge1xuICAvKipcbiAgICogVGhlIHRhc2sgc2hvdWxkIHNwZWNpZnkgdGhlIEVDMiBsYXVuY2ggdHlwZS5cbiAgICovXG4gIEVDMixcblxuICAvKipcbiAgICogVGhlIHRhc2sgc2hvdWxkIHNwZWNpZnkgdGhlIEZhcmdhdGUgbGF1bmNoIHR5cGUuXG4gICAqL1xuICBGQVJHQVRFLFxuXG4gIC8qKlxuICAgKiBUaGUgdGFzayBjYW4gc3BlY2lmeSBlaXRoZXIgdGhlIEVDMiBvciBGYXJnYXRlIGxhdW5jaCB0eXBlcy5cbiAgICovXG4gIEVDMl9BTkRfRkFSR0FURSxcblxuICAvKipcbiAgICogVGhlIHRhc2sgc2hvdWxkIHNwZWNpZnkgdGhlIEV4dGVybmFsIGxhdW5jaCB0eXBlLlxuICAgKi9cbiAgRVhURVJOQUxcbn1cblxuLyoqXG4gKiBBbiBleHRlbnNpb24gZm9yIFRhc2sgRGVmaW5pdGlvbnNcbiAqXG4gKiBDbGFzc2VzIHRoYXQgd2FudCB0byBtYWtlIGNoYW5nZXMgdG8gYSBUYXNrRGVmaW5pdGlvbiAoc3VjaCBhc1xuICogYWRkaW5nIGhlbHBlciBjb250YWluZXJzKSBjYW4gaW1wbGVtZW50IHRoaXMgaW50ZXJmYWNlLCBhbmQgY2FuXG4gKiB0aGVuIGJlIFwiYWRkZWRcIiB0byBhIFRhc2tEZWZpbml0aW9uIGxpa2Ugc286XG4gKlxuICogICAgdGFza0RlZmluaXRpb24uYWRkRXh0ZW5zaW9uKG5ldyBNeUV4dGVuc2lvbihcInNvbWVfcGFyYW1ldGVyXCIpKTtcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJVGFza0RlZmluaXRpb25FeHRlbnNpb24ge1xuICAvKipcbiAgICogQXBwbHkgdGhlIGV4dGVuc2lvbiB0byB0aGUgZ2l2ZW4gVGFza0RlZmluaXRpb25cbiAgICpcbiAgICogQHBhcmFtIHRhc2tEZWZpbml0aW9uIFtkaXNhYmxlLWF3c2xpbnQ6cmVmLXZpYS1pbnRlcmZhY2VdXG4gICAqL1xuICBleHRlbmQodGFza0RlZmluaXRpb246IFRhc2tEZWZpbml0aW9uKTogdm9pZDtcbn1cblxuLyoqXG4gKiBSZXR1cm4gdHJ1ZSBpZiB0aGUgZ2l2ZW4gdGFzayBkZWZpbml0aW9uIGNhbiBiZSBydW4gb24gYW4gRUMyIGNsdXN0ZXJcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzRWMyQ29tcGF0aWJsZShjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5KTogYm9vbGVhbiB7XG4gIHJldHVybiBbQ29tcGF0aWJpbGl0eS5FQzIsIENvbXBhdGliaWxpdHkuRUMyX0FORF9GQVJHQVRFXS5pbmNsdWRlcyhjb21wYXRpYmlsaXR5KTtcbn1cblxuLyoqXG4gKiBSZXR1cm4gdHJ1ZSBpZiB0aGUgZ2l2ZW4gdGFzayBkZWZpbml0aW9uIGNhbiBiZSBydW4gb24gYSBGYXJnYXRlIGNsdXN0ZXJcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzRmFyZ2F0ZUNvbXBhdGlibGUoY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eSk6IGJvb2xlYW4ge1xuICByZXR1cm4gW0NvbXBhdGliaWxpdHkuRkFSR0FURSwgQ29tcGF0aWJpbGl0eS5FQzJfQU5EX0ZBUkdBVEVdLmluY2x1ZGVzKGNvbXBhdGliaWxpdHkpO1xufVxuXG4vKipcbiAqIFJldHVybiB0cnVlIGlmIHRoZSBnaXZlbiB0YXNrIGRlZmluaXRpb24gY2FuIGJlIHJ1biBvbiBhIEVDUyBBbnl3aGVyZSBjbHVzdGVyXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc0V4dGVybmFsQ29tcGF0aWJsZShjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5KTogYm9vbGVhbiB7XG4gIHJldHVybiBbQ29tcGF0aWJpbGl0eS5FWFRFUk5BTF0uaW5jbHVkZXMoY29tcGF0aWJpbGl0eSk7XG59XG4iXX0=