"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PropagatedTagSource = exports.DeploymentControllerType = exports.LaunchType = exports.BaseService = exports.ListenerConfig = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("../../../aws-applicationautoscaling");
const cloudwatch = require("../../../aws-cloudwatch");
const ec2 = require("../../../aws-ec2");
const elbv2 = require("../../../aws-elasticloadbalancingv2");
const iam = require("../../../aws-iam");
const cloudmap = require("../../../aws-servicediscovery");
const core_1 = require("../../../core");
const task_definition_1 = require("../base/task-definition");
const cluster_1 = require("../cluster");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_ApplicationListener(listener);
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddApplicationTargetsProps(props);
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_NetworkListener(listener);
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_AddNetworkTargetsProps(props);
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
_a = JSII_RTTI_SYMBOL_1;
ListenerConfig[_a] = { fqn: "monocdk.aws_ecs.ListenerConfig", version: "1.149.0" };
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        var _c;
        const props = this.props || {};
        const protocol = props.protocol;
        const port = (_c = props.port) !== null && _c !== void 0 ? _c : (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80);
        this.listener.addTargets(id, {
            ...props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        var _c, _d;
        const port = (_d = (_c = this.props) === null || _c === void 0 ? void 0 : _c.port) !== null && _d !== void 0 ? _d : 80;
        this.listener.addTargets(id, {
            ...this.props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        jsiiDeprecationWarnings.monocdk_aws_ecs_BaseServiceProps(props);
        jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinition(taskDefinition);
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        this.taskDefinition = taskDefinition;
        // launchType will set to undefined if using external DeploymentController or capacityProviderStrategies
        const launchType = ((_c = props.deploymentController) === null || _c === void 0 ? void 0 : _c.type) === DeploymentControllerType.EXTERNAL ||
            props.capacityProviderStrategies !== undefined ?
            undefined : props.launchType;
        const propagateTagsFromSource = (_e = (_d = props.propagateTaskTagsFrom) !== null && _d !== void 0 ? _d : props.propagateTags) !== null && _e !== void 0 ? _e : PropagatedTagSource.NONE;
        this.resource = new ecs_generated_1.CfnService(this, 'Service', {
            desiredCount: props.desiredCount,
            serviceName: this.physicalName,
            loadBalancers: core_1.Lazy.any({ produce: () => this.loadBalancers }, { omitEmptyArray: true }),
            deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent,
                deploymentCircuitBreaker: props.circuitBreaker ? {
                    enable: true,
                    rollback: (_f = props.circuitBreaker.rollback) !== null && _f !== void 0 ? _f : false,
                } : undefined,
            },
            propagateTags: propagateTagsFromSource === PropagatedTagSource.NONE ? undefined : props.propagateTags,
            enableEcsManagedTags: (_g = props.enableECSManagedTags) !== null && _g !== void 0 ? _g : false,
            deploymentController: props.circuitBreaker ? {
                type: DeploymentControllerType.ECS,
            } : props.deploymentController,
            launchType: launchType,
            enableExecuteCommand: props.enableExecuteCommand,
            capacityProviderStrategy: props.capacityProviderStrategies,
            healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod),
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.any({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }),
            serviceRegistries: core_1.Lazy.any({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (((_h = props.deploymentController) === null || _h === void 0 ? void 0 : _h.type) === DeploymentControllerType.EXTERNAL) {
            core_1.Annotations.of(this).addWarning('taskDefinition and launchType are blanked out when using external deployment controller.');
        }
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
        if (props.enableExecuteCommand) {
            this.enableExecuteCommand();
            const logging = (_k = (_j = this.cluster.executeCommandConfiguration) === null || _j === void 0 ? void 0 : _j.logging) !== null && _k !== void 0 ? _k : cluster_1.ExecuteCommandLogging.DEFAULT;
            if ((_l = this.cluster.executeCommandConfiguration) === null || _l === void 0 ? void 0 : _l.kmsKey) {
                this.enableExecuteCommandEncryption(logging);
            }
            if (logging !== cluster_1.ExecuteCommandLogging.NONE) {
                this.executeCommandLogConfiguration();
            }
        }
        this.node.defaultChild = this.resource;
    }
    /**
     * Import an existing ECS/Fargate Service using the service cluster format.
     * The format is the "new" format "arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name".
     * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html#ecs-resource-ids
     */
    static fromServiceArnWithCluster(scope, id, serviceArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.splitArn(serviceArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const resourceName = arn.resourceName;
        if (!resourceName) {
            throw new Error('Missing resource Name from service ARN: ${serviceArn}');
        }
        const resourceNameParts = resourceName.split('/');
        if (resourceNameParts.length !== 2) {
            throw new Error(`resource name ${resourceName} from service ARN: ${serviceArn} is not using the ARN cluster format`);
        }
        const clusterName = resourceNameParts[0];
        const serviceName = resourceNameParts[1];
        const clusterArn = core_1.Stack.of(scope).formatArn({
            partition: arn.partition,
            region: arn.region,
            account: arn.account,
            service: 'ecs',
            resource: 'cluster',
            resourceName: clusterName,
        });
        const cluster = cluster_1.Cluster.fromClusterArn(scope, `${id}Cluster`, clusterArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = serviceArn;
                this.serviceName = serviceName;
                this.cluster = cluster;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: serviceArn,
        });
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    executeCommandLogConfiguration() {
        var _c, _d;
        const logConfiguration = (_c = this.cluster.executeCommandConfiguration) === null || _c === void 0 ? void 0 : _c.logConfiguration;
        this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        const logGroupArn = (logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.cloudWatchLogGroup) ? `arn:${this.stack.partition}:logs:${this.stack.region}:${this.stack.account}:log-group:${logConfiguration.cloudWatchLogGroup.logGroupName}:*` : '*';
        this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: [logGroupArn],
        }));
        if ((_d = logConfiguration === null || logConfiguration === void 0 ? void 0 : logConfiguration.s3Bucket) === null || _d === void 0 ? void 0 : _d.bucketName) {
            this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
                actions: [
                    's3:GetBucketLocation',
                ],
                resources: ['*'],
            }));
            this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
                actions: [
                    's3:PutObject',
                ],
                resources: [`arn:${this.stack.partition}:s3:::${logConfiguration.s3Bucket.bucketName}/*`],
            }));
            if (logConfiguration.s3EncryptionEnabled) {
                this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetEncryptionConfiguration',
                    ],
                    resources: [`arn:${this.stack.partition}:s3:::${logConfiguration.s3Bucket.bucketName}`],
                }));
            }
        }
    }
    enableExecuteCommandEncryption(logging) {
        var _c, _d, _e, _f, _g, _h, _j, _k;
        this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kms:Decrypt',
                'kms:GenerateDataKey',
            ],
            resources: [`${(_d = (_c = this.cluster.executeCommandConfiguration) === null || _c === void 0 ? void 0 : _c.kmsKey) === null || _d === void 0 ? void 0 : _d.keyArn}`],
        }));
        (_f = (_e = this.cluster.executeCommandConfiguration) === null || _e === void 0 ? void 0 : _e.kmsKey) === null || _f === void 0 ? void 0 : _f.addToResourcePolicy(new iam.PolicyStatement({
            actions: [
                'kms:*',
            ],
            resources: ['*'],
            principals: [new iam.ArnPrincipal(`arn:${this.stack.partition}:iam::${this.stack.account}:root`)],
        }));
        if (logging === cluster_1.ExecuteCommandLogging.DEFAULT || ((_h = (_g = this.cluster.executeCommandConfiguration) === null || _g === void 0 ? void 0 : _g.logConfiguration) === null || _h === void 0 ? void 0 : _h.cloudWatchEncryptionEnabled)) {
            (_k = (_j = this.cluster.executeCommandConfiguration) === null || _j === void 0 ? void 0 : _j.kmsKey) === null || _k === void 0 ? void 0 : _k.addToResourcePolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                resources: ['*'],
                principals: [new iam.ServicePrincipal(`logs.${this.stack.region}.amazonaws.com`)],
                conditions: {
                    ArnLike: { 'kms:EncryptionContext:aws:logs:arn': `arn:${this.stack.partition}:logs:${this.stack.region}:${this.stack.account}:*` },
                },
            }));
        }
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_IApplicationTargetGroup(targetGroup);
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancing_LoadBalancer(loadBalancer);
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * declare const listener: elbv2.ApplicationListener;
     * declare const service: ecs.BaseService;
     * listener.addTargets('ECS', {
     *   port: 80,
     *   targets: [service.loadBalancerTarget({
     *     containerName: 'MyContainer',
     *     containerPort: 1234,
     *   })],
     * });
     */
    loadBalancerTarget(options) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_LoadBalancerTargetOptions(options);
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            },
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * declare const listener: elbv2.ApplicationListener;
     * declare const service: ecs.BaseService;
     * service.registerLoadBalancerTargets(
     *   {
     *     containerName: 'web',
     *     containerPort: 80,
     *     newTargetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_EcsTarget(targets);
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol,
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroup);
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        jsiiDeprecationWarnings.monocdk_aws_applicationautoscaling_EnableScalingProps(props);
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', {
            serviceNamespace: appscaling.ServiceNamespace.ECS,
            resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`,
            dimension: 'ecs:service:DesiredCount',
            role: this.makeAutoScalingRole(),
            ...props,
        });
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        var _c;
        jsiiDeprecationWarnings.monocdk_aws_ecs_CloudMapOptions(options);
        const sdNamespace = (_c = options.cloudMapNamespace) !== null && _c !== void 0 ? _c : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error('Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.');
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error('SRV records must be used when network mode is Bridge or Host.');
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        const { containerName, containerPort } = determineContainerNameAndPort({
            taskDefinition: this.taskDefinition,
            dnsRecordType: dnsRecordType,
            container: options.container,
            containerPort: options.containerPort,
        });
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 },
            dnsTtl: options.dnsTtl,
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort,
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * Associates this service with a CloudMap service
     */
    associateCloudMapService(options) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_AssociateCloudMapServiceOptions(options);
        const service = options.service;
        const { containerName, containerPort } = determineContainerNameAndPort({
            taskDefinition: this.taskDefinition,
            dnsRecordType: service.dnsRecordType,
            container: options.container,
            containerPort: options.containerPort,
        });
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: service.serviceArn,
            containerName,
            containerPort,
        });
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensionsMap: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName },
            ...props,
        }).attachTo(this);
    }
    /**
     * This method returns the CloudWatch metric for this service's memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this service's CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     * @deprecated use configureAwsVpcNetworkingWithSecurityGroups instead.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        jsiiDeprecationWarnings.print("monocdk.aws_ecs.BaseService#configureAwsVpcNetworking", "use configureAwsVpcNetworkingWithSecurityGroups instead.");
        jsiiDeprecationWarnings.monocdk_aws_ec2_IVpc(vpc);
        jsiiDeprecationWarnings.monocdk_aws_ec2_SubnetSelection(vpcSubnets);
        jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroup);
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.list({ produce: () => [securityGroup.securityGroupId] }),
            },
        };
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworkingWithSecurityGroups(vpc, assignPublicIp, vpcSubnets, securityGroups) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_IVpc(vpc);
        jsiiDeprecationWarnings.monocdk_aws_ec2_SubnetSelection(vpcSubnets);
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroups === undefined || securityGroups.length === 0) {
            securityGroups = [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc })];
        }
        securityGroups.forEach((sg) => { this.connections.addSecurityGroup(sg); }, this);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: securityGroups.map((sg) => sg.securityGroupId),
            },
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.');
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.');
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort,
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName,
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        if (this.serviceRegistries.length >= 1) {
            throw new Error('Cannot associate with the given service discovery registry. ECS supports at most one service registry per service.');
        }
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.any({
            produce: () => { var _c; return (_c = providedHealthCheckGracePeriod === null || providedHealthCheckGracePeriod === void 0 ? void 0 : providedHealthCheckGracePeriod.toSeconds()) !== null && _c !== void 0 ? _c : (this.loadBalancers.length > 0 ? 60 : undefined); },
        });
    }
    enableExecuteCommand() {
        this.taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'ssmmessages:CreateControlChannel',
                'ssmmessages:CreateDataChannel',
                'ssmmessages:OpenControlChannel',
                'ssmmessages:OpenDataChannel',
            ],
            resources: ['*'],
        }));
    }
}
exports.BaseService = BaseService;
_b = JSII_RTTI_SYMBOL_1;
BaseService[_b] = { fqn: "monocdk.aws_ecs.BaseService", version: "1.149.0" };
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
    /**
     * The service will be launched using the EXTERNAL launch type
     */
    LaunchType["EXTERNAL"] = "EXTERNAL";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * The deployment controller type to use for the service.
 */
var DeploymentControllerType;
(function (DeploymentControllerType) {
    /**
     * The rolling update (ECS) deployment type involves replacing the current
     * running version of the container with the latest version.
     */
    DeploymentControllerType["ECS"] = "ECS";
    /**
     * The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy
     */
    DeploymentControllerType["CODE_DEPLOY"] = "CODE_DEPLOY";
    /**
     * The external (EXTERNAL) deployment type enables you to use any third-party deployment controller
     */
    DeploymentControllerType["EXTERNAL"] = "EXTERNAL";
})(DeploymentControllerType = exports.DeploymentControllerType || (exports.DeploymentControllerType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
/**
 * Determine the name of the container and port to target for the service registry.
 */
function determineContainerNameAndPort(options) {
    var _c, _d;
    // If the record type is SRV, then provide the containerName and containerPort to target.
    // We use the name of the default container and the default port of the default container
    // unless the user specifies otherwise.
    if (options.dnsRecordType === cloudmap.DnsRecordType.SRV) {
        // Ensure the user-provided container is from the right task definition.
        if (options.container && options.container.taskDefinition != options.taskDefinition) {
            throw new Error('Cannot add discovery for a container from another task definition');
        }
        const container = (_c = options.container) !== null && _c !== void 0 ? _c : options.taskDefinition.defaultContainer;
        // Ensure that any port given by the user is mapped.
        if (options.containerPort && !container.portMappings.some(mapping => mapping.containerPort === options.containerPort)) {
            throw new Error('Cannot add discovery for a container port that has not been mapped');
        }
        return {
            containerName: container.containerName,
            containerPort: (_d = options.containerPort) !== null && _d !== void 0 ? _d : options.taskDefinition.defaultContainer.containerPort,
        };
    }
    return {};
}
//# sourceMappingURL=data:application/json;base64,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