"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsServiceBase = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_certificatemanager_1 = require("../../../aws-certificatemanager");
const aws_ecs_1 = require("../../../aws-ecs");
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2");
const aws_route53_1 = require("../../../aws-route53");
const aws_route53_targets_1 = require("../../../aws-route53-targets");
const core_1 = require("../../../core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../../core");
/**
 * The base class for ApplicationMultipleTargetGroupsEc2Service and ApplicationMultipleTargetGroupsFargateService classes.
 */
class ApplicationMultipleTargetGroupsServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        jsiiDeprecationWarnings.monocdk_aws_ecs_patterns_ApplicationMultipleTargetGroupsServiceBaseProps(props);
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        this.internalDesiredCount = props.desiredCount;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                const protocolType = new Set();
                for (const listenerProps of lbProps.listeners) {
                    const protocol = this.createListenerProtocol(listenerProps.protocol, listenerProps.certificate);
                    if (listenerProps.certificate !== undefined && protocol !== undefined && protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                        throw new Error('The HTTPS protocol must be used when a certificate is given');
                    }
                    protocolType.add(protocol);
                    const listener = this.configListener(protocol, {
                        certificate: listenerProps.certificate,
                        domainName: lbProps.domainName,
                        domainZone: lbProps.domainZone,
                        listenerName: listenerProps.name,
                        loadBalancer: lb,
                        port: listenerProps.port,
                        sslPolicy: listenerProps.sslPolicy,
                    });
                    this.listeners.push(listener);
                }
                const domainName = this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
                for (const protocol of protocolType) {
                    new core_1.CfnOutput(this, `ServiceURL${lb.node.id}${protocol.toLowerCase()}`, { value: protocol.toLowerCase() + '://' + domainName });
                }
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            const protocol = this.createListenerProtocol();
            this.listener = this.configListener(protocol, {
                listenerName: 'PublicListener',
                loadBalancer: this.loadBalancer,
            });
            const domainName = this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
            new core_1.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_IVpc(vpc);
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_BaseService(service);
        jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinition(container);
        for (const targetProps of targets) {
            const conditions = [];
            if (targetProps.hostHeader) {
                conditions.push(aws_elasticloadbalancingv2_1.ListenerCondition.hostHeaders([targetProps.hostHeader]));
            }
            if (targetProps.pathPattern) {
                conditions.push(aws_elasticloadbalancingv2_1.ListenerCondition.pathPatterns([targetProps.pathPattern]));
            }
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                        protocol: targetProps.protocol,
                    }),
                ],
                conditions,
                priority: targetProps.priority,
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinition(container);
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, target.protocol || aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                    protocol: target.protocol,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== null && enableLoggingProp !== void 0 ? enableLoggingProp : true;
        const logDriver = logDriverProp !== null && logDriverProp !== void 0 ? logDriverProp : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        return logDriver;
    }
    configListener(protocol, props) {
        const listener = this.createListener(props, protocol);
        let certificate;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            certificate = this.createListenerCertificate(props.listenerName, props.certificate, props.domainName, props.domainZone);
        }
        else {
            certificate = undefined;
        }
        if (certificate !== undefined) {
            listener.addCertificates(`Arns${props.listenerName}`, [aws_elasticloadbalancingv2_1.ListenerCertificate.fromArn(certificate.certificateArn)]);
        }
        return listener;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== null && publicLoadBalancer !== void 0 ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, name, lbProps);
    }
    createListenerProtocol(listenerProtocol, certificate) {
        return listenerProtocol !== null && listenerProtocol !== void 0 ? listenerProtocol : (certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
    }
    createListenerCertificate(listenerName, certificate, domainName, domainZone) {
        if (typeof domainName === 'undefined' || typeof domainZone === 'undefined') {
            throw new Error('A domain name and zone is required when using the HTTPS protocol');
        }
        if (certificate !== undefined) {
            return certificate;
        }
        else {
            return new aws_certificatemanager_1.Certificate(this, `Certificate${listenerName}`, {
                domainName,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(domainZone),
            });
        }
    }
    createListener({ loadBalancer, listenerName, port, sslPolicy }, protocol) {
        return loadBalancer.addListener(listenerName, {
            protocol,
            open: true,
            port,
            sslPolicy,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        return domainName;
    }
}
exports.ApplicationMultipleTargetGroupsServiceBase = ApplicationMultipleTargetGroupsServiceBase;
_a = JSII_RTTI_SYMBOL_1;
ApplicationMultipleTargetGroupsServiceBase[_a] = { fqn: "monocdk.aws_ecs_patterns.ApplicationMultipleTargetGroupsServiceBase", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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