"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = exports.NetworkMode = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const ecr = require("../../aws-ecr");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line
const assets_1 = require("../../assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core");
/**
 * networking mode on build time supported by docker
 */
class NetworkMode {
    /**
     * @param mode The networking mode to use for docker build
     */
    constructor(mode) {
        this.mode = mode;
    }
    /**
     * Reuse another container's network stack
     *
     * @param containerId The target container's id or name
     */
    static fromContainer(containerId) {
        return new NetworkMode(`container:${containerId}`);
    }
    /**
     * Used to specify a custom networking mode
     * Use this if the networking mode name is not yet supported by the CDK.
     *
     * @param mode The networking mode to use for docker build
     */
    static custom(mode) {
        return new NetworkMode(mode);
    }
}
exports.NetworkMode = NetworkMode;
_a = JSII_RTTI_SYMBOL_1;
NetworkMode[_a] = { fqn: "monocdk.aws_ecr_assets.NetworkMode", version: "1.149.0" };
/**
 * The default networking mode if omitted, create a network stack on the default Docker bridge
 */
NetworkMode.DEFAULT = new NetworkMode('default');
/**
 * Use the Docker host network stack
 */
NetworkMode.HOST = new NetworkMode('host');
/**
 * Disable the network stack, only the loopback device will be created
 */
NetworkMode.NONE = new NetworkMode('none');
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_2.Construct {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_ecr_assets_DockerImageAssetProps(props);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        this.dockerfilePath = props.file || 'Dockerfile';
        const file = path.join(dir, this.dockerfilePath);
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        const defaultIgnoreMode = core_1.FeatureFlags.of(this).isEnabled(cxapi.DOCKER_IGNORE_SUPPORT)
            ? core_1.IgnoreMode.DOCKER : core_1.IgnoreMode.GLOB;
        let ignoreMode = (_c = props.ignoreMode) !== null && _c !== void 0 ? _c : defaultIgnoreMode;
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            const dockerIgnorePatterns = fs.readFileSync(ignore).toString().split('\n').filter(e => !!e);
            exclude = [
                ...dockerIgnorePatterns,
                ...exclude,
                // Ensure .dockerignore is included no matter what.
                '!.dockerignore',
            ];
        }
        // Ensure the Dockerfile is included no matter what.
        exclude.push('!' + path.basename(file));
        // Ensure the cdk.out folder is not included to avoid infinite loops.
        const cdkout = (_e = (_d = core_1.Stage.of(this)) === null || _d === void 0 ? void 0 : _d.outdir) !== null && _e !== void 0 ? _e : 'cdk.out';
        exclude.push(cdkout);
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (((_f = props.invalidation) === null || _f === void 0 ? void 0 : _f.extraHash) !== false && props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (((_g = props.invalidation) === null || _g === void 0 ? void 0 : _g.buildArgs) !== false && props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (((_h = props.invalidation) === null || _h === void 0 ? void 0 : _h.target) !== false && props.target) {
            extraHash.target = props.target;
        }
        if (((_j = props.invalidation) === null || _j === void 0 ? void 0 : _j.file) !== false && props.file) {
            extraHash.file = props.file;
        }
        if (((_k = props.invalidation) === null || _k === void 0 ? void 0 : _k.repositoryName) !== false && props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        if (((_l = props.invalidation) === null || _l === void 0 ? void 0 : _l.networkMode) !== false && props.networkMode) {
            extraHash.networkMode = props.networkMode;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new core_1.AssetStaging(this, 'Staging', {
            ...props,
            follow: (_m = props.followSymlinks) !== null && _m !== void 0 ? _m : toSymlinkFollow(props.follow),
            exclude,
            ignoreMode,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.assetHash;
        this.assetHash = staging.assetHash;
        const stack = core_1.Stack.of(this);
        this.assetPath = staging.relativeStagedPath(stack);
        this.dockerBuildArgs = props.buildArgs;
        this.dockerBuildTarget = props.target;
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: this.assetPath,
            dockerBuildArgs: this.dockerBuildArgs,
            dockerBuildTarget: this.dockerBuildTarget,
            dockerFile: props.file,
            sourceHash: staging.assetHash,
            networkMode: (_o = props.networkMode) === null || _o === void 0 ? void 0 : _o.mode,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     */
    addResourceMetadata(resource, resourceProperty) {
        jsiiDeprecationWarnings.monocdk_CfnResource(resource);
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the resourceProperty of this resource
        // points to a local path and include the path to de dockerfile, docker build args, and target,
        // in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKERFILE_PATH_KEY] = this.dockerfilePath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKER_BUILD_ARGS_KEY] = this.dockerBuildArgs;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKER_BUILD_TARGET_KEY] = this.dockerBuildTarget;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
}
exports.DockerImageAsset = DockerImageAsset;
_b = JSII_RTTI_SYMBOL_1;
DockerImageAsset[_b] = { fqn: "monocdk.aws_ecr_assets.DockerImageAsset", version: "1.149.0" };
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
function toSymlinkFollow(follow) {
    switch (follow) {
        case undefined: return undefined;
        case assets_1.FollowMode.NEVER: return core_1.SymlinkFollowMode.NEVER;
        case assets_1.FollowMode.ALWAYS: return core_1.SymlinkFollowMode.ALWAYS;
        case assets_1.FollowMode.BLOCK_EXTERNAL: return core_1.SymlinkFollowMode.BLOCK_EXTERNAL;
        case assets_1.FollowMode.EXTERNAL: return core_1.SymlinkFollowMode.EXTERNAL;
    }
}
//# sourceMappingURL=data:application/json;base64,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